<?php

namespace Procoders\Cf7zh\Admin;

if (!defined('ABSPATH')) {
	exit('restricted access');
}

use com\zoho\api\authenticator\store\TokenStore;
use com\zoho\api\authenticator\OAuthToken;
use com\zoho\api\authenticator\Token;
use com\zoho\crm\api\exception\SDKException;
use com\zoho\crm\api\UserSignature;
use Procoders\Cf7zh\Functions;

class WpdbTokenStore implements TokenStore
{
	/**
	 * Table name for storing tokens.
	 *
	 * @var string
	 */
	private string $tableName;

	/**
	 * Constructor to initialize table name and check its existence.
	 *
	 * @param string $tableName Name of the database table.
	 */
	public function __construct(string $tableName = '')
	{
		global $wpdb;

		// Set table name
		$this->tableName = empty($tableName) ? $wpdb->prefix . 'cf7zh_oauth_tokens' : $tableName;

		// Ensure the table exists
		$this->createTableIfNotExists();
	}

	/**
	 * Generates a unique identifier.
	 *
	 * @return string A unique identifier string.
	 */
	private function generateId(): string
	{
		return uniqid();
	}

	/**
	 * Save OAuthToken into the database.
	 *
	 * @param OAuthToken $token The OAuthToken to be saved.
	 *
	 * @return void
	 * @throws SDKException
	 */
	public function saveToken(Token $token): void
	{
		global $wpdb;

		$user = new UserSignature(get_option('Cf7zh_current_user_email'));

		if ($token->getId() == null) {
			$newId = $this->generateId();
			$token->setId($newId);
		}
		try {
			$data = [
				'id' => $token->getId(),
				'user_name' => $user->getName(),
				'client_id' => $token->getClientId(),
				'client_secret' => $token->getClientSecret(),
				'refresh_token' => $token->getRefreshToken(),
				'access_token' => $token->getAccessToken(),
				'expiry_time' => $token->getExpiresIn(),
				'redirect_url' => $token->getRedirectURL(),
				'api_domain' => $token->getAPIDomain()
			];

			// Replace or update the token
			$wpdb->insert($this->tableName, $data);

		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error saving token: ' . esc_attr($e->getMessage()));
		}
	}

	/**
	 * Retrieve a stored OAuthToken from the database.
	 *
	 * @param string $client_id The user signature to identify the token.
	 * @param string $clientId Client ID to identify the token.
	 * @param string $clientSecret Client Secret to identify the token.
	 * @param string|null $grantToken The grant token used for authorization.
	 * @param string|null $refreshToken
	 *
	 * @return OAuthToken|null
	 */
	public function getToken(string $client_id): ?OAuthToken
	{
		global $wpdb;

		$result = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM %i WHERE client_id = %s ORDER BY id DESC LIMIT 1",
				$this->tableName, $client_id
			), ARRAY_A);

		if ($result === null)
			return null;

		$user = new UserSignature($result['user_name']);

		if ($result) {
			$token = new OAuthToken($result['client_id'], $result['client_secret'], $result['grant_token'], $result['refresh_token'], $result['redirect_url']);
			$token->setAccessToken($result['access_token']);
			$token->setExpiresIn($result['expiry_time']);
			$token->setAPIDomain($result['api_domain']);
			$token->setUserSignature($user);

			return $token;
		}

		return null;
	}

	/**
	 * Remove a stored OAuthToken from the database.
	 *
	 * @param string $id The user signature whose token needs to be deleted.
	 *
	 * @return void
	 * @throws SDKException
	 */
	public function deleteToken($id): void
	{
		global $wpdb;

		try {
			$wpdb->delete($this->tableName, ['id' => $id]);
		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error saving token: ' . esc_attr($e->getMessage()));
		}
	}

	/**
	 * Delete all stored OAuthTokens.
	 *
	 * @return void
	 * @throws SDKException
	 */
	public function deleteTokens(): void
	{
		global $wpdb;

		try {
			$wpdb->query(
				$wpdb->prepare(
					"TRUNCATE TABLE %i",
					$this->tableName
				));
		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error deleting all tokens: ' . esc_attr($e->getMessage()));
		}
	}

	/**
	 * Ensure the database table exists.
	 *
	 * @return void
	 */
	private function createTableIfNotExists(): void
	{
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();
		$sql = "CREATE TABLE IF NOT EXISTS $this->tableName (
            id VARCHAR(100) NOT NULL,
            user_name VARCHAR(255) NOT NULL,
            client_id VARCHAR(255),
            client_secret VARCHAR(255),
            refresh_token VARCHAR(255),
            access_token VARCHAR(255),
            expiry_time VARCHAR(20),
            redirect_url VARCHAR(255),
            api_domain VARCHAR(255),
            PRIMARY KEY (id)
        ) $charset_collate;";

		require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
		dbDelta($sql);
	}

	/**
	 * Finds a token by matching fields in the given Token instance.
	 *
	 * @param Token $token The token instance containing details to search for.
	 *
	 * @return Token|null The matching token if found, otherwise null.
	 * @throws SDKException If an error occurs during token retrieval.
	 */
	public function findToken(Token $token): ?Token
	{
		global $wpdb;

		try {

			$result = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM %i WHERE client_id = %s AND client_secret = %s AND refresh_token = %s LIMIT 1",
					$this->tableName,
					$token->getClientId(),
					$token->getClientSecret(),
					$token->getRefreshToken()
				),
				ARRAY_A);

			if ($result) {
				$foundToken = new OAuthToken(
					$result['client_id'],
					$result['client_secret'],
					$result['refresh_token'],
					$result['redirect_url']
				);

				$foundToken->setAccessToken($result['access_token']);
				$token->setExpiresIn($result['expiry_time']);
				$token->setAPIDomain($result['api_domain']);
				$foundToken->setUserSignature($result['id']);

				return $foundToken;
			}

			return null;
		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error finding token: ' . esc_attr($e->getMessage()));
		}
	}

	/**
	 * Find token by user email.
	 *
	 * @param string $email User email to search for.
	 *
	 * @return OAuthToken|null Found token or null if not found.
	 * @throws SDKException
	 */
	public function getTokenByEmail(string $email): ?OAuthToken
	{
		global $wpdb;

		try {
			$result = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM %i WHERE user_name = %s LIMIT 1",
					$this->tableName,
					$email
				),
				ARRAY_A);

			if ($result) {
				$token = new OAuthToken($result['client_id'], $result['client_secret'], $result['grant_token'], $result['refresh_token'], $result['redirect_url']);
				$token->setId($result['id']);
				$token->setAccessToken($result['access_token']);
				$token->setExpiresIn($result['expiry_time']);
				$token->setAPIDomain($result['api_domain']);

				return $token;
			}

			return null;
		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error searching token by email: ' . esc_attr($e->getMessage()));
		}
	}

	/**
	 * Finds a token by its unique identifier.
	 *
	 * @param string $id The unique identifier of the token.
	 *
	 * @return Token|null The matching token if found, otherwise null.
	 * @throws SDKException If an error occurs during token retrieval.
	 */
	public function findTokenById($id): ?Token
	{
		global $wpdb;

		try {

			$result = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT * FROM %i WHERE id = %s LIMIT 1",
					$this->tableName, $id
				),
				ARRAY_A);

			if ($result) {
				$foundToken = new OAuthToken(
					$result['client_id'],
					$result['client_secret'],
					$result['refresh_token'],
					$result['redirect_url']
				);

				$foundToken->setAccessToken($result['access_token']);
				$foundToken->setExpiresIn($result['expiry_time']);
				$foundToken->setAPIDomain($result['api_domain']);
				$foundToken->setUserSignature($result['id']);

				return $foundToken;
			}

			return null;
		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error finding token by ID: ' . esc_attr($e->getMessage()));
		}
	}

	/**
	 * Retrieves all stored tokens.
	 *
	 * @return array An array of all stored tokens.
	 * @throws SDKException If an error occurs during token retrieval.
	 */
	public function getTokens(): array
	{
		global $wpdb;

		try {
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM %i",
					$this->tableName
				),
				ARRAY_A
			);

			$tokens = [];

			foreach ($results as $result) {
				$token = new OAuthToken(
					$result['client_id'],
					$result['client_secret'],
					$result['refresh_token'],
					$result['redirect_url']
				);

				$token->setAccessToken($result['access_token']);
				$token->setExpiresIn($result['expiry_time']);
				$token->setAPIDomain($result['api_domain']);
				$token->setUserSignature($result['id']);

				$tokens[] = $token;
			}

			return $tokens;
		} catch (\Exception $e) {
			throw new SDKException(esc_attr($e->getCode()), 'Error retrieving all tokens: ' . esc_attr($e->getMessage()));
		}
	}
}