<?php
/**
 * Initialize and display admin panel output.
 *
 * @package Cf7zh
 */

namespace Procoders\Cf7zh\Admin;

if ( ! defined( 'ABSPATH' ) ) {
	exit( 'restricted access' );
}

use Procoders\Cf7zh\Loader as Loader;

/**
 * Class Settings
 */
class Settings {

	/**
	 * Handles and updates settings submitted from the admin panel.
	 * Renders the settings template with updated values.
	 */
	public function settings_callback(): void {
		// Check if user has the capability to manage options
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'connect-cf7-to-zoho' ) );
		}

		$template = new Loader();
		$settings = [];

		$notification_subject_default = esc_html__( 'API Error Notification', 'connect-cf7-to-zoho' );

		// Check for 'submit' submission.
		if ( isset( $_POST['submit'] ) ) {
			// Nonce verification is handled in updateOptionFields()
			$this->updateOptionFields();
		}

		// Get saved options
		$settings['client_id']          = get_option( 'Cf7zh_client_id' );
		$settings['client_secret']      = get_option( 'Cf7zh_client_secret' );
		$settings['current_user_email'] = get_option( 'Cf7zh_current_user_email' );
		$settings['zoho_auth_url']      = $this->get_zoho_auth_url();

		// Get notification_subject, set default if not exists.
		$settings['notification_subject'] = get_option( 'Cf7zh_notification_subject', $notification_subject_default );
		$settings['notification_send_to'] = get_option( 'Cf7zh_notification_send_to' );
		$settings['uninstall']            = get_option( 'Cf7zh_uninstall' );

		$template->set_template_data(
			[
				'template' => $template,
				'settings' => $settings,
			]
		)->get_template_part( 'admin/settings' );
	}

	public function get_zoho_auth_url(): string {
		$clientId        = get_option( 'Cf7zh_client_id' );
		$accounts_server = get_option( 'Cf7zh_accounts_server' );
		$accounts_server = $accounts_server ?: 'https://accounts.zoho.com';

		if ( ! $clientId ) {
			return '#';
		}

		$baseUrl = $accounts_server . '/oauth/v2/auth';
		$params  = [
			'scope'         => 'ZohoCRM.modules.ALL,ZohoCRM.settings.ALL,ZohoCRM.users.ALL,ZohoCRM.org.ALL',
			'client_id'     => $clientId,
			'response_type' => 'code',
			'redirect_uri'  => get_site_url() . Cf7zh_REDIRECT_URL,
			'access_type'   => 'offline',
		];

		return $baseUrl . '?' . http_build_query( $params );
	}

	/**
	 * Iterates over defined option fields and updates each with the submitted value
	 * Casts to int if the option is 'Cf7zh_uninstall'
	 */
	private function updateOptionFields(): void {
		// Verify nonce regardless of $_POST being empty or not
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['_wpnonce'] ), 'Cf7zh_submit_form' ) ) {
			wp_die( esc_html__( 'Security check failed. Please try again.', 'connect-cf7-to-zoho' ) );
		}

		// Check if user has the capability to manage options
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'connect-cf7-to-zoho' ) );
		}

		$option_fields = [
			'Cf7zh_client_id',
			'Cf7zh_client_secret',
			'Cf7zh_current_user_email',
			'Cf7zh_notification_subject',
			'Cf7zh_notification_send_to',
			'Cf7zh_uninstall',
		];

		// perform update_option for each option field.
		foreach ( $option_fields as $option_field ) {
			// Only update if the field is set in $_POST
			if ( isset( $_POST[ $option_field ] ) ) {
				$field_value = sanitize_text_field( wp_unslash( $_POST[ $option_field ] ) );

				// Casting to int if it's 'Cf7zh_uninstall'
				if ( 'Cf7zh_uninstall' === $option_field ) {
					$field_value = (int) $field_value;
				}

				update_option( $option_field, $field_value );
			}
		}
	}

	private function reset_forms(): void {
		$all_forms_ids = get_posts( [
			'fields'         => 'ids',
			'posts_per_page' => - 1,
			'post_type'      => 'Cf7zh_contact_form'
		] );
		foreach ( $all_forms_ids as $id ) {
			delete_post_meta( $id, 'Cf7zh_active' );
		}
	}


}
