<?php
/**
 * Initialize and display main admin panel output.
 *
 * @package Cf7zh
 */

namespace Procoders\Cf7zh\Admin;

if (!defined('ABSPATH')) {
	exit('restricted access');
}

use com\zoho\api\authenticator\OAuthBuilder;
use com\zoho\api\logger\Levels;
use com\zoho\api\logger\LogBuilder;

use com\zoho\crm\api\dc\EUDataCenter;
use com\zoho\crm\api\dc\INDataCenter;
use com\zoho\crm\api\dc\USDataCenter;
use com\zoho\crm\api\dc\CNDataCenter;
use com\zoho\crm\api\dc\AUDataCenter;
use com\zoho\crm\api\dc\CADataCenter;
use com\zoho\crm\api\dc\JPDataCenter;

use com\zoho\crm\api\InitializeBuilder;
use com\zoho\crm\api\ParameterMap;
use com\zoho\crm\api\SDKConfigBuilder;
use com\zoho\crm\api\exception\SDKException;

use com\zoho\crm\api\fields\FieldsOperations;
use com\zoho\crm\api\fields\GetFieldsParam;

use Procoders\Cf7zh\Functions as Functions;
use Procoders\Cf7zh\Includes\ZohoFields as ZohoFields;
use Procoders\Cf7zh\Loader as Loader;
use Procoders\Cf7zh\Admin\WpdbTokenStore as WpdbTokenStore;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Class Functions
 */
class Init
{

	/**
	 * Initializes the callback function for the given request
	 *
	 * @return void
	 */
	public function init_callback(): void
	{
		$template = new Loader();

		// Check if user has the capability to access this page
		if (!current_user_can('manage_options')) {
			wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'connect-cf7-to-zoho'));
		}

		// If we're processing a form submission, verify the nonce
		if (isset($_POST['submit'])) {
			if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_key($_POST['_wpnonce']), 'Cf7zh_submit_form')) {
				wp_die(esc_html__('Security check failed. Please try again.', 'connect-cf7-to-zoho'));
			}
		}

		if (!isset($_REQUEST['id'])) {
			// Lets Get all forms.
			$this->getFormList();
			return;
		}

		// Validate and sanitize the ID parameter
		$id_param = isset($_REQUEST['id']) ? sanitize_text_field(wp_unslash($_REQUEST['id'])) : '';

		// Ensure ID is a valid digit
		$id = ctype_digit($id_param ?? '') ? (int)$id_param : 1;

		// Verify the user can edit this specific form
		if (!current_user_can('edit_post', $id)) {
			wp_die(esc_html__('You do not have sufficient permissions to edit this form.', 'connect-cf7-to-zoho'));
		}

		// Process form submission if present
		if (isset($_POST['submit'])) {
			$this->updateMetaFields($id);
			$message = $this->getSubmitStatusMessage();
		}

		// Get and display form data
		$form = $this->getFormData($id);
		$form['message'] = $message ?? false;
		$template->set_template_data(
			[
				'template' => $template,
				'form' => $form,
			]
		)->get_template_part('admin/form');
	}

	public function set_route(): void
	{
		register_rest_route('wp/v2/cf7zh', 'auth', [
			'methods' => 'GET',
			'callback' => [$this, 'handle_zoho_auth_callback'],
			'permission_callback' => '__return_true',
		]);
	}

	/**
	 * Handler for the /wp-json/wp/v2/cf7zh/auth endpoint.
	 * Stores Zoho tokens in the database after successful OAuth callback.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response
	 */
	public function handle_zoho_auth_callback(WP_REST_Request $request): void
	{
		// Check for required parameters in the request
		$auth_code = $request->get_param('code');
		$accounts_server = $request->get_param('accounts-server');

		if (empty($auth_code) || empty($accounts_server)) {
			// Log error and redirect
			Functions::return_error('CF7 to Zoho: OAuth callback received without required parameters.');
			wp_safe_redirect(get_site_url() . '/wp-admin/admin.php?page=cfzh_settings&cf7zh_error=missing_params');
			exit;
		}

		// Sanitize received data
		$auth_code = sanitize_text_field($auth_code);
		$accounts_server = esc_url_raw($accounts_server); // Stricter sanitization for URL

		// Optional: Additional validation for accounts_server URL format
		if (!filter_var($accounts_server, FILTER_VALIDATE_URL)) {
			Functions::return_error('CF7 to Zoho: OAuth callback received with invalid accounts-server URL.');
			wp_safe_redirect(get_site_url() . '/wp-admin/admin.php?page=cfzh_settings&cf7zh_error=invalid_url');
			exit;
		}

		update_option('Cf7zh_grand_token', $auth_code);
		update_option('Cf7zh_accounts_server', $accounts_server);

		$clientId = get_option('Cf7zh_client_id');
		$clientSecret = get_option('Cf7zh_client_secret');

		if (!$clientId || !$clientSecret) {
			Functions::return_error('CF7 to Zoho: OAuth callback received, but Client ID or Client Secret are not set.');
			wp_safe_redirect(get_site_url() . '/wp-admin/admin.php?page=cfzh_settings&cf7zh_error=no_settings');
			exit;
		}

		$this->initializeZohoSDK($clientId, $clientSecret, $auth_code, $accounts_server);

	}

	/**
	 * Initializes the Zoho SDK.
	 *
	 * @param string $clientId Zoho Client ID.
	 * @param string $clientSecret Zoho Client Secret.
	 * @param string $authCode Zoho Auth Code.
	 * @param string $accountsServer Zoho Accounts Server URL.
	 * @throws \Exception If Zoho SDK initialization fails.
	 */
	private function initializeZohoSDK(string $clientId, string $clientSecret, string $authCode, string $accountsServer): void
	{
		global $wpdb;

		try {
			$tokenStore = new WpdbTokenStore($wpdb->prefix . Cf7zh_TOKEN_TABLE);

			$logger = (new LogBuilder())
				->level(Levels::INFO)
				->filePath(Functions::get_logs_path() . 'zoho.log')
				->build();

			// Determine Zoho environment based on the received accounts_server
			$domainEnding = $this->getDomainEndingFromURL($accountsServer);
			switch ($domainEnding) {
				case 'US':
					$environment = USDataCenter::PRODUCTION();
					break;
				case 'EU':
					$environment = EUDataCenter::PRODUCTION();
					break;
				case 'IN':
					$environment = INDataCenter::PRODUCTION();
					break;
				case 'CN':
					$environment = CNDataCenter::PRODUCTION();
					break;
				case 'AU':
					$environment = AUDataCenter::PRODUCTION();
					break;
				case 'CA':
					$environment = CADataCenter::PRODUCTION();
					break;
				case 'JP':
					$environment = JPDataCenter::PRODUCTION();
					break;
				default:
					// Handle unknown domain
					Functions::return_error('CF7 to Zoho: Unknown Zoho accounts server domain ending: ' . $accountsServer);
					wp_safe_redirect(get_site_url() . '/wp-admin/admin.php?page=cfzh_settings&cf7zh_error=inv_data_center');
					exit;
			}

			$sdkConfig = (new SDKConfigBuilder())
				->autoRefreshFields(false)
				->pickListValidation(false)
				->sslVerification(true)
				->build();

			// Build OAuth token for initialization
			$token = (new OAuthBuilder())
				->clientId($clientId)
				->clientSecret($clientSecret)
				->grantToken($authCode)
				->redirectURL(get_site_url() . Cf7zh_REDIRECT_URL)
				->build();

			// Initialize Zoho SDK
			(new InitializeBuilder())
				->environment($environment)
				->token($token)
				->store($tokenStore)
				->logger($logger)
				->SDKConfig($sdkConfig)
				->resourcePath(get_temp_dir())
				->initialize();

			// success req
			update_option('Cf7zh_access_token_stored_flag', true);
			wp_safe_redirect(get_site_url() . '/wp-admin/admin.php?page=cfzh_settings&cf7zh_success=zoho_auth');
			exit;
		} catch (\Exception $e) {
			update_option('Cf7zh_access_token_stored_flag', false);
			wp_safe_redirect(get_site_url() . '/wp-admin/admin.php?page=cfzh_settings&cf7zh_error=zoho_auth_error');
			exit;
		}
	}

	public function zohoInit(): void
	{
		global $wpdb;

		$clientId = get_option('Cf7zh_client_id');
		$clientServer = get_option('Cf7zh_accounts_server');

		switch ($this->getDomainEndingFromURL($clientServer)) {
			case
			'US':
				$environment = USDataCenter::PRODUCTION();
				break;
			case 'EU':
				$environment = EUDataCenter::PRODUCTION();
				break;
			case 'IN':
				$environment = INDataCenter::PRODUCTION();
				break;
			case 'CN':
				$environment = CNDataCenter::PRODUCTION();
				break;
			case 'AU':
				$environment = AUDataCenter::PRODUCTION();
				break;
			case 'CA':
				$environment = CADataCenter::PRODUCTION();
				break;
			case 'JP':
				$environment = JPDataCenter::PRODUCTION();
				break;
			default:
				$environment = EUDataCenter::PRODUCTION();
		}

		if ($clientServer && $clientId) {
			$tokenStore = new WpdbTokenStore($wpdb->prefix . Cf7zh_TOKEN_TABLE);
			$user = $tokenStore->getToken($clientId);
			if ($user !== null) {
				$token = (new OAuthBuilder())
					->clientId($user->getClientId())
					->clientSecret($user->getClientSecret())
					->refreshToken($user->getRefreshToken())
					->build();
				(new InitializeBuilder())
					->environment($environment)
					->token($token)
					->initialize();
			}
		}
	}

	/**
	 * Returns an array containing form data for the given ID
	 *
	 * @param int $id The ID of the form.
	 *
	 * @return array The form data including various fields and metadata
	 */
	private function getFormData(int $id): array
	{
		$_form = get_post_meta($id, '_form', true);
		$pd_fields = $this->getFields(array_keys(ZohoFields::$breakFields));

		return [
			'Cf7zh_active' => get_post_meta($id, 'Cf7zh_active', true),
			'Cf7zh_update_person' => get_post_meta($id, 'Cf7zh_update_person', true),
			'Cf7zh_update_org' => get_post_meta($id, 'Cf7zh_update_org', true),
			'Cf7zh_fields' => get_post_meta($id, 'Cf7zh_fields', true),
			'title' => get_the_title($id),
			'_form' => $_form,
			'pd_fields' => $pd_fields,
			'cf7_fields' => $this->get_cf7_fields($_form),
		];
	}

	/**
	 * Returns an array containing the fields information for each group
	 *
	 * @param array $groups The array of groups
	 *
	 * @return array|null The array containing the fields information or null if no fields found
	 */
	private function getFields(array $groups): ?array
	{
		$fields = [];
		foreach ($groups as $group) {
			$fields_ = get_option('Cf7zh_' . $group);

			if ($fields_) {
				foreach ($fields_ as $key => $field_) {
					if (is_object($field_)) {
						if (in_array($field_->apiName, ZohoFields::$breakFields[$group])) {
							continue;
						}
						$fields[$group][$key]['id'] = $field_->id;
						$fields[$group][$key]['name'] = $field_->apiName;
						$fields[$group][$key]['label'] = $field_->displayLabel;
						$fields[$group][$key]['type'] = $field_->dataType;
						$fields[$group][$key]['required'] = $field_->required;
						$fields[$group][$key]['customField'] = $field_->customField;
						$fields[$group][$key]['fieldLabel'] = $field_->fieldLabel;
					} else {
						if (in_array($field_['apiName'], ZohoFields::$breakFields[$group])) {
							continue;
						}
						$fields[$group][$key]['id'] = $field_['id'];
						$fields[$group][$key]['name'] = $field_['apiName'];
						$fields[$group][$key]['label'] = $field_['displayLabel'];
						$fields[$group][$key]['type'] = $field_['dataType'];
						$fields[$group][$key]['required'] = $field_['required'];
						$fields[$group][$key]['customField'] = $field_['customField'];
						$fields[$group][$key]['fieldLabel'] = $field_['fieldLabel'];
					}
				}
			}
		}

		return $fields;
	}

	/**
	 * Retrieves a list of contact forms and their details
	 *
	 * @return void
	 * @throws SDKException
	 */
	public function getFormList(): void
	{
		$template = new loader();

		if ($this->syncCF7withZoho()) {
			$forms = new \WP_Query(
				[
					'post_type' => 'wpcf7_contact_form',
					'order' => 'ASC',
					'posts_per_page' => -1,
				]
			);

			$forms_array = [];
			while ($forms->have_posts()) {
				$forms->the_post();
				$id = get_the_ID();
				$forms_array[$id]['title'] = get_the_title();
				$forms_array[$id]['status'] = get_post_meta(get_the_ID(), 'Cf7zh_active', true);
				$forms_array[$id]['link'] = menu_page_url(functions::get_plugin_slug(), 0) . '&id=' . $id;
			}
			wp_reset_postdata();

			$template->set_template_data(
				[
					'template' => $template,
					'forms' => $forms_array,
				]
			)->get_template_part('admin/formList');
		} else {
			// TODO: Lets make the error template.
			$template->set_template_data(
				[
					'template' => $template,
					'forms' => false,
				]
			)->get_template_part('admin/formList');
		}
	}

	/**
	 * Отримання всіх доступних полів для об'єкта Lead у Zoho CRM
	 *
	 * @return array Масив полів Lead
	 * @throws SDKException
	 */
	public function getZohoLeadFields()
	{
		try {
			$this->zohoInit();
			$fieldsOperations = new FieldsOperations();
			$paramInstance = new ParameterMap();

			$paramInstance->add(GetFieldsParam::module(), "Leads");

			$response = $fieldsOperations->getFields($paramInstance);

			if ($response != null) {
				$statusCode = $response->getStatusCode();

				if ($statusCode == 200) {
					$responseHandler = $response->getObject();

					if ($responseHandler instanceof \com\zoho\crm\api\fields\ResponseWrapper) {
						$fields = $responseHandler->getFields();

						$leadFields = [];

						foreach ($fields as $field) {
							$fieldInfo = [
								'id' => $field->getId(),
								'apiName' => $field->getAPIName(),
								'displayLabel' => $field->getDisplayLabel(),
								'dataType' => $field->getDataType(),
								'required' => $field->getSystemMandatory(),
								'customField' => $field->getCustomField(),
								'fieldLabel' => $field->getFieldLabel()
							];

							// Add information about list values ​​if it is a list/picklist type field
							if ($field->getDataType() == 'picklist' && method_exists($field, 'getPickListValues')) {
								$pickListValues = $field->getPickListValues();
								$values = [];

								if ($pickListValues != null) {
									foreach ($pickListValues as $pickListValue) {
										$values[] = [
											'displayValue' => $pickListValue->getDisplayValue(),
											'actualValue' => $pickListValue->getActualValue(),
										];
									}
								}

								$fieldInfo['pickListValues'] = $values;
							}

							$leadFields[] = $fieldInfo;
						}

						return [
							'success' => true,
							'fields' => $leadFields
						];
					} else if ($responseHandler instanceof \com\zoho\crm\api\fields\APIException) {
						$exception = $responseHandler;

						return [
							'success' => false,
							'code' => $exception->getCode()->getValue(),
							'message' => $exception->getMessage()->getValue()
						];
					}
				} else {
					return [
						'success' => false,
						'message' => "Error retrieving fields. Status code: " . $statusCode
					];
				}
			}
		} catch (Exception $e) {
			return [
				'success' => false,
				'message' => 'Error: ' . $e->getMessage()
			];
		}

		return [
			'success' => false,
			'message' => 'Unknown error while retrieving Lead fields'
		];
	}


	/**
	 * Syncs Contact Form 7 with Zoho by updating the options with required data from Zoho
	 *
	 * @return bool True if the sync is successful, otherwise False
	 * @throws SDKException
	 */
	private function syncCF7withZoho(): bool
	{

		try {
			$res = $this->getZohoLeadFields();
			if ($res['success'] == true) {
				update_option('Cf7zh_leads', $res['fields']);
			}
		} catch (SDKException $e) {
			Functions::return_error($e->getMessage());
			throw new SDKException(esc_attr($e->getCode()), 'Error cf7 sync: ' . esc_attr($e->getMessage()));
		}

		return true;
	}

	/**
	 * Returns an array containing a status message and a success flag for the submission status
	 *
	 * @return array The status message and a success flag
	 */
	private function getSubmitStatusMessage(): array
	{
		return [
			'text' => esc_html__('Integration settings saved.', 'connect-cf7-to-zoho'),
			'success' => true,
		];
	}

	/**
	 * Updates meta fields for a specified ID
	 *
	 * @param int $id The ID of the post to update meta fields for.
	 *
	 * @return void
	 */
	private function updateMetaFields(int $id): void
	{
		// Verify nonce regardless of $_POST being empty or not
		if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_key($_POST['_wpnonce']), 'Cf7zh_submit_form')) {
			wp_die(esc_html__('Security check failed. Please try again.', 'connect-cf7-to-zoho'));
		}

		// Check if user has the capability to edit posts
		if (!current_user_can('edit_post', $id)) {
			wp_die(esc_html__('You do not have sufficient permissions to edit this form.', 'connect-cf7-to-zoho'));
		}

		$meta_fields = [
			'Cf7zh_fields',
			'Cf7zh_active',
			'Cf7zh_update_person',
			'Cf7zh_update_org',
		]; // define the meta fields.

		// Default values for checkbox fields
		$default_values = [
			'Cf7zh_active' => '0',
			'Cf7zh_update_person' => '0',
			'Cf7zh_update_org' => '0',
		];

		// perform update_post_meta for each option field.
		foreach ($meta_fields as $meta_field) {
			// For checkbox fields, use default value if not set
			if (isset($default_values[$meta_field]) && !isset($_POST[$meta_field])) {
				$field_value = $default_values[$meta_field];
			} // For other fields, only update if they are set
			elseif (isset($_POST[$meta_field])) {
				$field_value = sanitize_post(wp_unslash($_POST[$meta_field]));
			} // Skip fields that are not set and don't have defaults
			else {
				continue;
			}

			update_post_meta(
				$id,
				$meta_field,
				$field_value
			);
		}
	}

	/**
	 * Returns an array containing CF7 fields extracted from the given form
	 *
	 * @param string $_form The form content from which to extract CF7 fields.
	 *
	 * @return array|null The CF7 fields extracted from the form content, or null if no fields found.
	 */
	private function get_cf7_fields(string $_form): bool|array
	{
		preg_match_all('#\[([^\]]*)\]#', $_form, $matches);
		if (null === $matches) {
			return false;
		}

		$cf7_fields = [];
		foreach ($matches[1] as $match) {
			$match_explode = explode(' ', $match);
			$field_type = str_replace('*', '', $match_explode[0]);
			// Continue in iteration if the field type is 'submit'.
			if ('submit' === $field_type) {
				continue;
			}
			if (isset($match_explode[1])) {
				$cf7_fields[$match_explode[1]] = [
					'key' => $match_explode[1],
					'type' => $field_type,
				];
			}
		}

		return $cf7_fields;
	}

	private function getDomainEndingFromURL($accounts_server)
	{
		if (!filter_var($accounts_server, FILTER_VALIDATE_URL)) {
			Functions::return_error(esc_attr($accounts_server));
			throw new Exception("Not valid URL: " . esc_attr($accounts_server));
		}
		$host = wp_parse_url($accounts_server, PHP_URL_HOST);
		$parts = explode(".", $host);
		$ending = end($parts);
		return strtoupper($ending);
	}

}
