<?php
/*
 * Plugin Name: ConfigiView – Simple Product & Material Configurator
 * Plugin URI: http://configiview.eranova.hr/
 * Description: Simple visual configurator for construction materials and custom products. Let users select colors, finishes, patterns, and predefined options with visual examples.
 * Version: 1.0.0
 * Requires at least: 6.2
 * Requires PHP: 7.4
 * Author: Eranova
 * Author URI: https://eranova.hr/
 * License:           GPL v2 or later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: configiview
 * Domain Path: /languages
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CONFIGIVIEW_VERSION', '1.0.0');
define('CONFIGIVIEW_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CONFIGIVIEW_PLUGIN_URL', plugin_dir_url(__FILE__));

class ConfigiView {
    public const OPTION_COLORS = 'configiview_colors';
    public const OPTION_PATTERNS = 'configiview_patterns';
    public const OPTION_BASE_GROUP_VISIBILITY = 'configiview_base_group_visibility';
    public const OPTION_EXTRA_GROUPS = 'configiview_extra_groups';
    public const OPTION_COMBO_PREVIEWS = 'configiview_combo_previews';
    public const OPTION_COMBO_VISIBILITY = 'configiview_combo_visibility';
    public const OPTION_LABELS = 'configiview_labels';
    public const OPTION_EMAIL = 'configiview_email';
    public const OPTION_CTA_SETTINGS = 'configiview_cta_settings';
    public const OPTION_FRONTEND_SETTINGS = 'configiview_frontend_settings';
    public const OPTION_DEFAULTS_INITIALIZED = 'configiview_defaults_initialized';

    public static function is_pro_active() {
        return true;
    }

    public static function is_pro() {
        return true;
    }

    public static function get_cached_license_status() {
        return true;
    }

    public static function validate_license($force = false) {
        return true;
    }

    private static function get_admin_menu_icon() {
        static $icon = null;
        if (null !== $icon) {
            return $icon;
        }

        $icon = 'dashicons-screenoptions';
        $svg_path = trailingslashit(CONFIGIVIEW_PLUGIN_DIR) . 'assets/images/configiview.svg';
        if (!is_readable($svg_path)) {
            return $icon;
        }

        $svg = file_get_contents($svg_path);
        if (!is_string($svg) || '' === $svg) {
            return $icon;
        }

        // WP admin menu icons work best as black-filled SVG data URIs.
        $svg = str_replace('currentColor', '#000000', $svg);
        $icon = 'data:image/svg+xml;base64,' . base64_encode($svg);

        return $icon;
    }
    
    public function __construct() {
        add_action('plugins_loaded', array($this, 'load_textdomain'));

        // Register shortcode
        add_shortcode('configiview', array($this, 'render_visualizer'));
        
        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));
        add_action('init', array($this, 'maybe_set_defaults'));
        
        // Admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_post_configiview_reset_defaults', array($this, 'handle_reset_defaults'));
        
        // Add media upload support
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }

    public static function activate() {
        if (false === get_option(self::OPTION_COLORS, false)) {
            update_option(self::OPTION_COLORS, self::get_default_colors());
        }

        if (false === get_option(self::OPTION_PATTERNS, false)) {
            update_option(self::OPTION_PATTERNS, self::get_default_patterns());
        }
        if (false === get_option(self::OPTION_BASE_GROUP_VISIBILITY, false)) {
            update_option(self::OPTION_BASE_GROUP_VISIBILITY, self::get_default_base_group_visibility());
        }
        if (false === get_option(self::OPTION_EXTRA_GROUPS, false)) {
            update_option(self::OPTION_EXTRA_GROUPS, array());
        }

        if (false === get_option(self::OPTION_COMBO_PREVIEWS, false)) {
            update_option(self::OPTION_COMBO_PREVIEWS, self::get_default_combo_previews());
        }
        if (false === get_option(self::OPTION_COMBO_VISIBILITY, false)) {
            update_option(self::OPTION_COMBO_VISIBILITY, array());
        }

        if (false === get_option(self::OPTION_LABELS, false)) {
            update_option(self::OPTION_LABELS, self::get_default_labels());
        }

        if (false === get_option(self::OPTION_EMAIL, false)) {
            update_option(self::OPTION_EMAIL, get_option('admin_email'));
        }

        if (false === get_option(self::OPTION_CTA_SETTINGS, false)) {
            update_option(self::OPTION_CTA_SETTINGS, self::get_default_cta_settings());
        }
        if (false === get_option(self::OPTION_FRONTEND_SETTINGS, false)) {
            update_option(self::OPTION_FRONTEND_SETTINGS, self::get_default_frontend_settings());
        }
        update_option(self::OPTION_DEFAULTS_INITIALIZED, 1);
    }

    public function maybe_set_defaults() {
        if ('1' === (string) get_option(self::OPTION_DEFAULTS_INITIALIZED, '0')) {
            return;
        }

        self::activate();
    }

    public function load_textdomain() {
        $locale = function_exists('determine_locale') ? determine_locale() : get_locale();
        $locales = array($locale);
        if (strpos($locale, '_') !== false) {
            $locales[] = strtolower((string) strtok($locale, '_'));
        }

        foreach (array_unique($locales) as $candidate_locale) {
            $mofile = CONFIGIVIEW_PLUGIN_DIR . 'languages/configiview-' . $candidate_locale . '.mo';
            if (is_readable($mofile)) {
                load_textdomain('configiview', $mofile);
                break;
            }
        }
    }

    public static function get_default_image() {
        return '';
    }

    public static function get_default_colors() {
        return array();
    }

    public static function get_default_patterns() {
        return array();
    }

    public static function get_default_combo_previews() {
        return array();
    }

    public static function get_default_cta_button_label() {
        return __('Send Inquiry', 'configiview');
    }

    public static function get_default_cta_subject() {
        return __('Inquiry about {group1} + {group2}', 'configiview');
    }

    public static function get_default_cta_body() {
        return __('Hello, I am interested in this configuration: {group1} + {group2}. Please send me an offer and more details.', 'configiview');
    }

    public static function get_default_cta_settings() {
        return array(
            'mode' => 'mailto',
            'button_label' => self::get_default_cta_button_label(),
            'subject_template' => self::get_default_cta_subject(),
            'body_template' => self::get_default_cta_body(),
            'shortcode' => '',
        );
    }

    public static function get_cta_settings() {
        $defaults = self::get_default_cta_settings();
        $saved = get_option(self::OPTION_CTA_SETTINGS, array());
        if (!is_array($saved)) {
            return $defaults;
        }

        $mode_raw = isset($saved['mode']) ? sanitize_key((string) $saved['mode']) : $defaults['mode'];
        $mode = in_array($mode_raw, array('mailto', 'shortcode'), true) ? $mode_raw : $defaults['mode'];

        return array(
            'mode' => $mode,
            'button_label' => isset($saved['button_label']) && '' !== trim((string) $saved['button_label'])
                ? sanitize_text_field((string) $saved['button_label'])
                : $defaults['button_label'],
            'subject_template' => isset($saved['subject_template']) && '' !== trim((string) $saved['subject_template'])
                ? sanitize_text_field((string) $saved['subject_template'])
                : $defaults['subject_template'],
            'body_template' => isset($saved['body_template']) && '' !== trim((string) $saved['body_template'])
                ? sanitize_textarea_field((string) $saved['body_template'])
                : $defaults['body_template'],
            'shortcode' => isset($saved['shortcode']) ? sanitize_text_field((string) $saved['shortcode']) : '',
        );
    }

    public static function get_default_frontend_settings() {
        return array(
            'accordion_enabled' => 1,
        );
    }

    public static function get_frontend_settings() {
        $defaults = self::get_default_frontend_settings();
        $saved = get_option(self::OPTION_FRONTEND_SETTINGS, array());
        if (!is_array($saved)) {
            return $defaults;
        }

        return array(
            'accordion_enabled' => isset($saved['accordion_enabled']) && (int) $saved['accordion_enabled'] === 1 ? 1 : 0,
        );
    }

    public static function get_default_labels() {
        return array(
            'group_one_singular' => __('Color', 'configiview'),
            'group_one_plural' => __('Colors', 'configiview'),
            'group_two_singular' => __('Pattern', 'configiview'),
            'group_two_plural' => __('Patterns', 'configiview'),
        );
    }

    public static function get_default_base_group_visibility() {
        return array(
            'group_1' => 1,
            'group_2' => 1,
        );
    }

    public static function get_base_group_visibility() {
        $defaults = self::get_default_base_group_visibility();
        if (!self::is_pro_active()) {
            return $defaults;
        }

        $saved = get_option(self::OPTION_BASE_GROUP_VISIBILITY, array());
        if (!is_array($saved)) {
            return $defaults;
        }

        return array(
            'group_1' => isset($saved['group_1']) && (int) $saved['group_1'] === 0 ? 0 : 1,
            'group_2' => isset($saved['group_2']) && (int) $saved['group_2'] === 0 ? 0 : 1,
        );
    }

    public static function normalize_choice_key($name) {
        $key = sanitize_title((string) $name);
        return strtolower(trim((string) $key));
    }

    public function enqueue_assets() {
        wp_enqueue_style(
            'configiview-style',
            CONFIGIVIEW_PLUGIN_URL . 'assets/css/style.css',
            array(),
            CONFIGIVIEW_VERSION
        );
        
        wp_enqueue_script(
            'configiview-script',
            CONFIGIVIEW_PLUGIN_URL . 'assets/js/visualizer.js',
            array('jquery'),
            CONFIGIVIEW_VERSION,
            true
        );
        
        $cta_settings = self::get_cta_settings();
        $cta_mode = ('shortcode' === $cta_settings['mode'] && self::is_pro_active()) ? 'shortcode' : 'mailto';
        $cta_button_label = $cta_settings['button_label'];
        $cta_subject = $cta_settings['subject_template'];
        $cta_body = $cta_settings['body_template'];
        $cta_email = sanitize_email((string) get_option(self::OPTION_EMAIL, ''));
        if ('' === $cta_email) {
            $cta_email = sanitize_email((string) get_option('admin_email'));
        }

        // Pass frontend settings to JavaScript
        wp_localize_script('configiview-script', 'configiViewAjax', array(
            'i18n' => array(
                'notSelected' => __('Not selected', 'configiview'),
                'ctaButtonLabel' => $cta_button_label,
                'ctaMode' => $cta_mode,
                'ctaEmail' => $cta_email,
                'ctaSubjectTemplate' => sanitize_text_field($cta_subject),
                'ctaBodyTemplate' => sanitize_textarea_field($cta_body),
                'gmailFallbackPrompt' => __('If your email app did not open, do you want to continue in Gmail?', 'configiview'),
                'noComboPreview' => __('No preview for this combination yet.', 'configiview'),
            ),
        ));
    }
    
    public function enqueue_admin_assets($hook) {
        if (!in_array($hook, array('toplevel_page_configiview-settings', 'settings_page_configiview-settings'), true)) {
            return;
        }
        $max_items = self::is_pro_active() ? 9999 : 5;
        $max_items_message = self::is_pro_active()
            ? __('No limit.', 'configiview')
            : __('Maximum 5 items allowed.', 'configiview');
        wp_enqueue_media();
        wp_enqueue_style(
            'configiview-admin-style',
            CONFIGIVIEW_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            CONFIGIVIEW_VERSION
        );
        wp_enqueue_script(
            'configiview-admin',
            CONFIGIVIEW_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            CONFIGIVIEW_VERSION,
            true
        );
        wp_localize_script('configiview-admin', 'configiViewAdmin', array(
            'mediaTitle' => __('Select Image', 'configiview'),
            'mediaButton' => __('Use this image', 'configiview'),
            'placeholderColorName' => __('Color name', 'configiview'),
            'placeholderPatternName' => __('Pattern name', 'configiview'),
            'uploadImage' => __('Upload Image', 'configiview'),
            'optionImageLabel' => __('Option Image', 'configiview'),
            'remove' => __('Remove', 'configiview'),
            'removeGroupLabel' => __('Remove Group', 'configiview'),
            'optionColorsKey' => self::OPTION_COLORS,
            'optionPatternsKey' => self::OPTION_PATTERNS,
            'maxItems' => $max_items,
            'maxItemsMessage' => $max_items_message,
        ));
        wp_enqueue_script(
            'configiview-settings-page',
            CONFIGIVIEW_PLUGIN_URL . 'assets/js/settings-page.js',
            array('jquery', 'configiview-admin'),
            CONFIGIVIEW_VERSION,
            true
        );
        wp_localize_script('configiview-settings-page', 'configiViewSettingsPage', array(
            'groupLabel' => __('Group', 'configiview'),
            'optionLabel' => __('Option', 'configiview'),
            'nameLabel' => __('Name', 'configiview'),
            'addLabel' => __('Add', 'configiview'),
            'addOptionLabel' => __('Add Option', 'configiview'),
            'uploadImage' => __('Upload Image', 'configiview'),
            'remove' => __('Remove', 'configiview'),
            'toggleGroup' => __('Toggle group', 'configiview'),
            'shownFrontendOptions' => __('Shown in frontend options.', 'configiview'),
            'smallImageOptionTile' => __('Small image for option tile.', 'configiview'),
            'actionLabel' => __('Action', 'configiview'),
            'optionNamePlaceholder' => __('Option name', 'configiview'),
            'removeGroupConfirm' => __('Remove this group and all its options?', 'configiview'),
            'resetConfirm' => __('Are you sure you want to reset all configurator data? This action cannot be undone.', 'configiview'),
            'removeLicenseConfirm' => __('Remove saved license key from this site?', 'configiview'),
            'submitVisibleTabs' => array('colors-patterns', 'combo-previews', 'settings'),
            'optionExtraGroupsKey' => self::OPTION_EXTRA_GROUPS,
            'isProActive' => self::is_pro_active() ? 1 : 0,
        ));
    }
    
    public function add_admin_menu() {
        $menu_page_title = __('ConfigiView Settings', 'configiview');
        $menu_title = __('ConfigiView', 'configiview');

        add_menu_page(
            $menu_page_title,
            $menu_title,
            'manage_options',
            'configiview-settings',
            array($this, 'settings_page'),
            self::get_admin_menu_icon(),
            30
        );
    }
    
    public function register_settings() {
        register_setting('configiview_settings', self::OPTION_COLORS, array($this, 'sanitize_choices'));
        register_setting('configiview_settings', self::OPTION_PATTERNS, array($this, 'sanitize_choices'));
        register_setting('configiview_settings', self::OPTION_BASE_GROUP_VISIBILITY, array($this, 'sanitize_base_group_visibility'));
        register_setting('configiview_settings', self::OPTION_EXTRA_GROUPS, array($this, 'sanitize_extra_groups'));
        register_setting('configiview_settings', self::OPTION_COMBO_PREVIEWS, array($this, 'sanitize_combo_previews'));
        register_setting('configiview_settings', self::OPTION_COMBO_VISIBILITY, array($this, 'sanitize_combo_visibility'));
        register_setting('configiview_settings', self::OPTION_LABELS, array($this, 'sanitize_labels'));
        register_setting('configiview_settings', self::OPTION_EMAIL, 'sanitize_email');
        register_setting('configiview_settings', self::OPTION_CTA_SETTINGS, array($this, 'sanitize_cta_settings'));
        register_setting('configiview_settings', self::OPTION_FRONTEND_SETTINGS, array($this, 'sanitize_frontend_settings'));
    }

    public function sanitize_choices($items) {
        if (!is_array($items)) {
            return array();
        }

        $sanitized = array();
        $max_items = self::is_pro_active() ? 9999 : 5;

        foreach ($items as $item) {
            $name = isset($item['name']) ? sanitize_text_field($item['name']) : '';
            $image = isset($item['image']) ? esc_url_raw($item['image']) : '';
            $preview = isset($item['preview']) ? esc_url_raw($item['preview']) : '';
            $key = isset($item['key']) ? sanitize_title((string) $item['key']) : '';

            if ($name === '' && $image === '') {
                continue;
            }

            if ($preview === '') {
                $preview = $image;
            }

            if ($key === '') {
                $key = self::normalize_choice_key($name);
            }

            $sanitized[] = array(
                'name' => $name,
                'key' => $key,
                'image' => $image,
                'preview' => $preview,
            );

            if (count($sanitized) >= $max_items) {
                break;
            }
        }

        return $sanitized;
    }

    public function sanitize_combo_previews($items) {
        if (!is_array($items)) {
            return array();
        }

        $sanitized = array();

        foreach ($items as $combo_key => $preview_url) {
            $key = strtolower(trim((string) $combo_key));
            $key = preg_replace('/[^a-z0-9\-\|_]/', '', $key);
            $url = esc_url_raw((string) $preview_url);

            if ($key === '' || $url === '') {
                continue;
            }

            $sanitized[$key] = $url;
        }

        return $sanitized;
    }

    public function sanitize_combo_visibility($items) {
        if (!self::is_pro_active()) {
            $existing = get_option(self::OPTION_COMBO_VISIBILITY, array());
            return is_array($existing) ? $existing : array();
        }

        if (!is_array($items)) {
            return array();
        }

        $sanitized = array();
        foreach ($items as $combo_key => $state) {
            $key = strtolower(trim((string) $combo_key));
            $key = preg_replace('/[^a-z0-9\-\|_]/', '', $key);
            if ('' === $key) {
                continue;
            }
            $value = sanitize_key((string) $state);
            $sanitized[$key] = ('hide' === $value) ? 'hide' : 'show';
        }

        return $sanitized;
    }

    public function sanitize_extra_groups($groups) {
        if (!self::is_pro_active()) {
            return array();
        }

        if (!is_array($groups)) {
            return array();
        }

        $sanitized_groups = array();
        $max_groups = 20;
        $max_items = 9999;

        foreach ($groups as $group) {
            if (!is_array($group)) {
                continue;
            }

            $singular = isset($group['singular']) ? sanitize_text_field($group['singular']) : '';
            $plural = isset($group['plural']) ? sanitize_text_field($group['plural']) : '';
            $items = isset($group['items']) && is_array($group['items']) ? $group['items'] : array();

            if ('' === $singular && '' === $plural && empty($items)) {
                continue;
            }

            if ('' === $singular && '' !== $plural) {
                $singular = $plural;
            }
            if ('' === $plural && '' !== $singular) {
                $plural = $singular;
            }

            $sanitized_items = array();
            foreach ($items as $item) {
                if (!is_array($item)) {
                    continue;
                }

                $name = isset($item['name']) ? sanitize_text_field($item['name']) : '';
                $image = isset($item['image']) ? esc_url_raw($item['image']) : '';
                $preview = isset($item['preview']) ? esc_url_raw($item['preview']) : '';
                $key = isset($item['key']) ? sanitize_title((string) $item['key']) : '';

                if ('' === $name && '' === $image) {
                    continue;
                }

                if ('' === $preview) {
                    $preview = $image;
                }
                if ('' === $key) {
                    $key = self::normalize_choice_key($name);
                }

                $sanitized_items[] = array(
                    'name' => $name,
                    'key' => $key,
                    'image' => $image,
                    'preview' => $preview,
                );

                if (count($sanitized_items) >= $max_items) {
                    break;
                }
            }

            $sanitized_groups[] = array(
                'singular' => $singular,
                'plural' => $plural,
                'items' => $sanitized_items,
            );

            if (count($sanitized_groups) >= $max_groups) {
                break;
            }
        }

        return $sanitized_groups;
    }

    public function sanitize_base_group_visibility($visibility) {
        $defaults = self::get_default_base_group_visibility();
        if (!self::is_pro_active()) {
            return $defaults;
        }

        if (!is_array($visibility)) {
            return $defaults;
        }

        return array(
            'group_1' => isset($visibility['group_1']) && (int) $visibility['group_1'] === 0 ? 0 : 1,
            'group_2' => isset($visibility['group_2']) && (int) $visibility['group_2'] === 0 ? 0 : 1,
        );
    }

    public function sanitize_labels($labels) {
        $defaults = self::get_default_labels();
        if (!is_array($labels)) {
            return $defaults;
        }

        return array(
            'group_one_singular' => isset($labels['group_one_singular']) && $labels['group_one_singular'] !== ''
                ? sanitize_text_field($labels['group_one_singular'])
                : $defaults['group_one_singular'],
            'group_one_plural' => isset($labels['group_one_plural']) && $labels['group_one_plural'] !== ''
                ? sanitize_text_field($labels['group_one_plural'])
                : $defaults['group_one_plural'],
            'group_two_singular' => isset($labels['group_two_singular']) && $labels['group_two_singular'] !== ''
                ? sanitize_text_field($labels['group_two_singular'])
                : $defaults['group_two_singular'],
            'group_two_plural' => isset($labels['group_two_plural']) && $labels['group_two_plural'] !== ''
                ? sanitize_text_field($labels['group_two_plural'])
                : $defaults['group_two_plural'],
        );
    }

    public function sanitize_cta_settings($settings) {
        $defaults = self::get_default_cta_settings();
        if (!self::is_pro_active()) {
            return $defaults;
        }
        if (!is_array($settings)) {
            return $defaults;
        }

        $mode_raw = isset($settings['mode']) ? sanitize_key((string) $settings['mode']) : $defaults['mode'];
        $mode = in_array($mode_raw, array('mailto', 'shortcode'), true) ? $mode_raw : $defaults['mode'];

        return array(
            'mode' => $mode,
            'button_label' => isset($settings['button_label']) && '' !== trim((string) $settings['button_label'])
                ? sanitize_text_field((string) $settings['button_label'])
                : $defaults['button_label'],
            'subject_template' => isset($settings['subject_template']) && '' !== trim((string) $settings['subject_template'])
                ? sanitize_text_field((string) $settings['subject_template'])
                : $defaults['subject_template'],
            'body_template' => isset($settings['body_template']) && '' !== trim((string) $settings['body_template'])
                ? sanitize_textarea_field((string) $settings['body_template'])
                : $defaults['body_template'],
            'shortcode' => isset($settings['shortcode']) ? sanitize_text_field((string) $settings['shortcode']) : '',
        );
    }

    public function sanitize_frontend_settings($settings) {
        $defaults = self::get_default_frontend_settings();
        if (!self::is_pro_active()) {
            return $defaults;
        }
        if (!is_array($settings)) {
            return $defaults;
        }

        return array(
            'accordion_enabled' => isset($settings['accordion_enabled']) && (int) $settings['accordion_enabled'] === 1 ? 1 : 0,
        );
    }
    
    public function handle_reset_defaults() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to perform this action.', 'configiview'));
        }

        check_admin_referer('configiview_reset_defaults');

        update_option(self::OPTION_COLORS, array());
        update_option(self::OPTION_PATTERNS, array());
        update_option(self::OPTION_BASE_GROUP_VISIBILITY, self::get_default_base_group_visibility());
        update_option(self::OPTION_EXTRA_GROUPS, array());
        update_option(self::OPTION_COMBO_PREVIEWS, array());
        update_option(self::OPTION_COMBO_VISIBILITY, array());
        update_option(self::OPTION_LABELS, self::get_default_labels());
        update_option(self::OPTION_CTA_SETTINGS, self::get_default_cta_settings());
        update_option(self::OPTION_FRONTEND_SETTINGS, self::get_default_frontend_settings());
        update_option(self::OPTION_DEFAULTS_INITIALIZED, 1);

        $redirect_url = add_query_arg(
            array(
                'page' => 'configiview-settings',
                'configiview_reset' => '1',
                'configiview_notice_nonce' => wp_create_nonce('configiview_admin_notice'),
            ),
            admin_url('admin.php')
        );

        wp_safe_redirect($redirect_url);
        exit;
    }


    public function settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        include CONFIGIVIEW_PLUGIN_DIR . 'admin/settings-page.php';
    }
    
    public function render_visualizer($atts) {
        $atts = shortcode_atts(array(
            'show_cta' => 'yes',
        ), $atts);
        $atts['show_cta'] = ('no' === strtolower((string) $atts['show_cta'])) ? 'no' : 'yes';

        ob_start();
        include CONFIGIVIEW_PLUGIN_DIR . 'templates/visualizer.php';
        return ob_get_clean();
    }
}

// Initialize plugin
register_activation_hook(__FILE__, array('ConfigiView', 'activate'));
new ConfigiView();

if (!function_exists('configiview_is_pro')) {
    function configiview_is_pro() {
        return ConfigiView::is_pro_active();
    }
}
