(function($) {
    'use strict';

    $(document).ready(function() {
        const ajaxData = (typeof configiViewAjax !== 'undefined') ? configiViewAjax : {};
        const i18n = ajaxData.i18n || {};
        const notSelectedText = i18n.notSelected || 'Not selected';
        const ctaButtonLabel = i18n.ctaButtonLabel || 'Send Inquiry';
        const ctaMode = i18n.ctaMode || 'mailto';
        const ctaEmail = i18n.ctaEmail || '';
        const ctaSubjectTemplate = i18n.ctaSubjectTemplate || 'Inquiry about {group1} + {group2}';
        const ctaBodyTemplate = i18n.ctaBodyTemplate || 'Hello, I am interested in this configuration: {group1} + {group2}. Please send me an offer and more details.';
        const gmailFallbackPrompt = i18n.gmailFallbackPrompt || 'If your email app did not open, do you want to continue in Gmail?';
        const noComboPreviewText = i18n.noComboPreview || 'No preview for this combination yet.';
        const accordionEnabled = (($('.configiview').data('accordion-enabled') || '').toString() === '1');

        const FloorViz = {
            selectedColor: '',
            selectedPattern: '',
            selectedColorKey: '',
            selectedPatternKey: '',
            selectedByGroup: {},
            selectedKeyByGroup: {},
            optionGroupsMeta: [],
            comboPreviews: {},
            comboVisibility: {},
            isUpdatingAvailability: false,

            init: function() {
                this.comboPreviews = this.getComboPreviews();
                this.comboVisibility = this.getComboVisibility();
                this.optionGroupsMeta = this.getOptionGroupsMeta();
                this.ensureOptionLabels();
                this.bindEvents();
                this.initAccordion();
                this.applyCtaLabel();
                this.selectInitialOptions();
                this.refreshAvailability();
            },

            ensureOptionLabels: function() {
                $('.configiview-color-item, .configiview-pattern-item').each(function() {
                    const $item = $(this);
                    const itemName = ($item.data('name') || '').toString().trim();
                    if (!itemName) {
                        return;
                    }

                    const $existingLabel = $item.find('.configiview-label');
                    if ($existingLabel.length) {
                        $existingLabel.text(itemName).show();
                        return;
                    }

                    $item.append(
                        $('<div/>', {
                            class: 'configiview-label',
                            text: itemName
                        })
                    );
                });
            },

            bindEvents: function() {
                $('.configiview-color-item').on('click', function() {
                    const $this = $(this);
                    if ($this.hasClass('is-unavailable')) {
                        return;
                    }
                    const previewSrc = $this.data('preview') || $this.find('img').attr('src');
                    const colorName = $this.data('name') || $this.find('.configiview-label').text();
                    const colorKey = ($this.data('key') || '').toString();

                    FloorViz.changeFloor(previewSrc, $this, 'color', colorName, colorKey);
                });

                $('.configiview-pattern-item').on('click', function() {
                    const $this = $(this);
                    if ($this.hasClass('is-unavailable')) {
                        return;
                    }
                    const previewSrc = $this.data('preview') || $this.find('img').attr('src');
                    const patternName = $this.data('name') || $this.find('.configiview-label').text();
                    const patternKey = ($this.data('key') || '').toString();

                    FloorViz.changeFloor(previewSrc, $this, 'pattern', patternName, patternKey);
                });

                $('.configiview-option-item').on('click', function() {
                    const $this = $(this);
                    if ($this.hasClass('is-unavailable')) {
                        return;
                    }
                    const groupIndex = parseInt(($this.data('group-index') || '').toString(), 10);
                    if (Number.isNaN(groupIndex) || groupIndex < 2) {
                        return;
                    }

                    const previewSrc = $this.data('preview') || $this.find('img').attr('src');
                    const optionName = $this.data('name') || $this.find('.configiview-label').text();
                    const optionKey = ($this.data('key') || '').toString();

                    FloorViz.changeGroupOption(previewSrc, $this, groupIndex, optionName, optionKey);
                });

                $('.configiview-sidebar').on('click', '.configiview-sidebar-title', function(e) {
                    if (!FloorViz.isAccordionActive()) {
                        return;
                    }
                    e.preventDefault();
                    e.stopPropagation();
                    const $title = $(this);
                    const panelId = ($title.attr('aria-controls') || '').toString();
                    if (!panelId) {
                        return;
                    }
                    FloorViz.expandGroupPanelById(panelId);
                });

                $('.configiview-sidebar').on('keydown', '.configiview-sidebar-title', function(e) {
                    if (!FloorViz.isAccordionActive()) {
                        return;
                    }
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        $(this).trigger('click');
                    }
                });

                $('.configiview-quick-inquiry').on('click', function(e) {
                    if (ctaMode === 'shortcode' && $('#configiviewQuickInquiryForm').length) {
                        e.preventDefault();
                        $('#configiviewQuickInquiryForm').stop(true, true).slideToggle(180);
                        return;
                    }

                    e.preventDefault();
                    FloorViz.openInquiryWithFallback();
                });

            },

            isAccordionActive: function() {
                return accordionEnabled && Array.isArray(this.optionGroupsMeta) && this.optionGroupsMeta.length > 2;
            },

            initAccordion: function() {
                if (!this.isAccordionActive()) {
                    return;
                }

                const self = this;
                $('.configiview-sidebar-section').each(function(index) {
                    const $section = $(this);
                    const $panel = $section.find('.configiview-accordion-panel').first();
                    if (index === 0) {
                        $section.removeClass('is-collapsed').addClass('is-expanded');
                        $section.find('.configiview-sidebar-title').attr('aria-expanded', 'true');
                        $panel.stop(true, true).slideDown(0);
                    } else {
                        $section.addClass('is-collapsed').removeClass('is-expanded');
                        $section.find('.configiview-sidebar-title').attr('aria-expanded', 'false');
                        $panel.stop(true, true).slideUp(0);
                    }
                });

                const $activeSection = $('.configiview-option-item.active').first().closest('.configiview-sidebar-section');
                if ($activeSection.length) {
                    const activeIndex = parseInt(($activeSection.data('group-index') || '').toString(), 10);
                    if (!Number.isNaN(activeIndex)) {
                        self.expandGroupPanel(activeIndex);
                    }
                }
            },

            expandGroupPanel: function(groupIndex) {
                if (!this.isAccordionActive()) {
                    return;
                }
                $('.configiview-sidebar-section').each(function() {
                    const $section = $(this);
                    const idx = parseInt(($section.data('group-index') || '').toString(), 10);
                    const isTarget = !Number.isNaN(idx) && idx === groupIndex;
                    const $panel = $section.find('.configiview-accordion-panel').first();
                    $section.toggleClass('is-collapsed', !isTarget).toggleClass('is-expanded', isTarget);
                    $section.find('.configiview-sidebar-title').attr('aria-expanded', isTarget ? 'true' : 'false');
                    if (isTarget) {
                        $panel.stop(true, true).slideDown(220);
                    } else {
                        $panel.stop(true, true).slideUp(220);
                    }
                });
            },

            expandGroupPanelById: function(panelId) {
                if (!this.isAccordionActive()) {
                    return;
                }
                const targetId = (panelId || '').toString();
                if (!targetId) {
                    return;
                }
                $('.configiview-sidebar-section').each(function() {
                    const $section = $(this);
                    const $title = $section.find('.configiview-sidebar-title').first();
                    const controls = ($title.attr('aria-controls') || '').toString();
                    const isTarget = controls === targetId;
                    const $panel = $section.find('.configiview-accordion-panel').first();
                    $section.toggleClass('is-collapsed', !isTarget).toggleClass('is-expanded', isTarget);
                    $title.attr('aria-expanded', isTarget ? 'true' : 'false');
                    if (isTarget) {
                        $panel.stop(true, true).slideDown(220);
                    } else {
                        $panel.stop(true, true).slideUp(220);
                    }
                });
            },

            applyCtaLabel: function() {
                $('.configiview-quick-inquiry').text(ctaButtonLabel);
                if (ctaMode === 'mailto') {
                    this.updateInquiryHref();
                } else {
                    $('#configiviewQuickInquiry').attr('href', '#configiviewQuickInquiryForm');
                }
            },

            selectInitialOptions: function() {
                const $firstColor = $('.configiview-color-item').first();
                const $firstPattern = $('.configiview-pattern-item').first();

                if ($firstColor.length) {
                    $firstColor.trigger('click');
                }

                if ($firstPattern.length) {
                    $firstPattern.trigger('click');
                }

                const handled = {0: true, 1: true};
                $('.configiview-option-item').each(function() {
                    const $item = $(this);
                    const groupIndex = parseInt(($item.data('group-index') || '').toString(), 10);
                    if (Number.isNaN(groupIndex) || handled[groupIndex]) {
                        return;
                    }

                    handled[groupIndex] = true;
                    $item.trigger('click');
                });

                if (this.isAccordionActive()) {
                    const $firstSection = $('.configiview-sidebar-section').first();
                    const firstIndex = parseInt(($firstSection.data('group-index') || '').toString(), 10);
                    if (!Number.isNaN(firstIndex)) {
                        this.expandGroupPanel(firstIndex);
                    }
                }
            },

            getComboPreviews: function() {
                const previewElement = $('.configiview-preview');
                const comboAttr = previewElement.attr('data-combo-previews');
                if (!comboAttr) {
                    return {};
                }

                try {
                    const parsed = JSON.parse(comboAttr);
                    return parsed && typeof parsed === 'object' ? parsed : {};
                } catch (error) {
                    return {};
                }
            },

            getOptionGroupsMeta: function() {
                const previewElement = $('.configiview-preview');
                const raw = previewElement.attr('data-option-groups');
                if (!raw) {
                    return [];
                }

                try {
                    const parsed = JSON.parse(raw);
                    return Array.isArray(parsed) ? parsed : [];
                } catch (error) {
                    return [];
                }
            },

            getComboVisibility: function() {
                const previewElement = $('.configiview-preview');
                const raw = previewElement.attr('data-combo-visibility');
                if (!raw) {
                    return {};
                }

                try {
                    const parsed = JSON.parse(raw);
                    return parsed && typeof parsed === 'object' ? parsed : {};
                } catch (error) {
                    return {};
                }
            },

            isComboVisible: function(comboKey) {
                const key = (comboKey || '').toString().trim().toLowerCase();
                if (!key) {
                    return true;
                }
                const state = (this.comboVisibility[key] || 'show').toString().toLowerCase();
                return state !== 'hide';
            },

            getItemsForGroup: function(groupIndex) {
                if (groupIndex === 0) {
                    return $('.configiview-color-item');
                }
                if (groupIndex === 1) {
                    return $('.configiview-pattern-item');
                }
                return $('.configiview-option-item[data-group-index="' + groupIndex + '"]');
            },

            getAllGroupKeys: function() {
                const keysByGroup = {};
                if (!Array.isArray(this.optionGroupsMeta)) {
                    return keysByGroup;
                }
                for (let i = 0; i < this.optionGroupsMeta.length; i += 1) {
                    const group = this.optionGroupsMeta[i] || {};
                    const groupIndex = parseInt(group.index, 10);
                    if (Number.isNaN(groupIndex)) {
                        continue;
                    }
                    const keys = [];
                    this.getItemsForGroup(groupIndex).each(function() {
                        const key = ($(this).data('key') || '').toString().trim().toLowerCase();
                        if (key) {
                            keys.push(key);
                        }
                    });
                    keysByGroup[groupIndex] = keys;
                }
                return keysByGroup;
            },

            hasVisibleComboForChoice: function(groupIndex, candidateKey) {
                const keysByGroup = this.getAllGroupKeys();
                const constraints = {};

                if (!Array.isArray(this.optionGroupsMeta) || !keysByGroup[groupIndex] || keysByGroup[groupIndex].length === 0) {
                    return false;
                }

                for (let i = 0; i < this.optionGroupsMeta.length; i += 1) {
                    const group = this.optionGroupsMeta[i] || {};
                    const idx = parseInt(group.index, 10);
                    if (Number.isNaN(idx)) {
                        continue;
                    }

                    if (idx === groupIndex) {
                        constraints[idx] = [candidateKey];
                        continue;
                    }
                    const selected = (this.selectedKeyByGroup[idx] || '').toString().trim().toLowerCase();
                    if (selected) {
                        constraints[idx] = [selected];
                    } else {
                        constraints[idx] = keysByGroup[idx] || [];
                    }

                    if (!constraints[idx].length) {
                        return false;
                    }
                }

                const groupIndexes = Object.keys(constraints).map(function(v) { return parseInt(v, 10); }).sort(function(a, b) { return a - b; });
                let hasVisible = false;
                const self = this;
                const walk = function(depth, parts) {
                    if (hasVisible) {
                        return;
                    }
                    if (depth >= groupIndexes.length) {
                        const comboKey = parts.join('|');
                        if (self.isComboVisible(comboKey)) {
                            hasVisible = true;
                        }
                        return;
                    }

                    const idx = groupIndexes[depth];
                    const list = constraints[idx] || [];
                    for (let i = 0; i < list.length; i += 1) {
                        const key = (list[i] || '').toString().trim().toLowerCase();
                        if (!key) {
                            continue;
                        }
                        const next = parts.slice();
                        next.push(key);
                        walk(depth + 1, next);
                        if (hasVisible) {
                            return;
                        }
                    }
                };
                walk(0, []);
                return hasVisible;
            },

            refreshAvailability: function() {
                const self = this;
                if (self.isUpdatingAvailability) {
                    return;
                }
                self.isUpdatingAvailability = true;

                if (Array.isArray(self.optionGroupsMeta)) {
                    for (let i = 0; i < self.optionGroupsMeta.length; i += 1) {
                        const group = self.optionGroupsMeta[i] || {};
                        const groupIndex = parseInt(group.index, 10);
                        if (Number.isNaN(groupIndex)) {
                            continue;
                        }

                        self.getItemsForGroup(groupIndex).each(function() {
                            const $item = $(this);
                            const key = ($item.data('key') || '').toString().trim().toLowerCase();
                            if (!key) {
                                $item.addClass('is-unavailable').attr('aria-disabled', 'true');
                                return;
                            }

                            const isAvailable = self.hasVisibleComboForChoice(groupIndex, key);
                            if (isAvailable) {
                                $item.removeClass('is-unavailable').removeAttr('aria-disabled');
                            } else {
                                $item.addClass('is-unavailable').attr('aria-disabled', 'true');
                            }
                        });

                        const $active = self.getItemsForGroup(groupIndex).filter('.active').first();
                        if (!$active.length) {
                            const $firstAvailable = self.getItemsForGroup(groupIndex).not('.is-unavailable').first();
                            if ($firstAvailable.length) {
                                $firstAvailable.trigger('click');
                            }
                        } else if ($active.hasClass('is-unavailable')) {
                            const $firstAvailable = self.getItemsForGroup(groupIndex).not('.is-unavailable').first();
                            if ($firstAvailable.length) {
                                $firstAvailable.trigger('click');
                            }
                        }
                    }
                }

                self.isUpdatingAvailability = false;
            },

            getCombinedPreviewSrc: function(fallbackSrc) {
                if (!Array.isArray(this.optionGroupsMeta) || this.optionGroupsMeta.length < 2) {
                    return {
                        src: fallbackSrc,
                        missingCombo: false
                    };
                }

                const comboKeys = [];
                for (let i = 0; i < this.optionGroupsMeta.length; i += 1) {
                    const group = this.optionGroupsMeta[i] || {};
                    const groupIndex = parseInt(group.index, 10);
                    if (Number.isNaN(groupIndex)) {
                        continue;
                    }

                    const selectedKey = (this.selectedKeyByGroup[groupIndex] || '').toString().trim();
                    if (!selectedKey) {
                        return {
                            src: fallbackSrc,
                            missingCombo: false
                        };
                    }
                    comboKeys.push(selectedKey);
                }

                const comboKey = comboKeys.join('|');
                if (this.comboPreviews[comboKey]) {
                    if (!this.isComboVisible(comboKey)) {
                        return {
                            src: '',
                            missingCombo: true
                        };
                    }
                    return {
                        src: this.comboPreviews[comboKey],
                        missingCombo: false
                    };
                }

                return {
                    src: '',
                    missingCombo: true
                };
            },

            renderPreview: function(fallbackSrc) {
                const previewState = this.getCombinedPreviewSrc(fallbackSrc);
                const $preview = $('.configiview-preview');
                const $emptyNote = $preview.find('.configiview-preview-empty-note');

                if (previewState.missingCombo || !previewState.src) {
                    $preview
                        .css('background-image', 'none')
                        .addClass('is-empty')
                        .removeClass('loading');
                    $emptyNote.text(noComboPreviewText);
                    return;
                }

                $preview
                    .css('background-image', 'url(' + previewState.src + ')')
                    .removeClass('is-empty')
                    .removeClass('loading');
                $emptyNote.text('');
            },

            changeFloor: function(imageSrc, $element, type, name, key) {
                $('.configiview-preview').addClass('loading');

                const normalizedKey = (key || '').toLowerCase();
                if (type === 'color') {
                    this.selectedColorKey = normalizedKey;
                } else {
                    this.selectedPatternKey = normalizedKey;
                }

                const itemClass = type === 'color' ? '.configiview-color-item' : '.configiview-pattern-item';
                $(itemClass).removeClass('active');
                $element.addClass('active');

                if (type === 'color') {
                    this.selectedColor = name;
                    $('#configiviewSelectedColor').text(name || notSelectedText);
                    this.selectedByGroup[0] = name || '';
                    this.selectedKeyByGroup[0] = normalizedKey;
                } else {
                    this.selectedPattern = name;
                    $('#configiviewSelectedPattern').text(name || notSelectedText);
                    this.selectedByGroup[1] = name || '';
                    this.selectedKeyByGroup[1] = normalizedKey;
                }

                this.renderPreview(imageSrc);
                this.updateInquiryHref();
                this.refreshAvailability();
            },

            changeGroupOption: function(imageSrc, $element, groupIndex, name, key) {
                const normalizedKey = (key || '').toLowerCase();
                this.selectedByGroup[groupIndex] = name || '';
                this.selectedKeyByGroup[groupIndex] = normalizedKey;

                $('.configiview-option-item[data-group-index="' + groupIndex + '"]').removeClass('active');
                $element.addClass('active');

                const $selection = $('#configiviewSelectedGroup' + groupIndex);
                if ($selection.length) {
                    $selection.text(name || notSelectedText);
                }

                this.renderPreview(imageSrc);
                this.updateInquiryHref();
                this.refreshAvailability();
            },

            injectSelection: function(template) {
                const placeholderValues = {
                    color: this.selectedColor || notSelectedText,
                    pattern: this.selectedPattern || notSelectedText
                };

                if (Array.isArray(this.optionGroupsMeta)) {
                    for (let i = 0; i < this.optionGroupsMeta.length; i += 1) {
                        const group = this.optionGroupsMeta[i] || {};
                        const groupIndex = parseInt(group.index, 10);
                        if (Number.isNaN(groupIndex)) {
                            continue;
                        }
                        const tokenKey = 'group' + (groupIndex + 1);
                        placeholderValues[tokenKey] = (this.selectedByGroup[groupIndex] || notSelectedText).toString();
                    }
                }

                return String(template).replace(/\{([a-z0-9_]+)\}/gi, function(match, tokenName) {
                    const normalized = String(tokenName || '').toLowerCase();
                    if (Object.prototype.hasOwnProperty.call(placeholderValues, normalized)) {
                        return placeholderValues[normalized];
                    }
                    return match;
                });
            },

            updateInquiryHref: function() {
                if (ctaMode !== 'mailto') {
                    return;
                }
                const recipient = String(ctaEmail).trim();
                const $cta = $('#configiviewQuickInquiry');
                if (!$cta.length) {
                    return;
                }

                const subject = this.injectSelection(ctaSubjectTemplate);
                const body = this.injectSelection(ctaBodyTemplate);
                const mailto = 'mailto:' + recipient +
                    '?subject=' + encodeURIComponent(subject) +
                    '&body=' + encodeURIComponent(body);

                $cta.attr('href', mailto);
            },

            openInquiryWithFallback: function() {
                const recipient = String(ctaEmail).trim();
                const subject = this.injectSelection(ctaSubjectTemplate);
                const body = this.injectSelection(ctaBodyTemplate);
                const mailto = 'mailto:' + recipient +
                    '?subject=' + encodeURIComponent(subject) +
                    '&body=' + encodeURIComponent(body);
                const gmailComposeUrl = 'https://mail.google.com/mail/?view=cm&fs=1' +
                    '&to=' + encodeURIComponent(recipient) +
                    '&su=' + encodeURIComponent(subject) +
                    '&body=' + encodeURIComponent(body);

                let timerId = null;
                const clearFallback = function() {
                    if (timerId) {
                        window.clearTimeout(timerId);
                        timerId = null;
                    }
                    window.removeEventListener('blur', clearFallback);
                    document.removeEventListener('visibilitychange', onVisibilityChange);
                };
                const onVisibilityChange = function() {
                    if (document.visibilityState === 'hidden') {
                        clearFallback();
                    }
                };

                window.addEventListener('blur', clearFallback);
                document.addEventListener('visibilitychange', onVisibilityChange);

                timerId = window.setTimeout(function() {
                    clearFallback();
                    if (window.confirm(gmailFallbackPrompt)) {
                        window.open(gmailComposeUrl, '_blank', 'noopener,noreferrer');
                    }
                }, 1200);

                window.location.href = mailto;
            }
        };

        if ($('.configiview').length) {
            FloorViz.init();
        }
    });

})(jQuery);
