<?php

/**
 * ConFab Plugin Uninstall Handler - WordPress.org Compliant
 * FILE: uninstall.php
 *
 * Handles plugin uninstallation with user-controlled data deletion and Plugin Check
 * compliance. Implements conservative defaults to preserve conference data unless
 * explicitly configured for complete removal by the user.
 *
 * @package confab
 * @since 2.0.0
 */

// Exit if uninstall not called from WordPress
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

/**
 * ConFab Uninstall Data Management
 * 
 * Respects user preference for data retention vs complete removal with Plugin Check
 * compliant database operations and comprehensive error handling.
 * Default behavior: Preserve all session data, remove only settings.
 * Nuclear option: Complete removal when explicitly enabled by user.
 */

// Check user preference for data deletion with Plugin Check compliant pattern
$confab_initial_confirmed = get_option( 'confab_delete_initial_confirmed', false );
$confab_final_confirmed = get_option( 'confab_delete_final_confirmed', false );
$confab_delete_all_data = $confab_initial_confirmed && $confab_final_confirmed;

if ( $confab_delete_all_data ) {
	// User has explicitly chosen complete data removal
	confab_nuclear_uninstall();
} else {
	// Conservative default: preserve data, remove only settings
	confab_conservative_uninstall();
}

/**
 * Conservative Uninstall: Preserve Conference Data
 * 
 * Removes only plugin settings while preserving all conference sessions,
 * tracks, locations, and associated metadata. This is the safe default
 * that protects valuable conference data from accidental loss.
 *
 * @since 2.0.0
 */
function confab_conservative_uninstall() {
	// Remove only plugin settings - preserve all conference data
	$settings_to_remove = array(
		'confab_field_byline',
		'confab_field_schedule_page_url',
		'confab_excerpt_length',
		'confab_delete_initial_confirmed',
		'confab_delete_final_confirmed',
		// Legacy settings from migration
		'wpcs_field_byline',
		'wpcs_field_schedule_page_url',
		'confab_version',
		'confab_migration_complete',
	);

	foreach ( $settings_to_remove as $setting ) {
		delete_option( $setting );
	}

	// Clear WordPress object cache for clean removal
	wp_cache_flush();
}

/**
 * Nuclear Uninstall: Complete Data Removal
 * 
 * Permanently removes ALL ConFab data including sessions, taxonomies,
 * metadata, and settings with Plugin Check compliant database operations.
 * This operation cannot be undone and should only be used when user has
 * explicitly confirmed data deletion.
 *
 * @since 2.0.0
 */
function confab_nuclear_uninstall() {
	global $wpdb;

	// Get all confab_session posts for comprehensive metadata cleanup
	$session_posts = get_posts( array(
		'post_type'      => 'confab_session',
		'post_status'    => 'any',
		'numberposts'    => -1,
		'fields'         => 'ids',
	) );

	// Delete all session posts and their metadata using WordPress functions
	foreach ( $session_posts as $post_id ) {
		// Delete all post metadata before we delete the post.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom post type cleanup requires direct database operations for complete metadata removal
		$wpdb->delete(
			$wpdb->postmeta,
			array( 'post_id' => $post_id ),
			array( '%d' )
		);
		
		// Delete the post using WordPress function
		wp_delete_post( $post_id, true );
	}

	// Safety net: Remove any remaining confab_session posts
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom post type cleanup for complete plugin data removal
	$wpdb->delete(
		$wpdb->posts,
		array( 'post_type' => 'confab_session' ),
		array( '%s' )
	);

	// Delete confab_track taxonomy and all terms using WordPress taxonomy functions
	$track_terms = get_terms( array(
		'taxonomy'   => 'confab_track',
		'hide_empty' => false,
		'fields'     => 'ids',
	) );

	if ( ! is_wp_error( $track_terms ) && ! empty( $track_terms ) ) {
		foreach ( $track_terms as $term_id ) {
			wp_delete_term( $term_id, 'confab_track' );
		}
	}

	// Delete confab_location taxonomy and all terms using WordPress taxonomy functions
	$location_terms = get_terms( array(
		'taxonomy'   => 'confab_location',
		'hide_empty' => false,
		'fields'     => 'ids',
	) );

	if ( ! is_wp_error( $location_terms ) && ! empty( $location_terms ) ) {
		foreach ( $location_terms as $term_id ) {
			wp_delete_term( $term_id, 'confab_location' );
		}
	}

	// Clean up legacy data from WP Conference Schedule migration
	confab_cleanup_legacy_data();

	// Remove all ConFab options using WordPress functions
	$options_to_remove = array(
		'confab_field_byline',
		'confab_field_schedule_page_url',
		'confab_excerpt_length',
		'confab_delete_initial_confirmed',
		'confab_delete_final_confirmed',
		// Legacy options from migration
		'wpcs_field_byline',
		'wpcs_field_schedule_page_url',
		'wpcs_version',
		'confab_version',
		'confab_migration_complete',
	);

	foreach ( $options_to_remove as $option ) {
		delete_option( $option );
	}

	// Clean up legacy metadata with Plugin Check compliant prepared statement
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Legacy metadata cleanup requires direct database operations for complete plugin migration data removal
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
			'_wpcs_%'
		)
	);

	// Clean up all remaining ConFab metadata with Plugin Check compliant prepared statement
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- ConFab metadata cleanup requires direct database operations for complete plugin data removal
	$wpdb->query(
		$wpdb->prepare(
			"DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
			'_confab_%'
		)
	);

	// Clear all WordPress caches for clean removal
	wp_cache_flush();
	
	// Clean up rewrite rules for custom post types and taxonomies
	flush_rewrite_rules();
}

/**
 * Clean up legacy WP Conference Schedule data with Plugin Check compliance.
 * 
 * Removes residual data from original plugin migration with proper database
 * operations, including wpcs_session posts and wpcs_track taxonomy terms.
 *
 * @since 2.0.0
 */
function confab_cleanup_legacy_data() {
	global $wpdb;

	// Remove legacy wpcs_session posts using WordPress functions
	$legacy_posts = get_posts( array(
		'post_type'      => 'wpcs_session',
		'post_status'    => 'any',
		'numberposts'    => -1,
		'fields'         => 'ids',
	) );

	foreach ( $legacy_posts as $post_id ) {
		wp_delete_post( $post_id, true );
	}

	// Safety net: Remove legacy wpcs_session post type posts with Plugin Check compliant operation
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Legacy post type cleanup requires direct database operations for complete migration data removal
	$wpdb->delete(
		$wpdb->posts,
		array( 'post_type' => 'wpcs_session' ),
		array( '%s' )
	);

	// Delete legacy wpcs_track taxonomy terms using WordPress taxonomy functions
	$legacy_track_terms = get_terms( array(
		'taxonomy'   => 'wpcs_track',
		'hide_empty' => false,
		'fields'     => 'ids',
	) );

	if ( ! is_wp_error( $legacy_track_terms ) && ! empty( $legacy_track_terms ) ) {
		foreach ( $legacy_track_terms as $term_id ) {
			wp_delete_term( $term_id, 'wpcs_track' );
		}
	}

	// Delete legacy wpcs_location taxonomy terms using WordPress taxonomy functions
	$legacy_location_terms = get_terms( array(
		'taxonomy'   => 'wpcs_location',
		'hide_empty' => false,
		'fields'     => 'ids',
	) );

	if ( ! is_wp_error( $legacy_location_terms ) && ! empty( $legacy_location_terms ) ) {
		foreach ( $legacy_location_terms as $term_id ) {
			wp_delete_term( $term_id, 'wpcs_location' );
		}
	}

	// Remove legacy options with comprehensive cleanup
	$legacy_options = array(
		'wpcs_version',
		'wpcs_migration_timestamp',
		'wpcs_field_schedule_page_url',
		'wpcs_field_byline',
	);

	foreach ( $legacy_options as $option ) {
		delete_option( $option );
	}
}