<?php
/**
 * Single Session Template
 * FILE: templates/session-template.php
 *
 * Template for displaying individual conference session posts with time, speaker,
 * track, and location information. Includes accessibility features and security validation.
 *
 * @package confab
 * @subpackage Templates
 * @since 2.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

get_header(); ?>

<section id="primary" class="content-area">
	<main id="main" class="site-main">

		<?php while ( have_posts() ) : ?>
			<?php
		the_post();
		
		// Get session data with security validation
		$confab_session_data = confab_get_template_session_data( get_the_ID() );
		?>

		<article id="post-<?php the_ID(); ?>" <?php post_class( 'confab-single-session' ); ?>>

			<header class="entry-header">
				<?php the_title( '<h1 class="entry-title confab-single-session-title">', '</h1>' ); ?>
				
				<?php if ( $confab_session_data['time_display'] ) : ?>
					<div class="confab-session-meta confab-session-time">
						<?php echo wp_kses_post( $confab_session_data['time_display'] ); ?>
					</div>
				<?php endif; ?>
			</header>

			<div class="entry-content">
				
				<?php if ( $confab_session_data['speakers_html'] ) : ?>
					<div class="confab-session-meta confab-session-speakers">
						<?php echo wp_kses_post( $confab_session_data['speakers_html'] ); ?>
					</div>
				<?php endif; ?>
					
					<div class="entry-meta confab-single-session-meta">
						<?php confab_render_session_taxonomies(); ?>
					</div>

					<?php if ( $session_data['speakers_html'] ) : ?>
						<div class="confab-session-speakers-header">
							<?php echo wp_kses_post( $session_data['speakers_html'] ); ?>
						</div>
					<?php endif; ?>

				</header>

				<div class="entry-content">
					<?php
					the_content(
						sprintf(
							wp_kses(
								/* translators: %s: Name of current post. Only visible to screen readers */
								__( 'Continue reading<span class="screen-reader-text"> "%s"</span>', 'confab' ),
								array(
									'span' => array(
										'class' => array(),
									),
								)
							),
							wp_kses_post( get_the_title() )
						)
					);

					wp_link_pages(
						array(
							'before' => '<div class="page-links">' . esc_html__( 'Pages:', 'confab' ),
							'after'  => '</div>',
						)
					);
					?>
				</div>
				
				<?php confab_render_return_to_schedule_link(); ?>

			</article>

		<?php endwhile; ?>

	</main>
</section>

<?php
get_sidebar();
get_footer();

/**
 * Get validated session data for template display with comprehensive security checks.
 *
 * @param int $post_id WordPress post ID for session
 * @return array Sanitized session data for template display
 * @since 2.0.0
 */
function confab_get_template_session_data( $post_id ) {
	$post_id = absint( $post_id );
	
	if ( ! $post_id ) {
		return confab_get_empty_session_data();
	}

	$post = get_post( $post_id );
	
	if ( ! $post || 'confab_session' !== $post->post_type ) {
		return confab_get_empty_session_data();
	}

	$time_format = get_option( 'time_format', 'g:i a' );
	$session_time = absint( confab_get_session_meta_compat( $post_id, '_confab_session_time' ) );
	$session_end_time = absint( confab_get_session_meta_compat( $post_id, '_confab_session_end_time' ) );
	$session_type = confab_get_session_meta_compat( $post_id, '_confab_session_type' );
	$session_speakers_text = confab_get_session_meta_compat( $post_id, '_confab_session_speakers' );

	// Validate session type
	if ( ! in_array( $session_type, array( 'session', 'custom', 'mainstage' ), true ) ) {
		$session_type = 'session';
	}

	// Build time display string
	$time_display = '';
	if ( $session_time ) {
		$session_date = wp_date( 'F j, Y', $session_time );
		
		if ( $session_end_time ) {
			$time_display = sprintf(
				/* translators: 1: session date, 2: start time, 3: end time */
				__( '%1$s from %2$s to %3$s', 'confab' ),
				esc_html( $session_date ),
				esc_html( wp_date( $time_format, $session_time ) ),
				esc_html( wp_date( $time_format, $session_end_time ) )
			);
		} else {
			$time_display = sprintf(
				/* translators: 1: session date, 2: session time */
				__( '%1$s at %2$s', 'confab' ),
				esc_html( $session_date ),
				esc_html( wp_date( $time_format, $session_time ) )
			);
		}
	}

	// Build speakers HTML with security validation
	$speakers_html = '';
	if ( $session_speakers_text ) {
		$speakers_html = sprintf(
			'<div class="confab-single-session-speakers"><strong>%s</strong> %s</div>',
			esc_html__( 'Speaker:', 'confab' ),
			esc_html( $session_speakers_text )
		);
		
		// Apply filter with proper validation
		$filtered_speakers = apply_filters( 'confab_filter_single_session_speakers', $speakers_html, $post_id );
		if ( is_string( $filtered_speakers ) && $filtered_speakers !== $post_id ) {
			$speakers_html = $filtered_speakers;
		}
	}

	return array(
		'post_id'       => $post_id,
		'session_time'  => $session_time,
		'session_end_time' => $session_end_time,
		'session_type'  => $session_type,
		'time_display'  => $time_display,
		'speakers_html' => $speakers_html,
	);
}

/**
 * Get empty session data array for error conditions.
 *
 * @return array Empty session data structure
 * @since 2.0.0
 */
function confab_get_empty_session_data() {
	return array(
		'post_id'       => 0,
		'session_time'  => 0,
		'session_end_time' => 0,
		'session_type'  => 'session',
		'time_display'  => '',
		'speakers_html' => '',
	);
}

/**
 * Render session taxonomy information with proper escaping and accessibility.
 *
 * @since 2.0.0
 */
function confab_render_session_taxonomies() {
	$post_id = get_the_ID();
	
	if ( ! $post_id ) {
		return;
	}

	echo '<ul class="confab-single-session-taxonomies">';
	
	// Display tracks
	confab_render_taxonomy_terms( $post_id, 'confab_track', 'fas fa-columns', __( 'Track:', 'confab' ) );
	
	// Display locations  
	confab_render_taxonomy_terms( $post_id, 'confab_location', 'fas fa-map-marker-alt', __( 'Location:', 'confab' ) );
	
	// Allow other plugins to add taxonomy displays
	do_action( 'confab_single_taxonomies', $post_id );
	
	echo '</ul>';
}

/**
 * Render individual taxonomy terms with icon and proper escaping.
 *
 * @param int    $post_id   WordPress post ID
 * @param string $taxonomy  Taxonomy name to display
 * @param string $icon      Font Awesome icon class
 * @param string $label     Screen reader label for taxonomy
 * @since 2.0.0
 */
function confab_render_taxonomy_terms( $post_id, $taxonomy, $icon, $label ) {
	$terms = get_the_terms( $post_id, $taxonomy );
	
	if ( is_wp_error( $terms ) || empty( $terms ) ) {
		return;
	}

	$term_names = wp_list_pluck( $terms, 'name' );
	$terms_string = implode( ', ', array_map( 'esc_html', $term_names ) );
	
	if ( $terms_string ) {
		printf(
			'<li class="confab-single-session-taxonomies-taxonomy confab-single-session-%s">' .
			'<i class="%s" aria-hidden="true"></i>' .
			'<span class="screen-reader-text">%s</span>' .
			'%s</li>',
			esc_attr( str_replace( 'confab_', '', $taxonomy ) ),
			esc_attr( $icon ),
			esc_html( $label ),
			esc_html( $terms_string )
		);
	}
}

/**
 * Render return to schedule link with URL validation.
 *
 * @since 2.0.0
 */
function confab_render_return_to_schedule_link() {
	$schedule_page_url = get_option( 'confab_field_schedule_page_url' );
	
	if ( ! $schedule_page_url ) {
		return;
	}

	$validated_url = esc_url_raw( $schedule_page_url );
	
	if ( ! filter_var( $validated_url, FILTER_VALIDATE_URL ) ) {
		return;
	}

	?>
	<footer class="entry-footer">	
		<p>
			<a href="<?php echo esc_url( $validated_url ); ?>" 
			   class="confab-return-to-schedule">
				<?php esc_html_e( 'Return to Schedule', 'confab' ); ?>
			</a>
		</p>
	</footer>
	<?php
}