<?php
/**
 * ConFab Time Migration Tool
 * 
 * Allows users to correct session times by comparing displayed vs expected times
 * and applying calculated offsets to all sessions.
 *
 * @package ConFab
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register migration tool page (no menu item - accessed via Settings page button)
 * 
 * This creates the page handler without adding a menu item, so users access it
 * through the button on the ConFab Settings page instead of cluttering the
 * Sessions menu with an admin-only tool.
 */
function confab_register_time_migration_page() {
	// Register the page without adding to menu - accessed via Settings page button
	add_submenu_page(
		null, // No parent menu = page exists but no menu item shown
		__( 'Time Migration Tool', 'confab' ),
		__( 'Time Migration', 'confab' ),
		'manage_options',
		'confab-time-migration',
		'confab_render_time_migration_page'
	);
}
add_action( 'admin_menu', 'confab_register_time_migration_page', 20 );

/**
 * Enqueue admin styles for migration page
 */
function confab_migration_enqueue_styles( $hook_suffix ) {
	// Only load on migration page
	if ( 'confab_session_page_confab-time-migration' === $hook_suffix ) {
		$version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : CONFAB_VERSION;
		
		wp_enqueue_style(
			'confab-admin-modern',
			CONFAB_URL . 'assets/css/confab-admin-modern.css',
			array(),
			$version
		);
		
		wp_enqueue_script(
			'confab-admin-settings',
			CONFAB_URL . 'assets/js/confab-admin-settings.js',
			array( 'jquery' ),
			$version,
			true
		);
	}
}
add_action( 'admin_enqueue_scripts', 'confab_migration_enqueue_styles' );

/**
 * Render the time migration tool page
 */
function confab_render_time_migration_page() {
	// Check user capabilities.
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions.', 'confab' ) );
	}

	// Handle form submission.
	$message = '';
	$message_type = '';
	
	if ( isset( $_POST['confab_time_migration_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_time_migration_nonce'] ) ), 'confab_time_migration_action' ) ) {
		if ( isset( $_POST['confab_apply_migration'] ) ) {
			$result = confab_apply_time_migration();
			if ( is_wp_error( $result ) ) {
				$message = $result->get_error_message();
				$message_type = 'error';
			} else {
				$message = sprintf(
					/* translators: %d: number of sessions updated */
					__( 'Success! Updated %d sessions with the calculated time offset.', 'confab' ),
					$result
				);
				$message_type = 'success';
			}
		}
	}

	// Get current step.
	$step = isset( $_GET['step'] ) ? intval( $_GET['step'] ) : 1;
	$sample_date = isset( $_POST['sample_date'] ) ? sanitize_text_field( wp_unslash( $_POST['sample_date'] ) ) : '';
	
	?>
	<div class="wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
	</div>
	
	<div class="confab-admin-wrapper">
		
		<!-- Modern Hero Banner -->
		<div class="confab-hero">
			<div class="confab-hero-content">
				<div class="confab-hero-branding">
					<img src="<?php echo esc_url( CONFAB_URL . 'assets/images/confab-logo.svg' ); ?>" 
						 alt="<?php esc_attr_e( 'ConFab Logo', 'confab' ); ?>" 
						 class="confab-hero-logo">
					<div class="confab-hero-text">
						<h1 class="confab-hero-title"><?php esc_html_e( 'ConFab • Time Migration Tool', 'confab' ); ?></h1>
						<p class="confab-hero-subtitle"><?php esc_html_e( 'Correct session times after migration or timezone changes', 'confab' ); ?></p>
					</div>
				</div>
			</div>
		</div>

		<?php if ( $message ) : ?>
			<div class="notice notice-<?php echo esc_attr( $message_type ); ?> is-dismissible">
				<p><?php echo esc_html( $message ); ?></p>
			</div>
		<?php endif; ?>

		<div class="confab-migration-container">
			<!-- Back to Settings Button -->
			<div style="margin-bottom: 20px;">
				<a href="<?php echo esc_url( admin_url( 'themes.php?page=confab-settings' ) ); ?>" class="button button-secondary">
					<span class="dashicons dashicons-arrow-left-alt" style="margin-top: 3px;"></span>
					<?php esc_html_e( 'Back to ConFab Settings', 'confab' ); ?>
				</a>
			</div>

			<!-- Collapsible Timezone Debug Information -->
			<details class="confab-timezone-debug">
				<summary class="confab-timezone-summary">
					<?php esc_html_e( '🔍 Show Timezone Information (for troubleshooting)', 'confab' ); ?>
				</summary>
				<div class="confab-timezone-content">
					<table class="widefat confab-timezone-table">
						<tr>
							<th><?php esc_html_e( 'Server Timezone:', 'confab' ); ?></th>
							<td><code><?php echo esc_html( date_default_timezone_get() ); ?></code></td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Server Offset from UTC:', 'confab' ); ?></th>
							<td>
								<?php
								$server_tz = new DateTimeZone( date_default_timezone_get() );
								$server_dt = new DateTime( 'now', $server_tz );
								$server_offset = $server_tz->getOffset( $server_dt );
								$server_offset_hours = $server_offset / 3600;
								echo esc_html( sprintf( '%d seconds (%+.1f hours)', $server_offset, $server_offset_hours ) );
								?>
							</td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'WordPress Timezone:', 'confab' ); ?></th>
							<td><code><?php echo esc_html( wp_timezone()->getName() ); ?></code></td>
						</tr>
						<tr>
							<th><?php esc_html_e( 'Current UTC Time:', 'confab' ); ?></th>
							<td><code><?php echo esc_html( gmdate( 'Y-m-d H:i:s' ) ); ?></code></td>
						</tr>
					<tr>
						<th><?php esc_html_e( 'Current Server Time:', 'confab' ); ?></th>
						<td><code><?php echo esc_html( gmdate( 'Y-m-d H:i:s' ) ); ?></code></td>
					</tr>
						<tr>
							<th><?php esc_html_e( 'Current WP Time:', 'confab' ); ?></th>
							<td><code><?php echo esc_html( wp_date( 'Y-m-d H:i:s' ) ); ?></code></td>
						</tr>
					</table>
				</div>
			</details>

			<?php if ( 'success' !== $message_type ) : ?>
			<div class="confab-migration-notice notice notice-warning">
				<p><strong><?php esc_html_e( 'Important: Backup Your Database First!', 'confab' ); ?></strong></p>
				<p><?php esc_html_e( 'This tool will modify all session times in your database. Please create a full database backup before proceeding. If something goes wrong, you can restore from the backup.', 'confab' ); ?></p>
			</div>
			<?php endif; ?>

			<?php if ( 1 === $step ) : ?>
				<!-- Step 1: Select Date -->
				<div class="confab-migration-step">
					<h2><?php esc_html_e( 'Step 1: Select a Conference Date', 'confab' ); ?></h2>
					<p><?php esc_html_e( 'Choose a date that has sessions scheduled. We\'ll show you the first two sessions from that date so you can verify the times.', 'confab' ); ?></p>
					
					<form method="post" action="<?php echo esc_url( add_query_arg( 'step', '2' ) ); ?>">
						<?php wp_nonce_field( 'confab_time_migration_action', 'confab_time_migration_nonce' ); ?>
						
						<table class="form-table">
							<tr>
								<th scope="row">
									<label for="sample_date"><?php esc_html_e( 'Conference Date', 'confab' ); ?></label>
								</th>
								<td>
									<input type="date" 
										   name="sample_date" 
										   id="sample_date" 
										   value="<?php echo esc_attr( $sample_date ); ?>"
										   required>
									<p class="description">
										<?php esc_html_e( 'Select a date that has sessions. Check your live site to find a date with scheduled sessions.', 'confab' ); ?>
									</p>
								</td>
							</tr>
						</table>
						
						<?php submit_button( __( 'Show Sample Sessions', 'confab' ) ); ?>
					</form>
				</div>

			<?php elseif ( 2 === $step ) : ?>
				<!-- Step 2: Review and Correct Times -->
				<div class="confab-migration-step">
					<h2><?php esc_html_e( 'Step 2: Review and Correct Session Times', 'confab' ); ?></h2>
					<p><?php esc_html_e( 'Below are the first two sessions from your selected date. Compare the "Current Display Time" with what you see on your live site, then enter the correct time that should be displayed.', 'confab' ); ?></p>
					
					<?php
					$sample_sessions = confab_get_sample_sessions( $sample_date );
					if ( empty( $sample_sessions ) ) :
					?>
						<div class="notice notice-error">
							<p><?php esc_html_e( 'No sessions found for the selected date. Please go back and choose a different date.', 'confab' ); ?></p>
						</div>
						<p><a href="<?php echo esc_url( remove_query_arg( 'step' ) ); ?>" class="button"><?php esc_html_e( 'Go Back', 'confab' ); ?></a></p>
					<?php else : ?>
						<form method="post" action="<?php echo esc_url( add_query_arg( 'step', '3' ) ); ?>">
							<?php wp_nonce_field( 'confab_time_migration_action', 'confab_time_migration_nonce' ); ?>
							<input type="hidden" name="sample_date" value="<?php echo esc_attr( $sample_date ); ?>">
							
							<table class="wp-list-table widefat fixed striped">
								<thead>
									<tr>
										<th><?php esc_html_e( 'Session', 'confab' ); ?></th>
										<th><?php esc_html_e( 'Current Stored Time', 'confab' ); ?></th>
										<th><?php esc_html_e( 'Correct Time (as shown on live site)', 'confab' ); ?></th>
									</tr>
								</thead>
								<tbody>
									<?php foreach ( $sample_sessions as $index => $session ) : ?>
										<tr>
													<td>
														<strong><?php echo esc_html( $session['title'] ); ?></strong>
														<input type="hidden" name="session_<?php echo esc_attr( $index ); ?>_current_timestamp" value="<?php echo esc_attr( $session['timestamp'] ); ?>">
														<input type="hidden" name="session_<?php echo esc_attr( $index ); ?>_title" value="<?php echo esc_attr( $session['title'] ); ?>">
													</td>
													<td>
														<?php
														// Use wp_date to show time in WordPress timezone (matches frontend display).
														$current_time = wp_date( 'Y-m-d H:i:s', $session['timestamp'] );
														echo esc_html( $current_time );
														?>
														<br><small><?php esc_html_e( '(UTC timestamp: ', 'confab' ); ?><?php echo esc_html( $session['timestamp'] ); ?>)</small>
													</td>
											<td>
												<input type="datetime-local" 
													   name="session_<?php echo esc_attr( $index ); ?>_correct_time" 
													   required
													   step="60">
												<p class="description">
													<?php esc_html_e( 'Enter the correct date and time exactly as it should appear', 'confab' ); ?>
												</p>
											</td>
										</tr>
									<?php endforeach; ?>
								</tbody>
							</table>
							
							<p class="description" style="margin-top: 15px;">
								<?php esc_html_e( 'Note: We only need the start time for calculation. The same offset will be applied to both start and end times of all sessions.', 'confab' ); ?>
							</p>
							
							<?php submit_button( __( 'Calculate Offset & Continue', 'confab' ) ); ?>
						</form>
					<?php endif; ?>
				</div>

			<?php elseif ( 3 === $step ) : ?>
				<!-- Step 3: Confirm and Apply -->
				<div class="confab-migration-step">
					<h2><?php esc_html_e( 'Step 3: Confirm Time Offset', 'confab' ); ?></h2>
					
					<?php
					// Only calculate offset if we have POST data (coming from Step 2)
					// After successful migration, just show a completion message
					if ( isset( $_POST['confab_time_migration_nonce'] ) ) {
						$offset_data = confab_calculate_time_offset();
					} else {
						$offset_data = new WP_Error( 'no_post_data', __( 'Migration complete. Use the form below to start a new migration if needed.', 'confab' ) );
					}
					
					if ( is_wp_error( $offset_data ) ) :
					?>
						<p><?php echo esc_html( $offset_data->get_error_message() ); ?></p>
						<p><a href="<?php echo esc_url( remove_query_arg( 'step' ) ); ?>" class="button"><?php esc_html_e( 'Start New Migration', 'confab' ); ?></a></p>
					<?php else : ?>
						<div class="confab-offset-summary">
							<h3><?php esc_html_e( 'Calculated Offset', 'confab' ); ?></h3>
							<table class="widefat">
								<tr>
									<th><?php esc_html_e( 'Offset in Hours', 'confab' ); ?></th>
									<td><strong><?php echo esc_html( number_format( $offset_data['hours'], 2 ) ); ?></strong> <?php esc_html_e( 'hours', 'confab' ); ?></td>
								</tr>
								<tr>
									<th><?php esc_html_e( 'Offset in Seconds', 'confab' ); ?></th>
									<td><strong><?php echo esc_html( number_format( $offset_data['seconds'] ) ); ?></strong> <?php esc_html_e( 'seconds', 'confab' ); ?></td>
								</tr>
								<tr>
									<th><?php esc_html_e( 'Direction', 'confab' ); ?></th>
									<td>
										<?php if ( $offset_data['seconds'] > 0 ) : ?>
											<strong style="color: green;"><?php esc_html_e( 'FORWARD (+)', 'confab' ); ?></strong>
											<p class="description"><?php esc_html_e( 'Session times will be moved later in time', 'confab' ); ?></p>
										<?php else : ?>
											<strong style="color: orange;"><?php esc_html_e( 'BACKWARD (-)', 'confab' ); ?></strong>
											<p class="description"><?php esc_html_e( 'Session times will be moved earlier in time', 'confab' ); ?></p>
										<?php endif; ?>
									</td>
								</tr>
							</table>
							
							<h3 style="margin-top: 20px;"><?php esc_html_e( 'Sample Calculation', 'confab' ); ?></h3>
							<table class="widefat">
								<thead>
									<tr>
										<th><?php esc_html_e( 'Session', 'confab' ); ?></th>
										<th><?php esc_html_e( 'Current Time', 'confab' ); ?></th>
										<th><?php esc_html_e( 'Will Become', 'confab' ); ?></th>
									</tr>
								</thead>
								<tbody>
									<?php foreach ( $offset_data['samples'] as $sample ) : ?>
										<tr>
											<td><?php echo esc_html( $sample['title'] ); ?></td>
											<td><?php echo esc_html( wp_date( 'Y-m-d H:i:s', $sample['current'] ) ); ?></td>
											<td><strong><?php echo esc_html( wp_date( 'Y-m-d H:i:s', $sample['new'] ) ); ?></strong></td>
										</tr>
									<?php endforeach; ?>
								</tbody>
							</table>
						</div>
						
						<div class="confab-migration-confirm" style="margin-top: 30px; padding: 20px; background: #fff; border: 2px solid #dc3232;">
							<h3 style="color: #dc3232;"><?php esc_html_e( 'Final Confirmation Required', 'confab' ); ?></h3>
							<p><strong><?php esc_html_e( 'This action cannot be undone!', 'confab' ); ?></strong></p>
							<p>
								<?php 
								echo esc_html( 
									sprintf(
										/* translators: %s: number of sessions */
										__( 'This will modify the time of ALL %s sessions in your database.', 'confab' ),
										number_format( $offset_data['total_sessions'] )
									) 
								); 
								?>
							</p>
							
							<form method="post">
								<?php wp_nonce_field( 'confab_time_migration_action', 'confab_time_migration_nonce' ); ?>
								<input type="hidden" name="offset_seconds" value="<?php echo esc_attr( $offset_data['seconds'] ); ?>">
								
								<p>
									<label>
										<input type="checkbox" name="confirm_backup" required>
										<?php esc_html_e( 'I have created a database backup and understand this action cannot be undone.', 'confab' ); ?>
									</label>
								</p>
								
								<?php submit_button( __( 'Apply Time Offset to All Sessions', 'confab' ), 'primary', 'confab_apply_migration' ); ?>
							</form>
						</div>
					<?php endif; ?>
				</div>
			<?php endif; ?>
			
			<!-- Migration History -->
			<?php confab_render_migration_logs(); ?>
			
		</div>
	</div>
	<?php
}

/**
 * Get sample sessions for a specific date
 *
 * @param string $date Date in Y-m-d format.
 * @return array Array of session data.
 */
function confab_get_sample_sessions( $date ) {
	$sessions = array();
	
	// Convert date to timestamp range (using UTC for query consistency).
	$timezone = wp_timezone();
	$start_of_day = new DateTime( $date . ' 00:00:00', $timezone );
	$end_of_day = new DateTime( $date . ' 23:59:59', $timezone );
	
	// Get start of day in UTC timestamp for meta query.
	$start_timestamp = $start_of_day->getTimestamp();
	$end_timestamp = $end_of_day->getTimestamp();
	
	$args = array(
		'post_type'      => 'confab_session',
		'posts_per_page' => 2,
		'orderby'        => 'meta_value_num',
		'meta_key'       => '_confab_session_time',
		'order'          => 'ASC',
		'meta_query'     => array(
			array(
				'key'     => '_confab_session_time',
				'value'   => array( $start_timestamp, $end_timestamp ),
				'compare' => 'BETWEEN',
				'type'    => 'NUMERIC',
			),
		),
	);
	
	$query = new WP_Query( $args );
	
	if ( $query->have_posts() ) {
		while ( $query->have_posts() ) {
			$query->the_post();
			$session_id = get_the_ID();
			$timestamp = get_post_meta( $session_id, '_confab_session_time', true );
			
			$sessions[] = array(
				'id'        => $session_id,
				'title'     => get_the_title(),
				'timestamp' => intval( $timestamp ),
			);
		}
	}
	wp_reset_postdata();
	
	return $sessions;
}

/**
 * Calculate time offset from user input
 *
 * @return array|WP_Error Offset data or error.
 */
function confab_calculate_time_offset() {
	if ( ! isset( $_POST['confab_time_migration_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_time_migration_nonce'] ) ), 'confab_time_migration_action' ) ) {
		return new WP_Error( 'invalid_nonce', __( 'Security check failed.', 'confab' ) );
	}
	
	$samples = array();
	$offsets = array();
	
	// Process each sample session.
	for ( $i = 0; $i < 2; $i++ ) {
		$current_key = 'session_' . $i . '_current_timestamp';
		$correct_key = 'session_' . $i . '_correct_time';
		
		if ( ! isset( $_POST[ $current_key ] ) || ! isset( $_POST[ $correct_key ] ) ) {
			continue;
		}
		
		$current_timestamp = intval( sanitize_text_field( wp_unslash( $_POST[ $current_key ] ) ) );
		$correct_time_string = sanitize_text_field( wp_unslash( $_POST[ $correct_key ] ) );
		
		// Parse correct time (user enters in WordPress timezone).
		$correct_datetime = DateTime::createFromFormat( 'Y-m-d\TH:i', $correct_time_string, wp_timezone() );
		if ( ! $correct_datetime ) {
			return new WP_Error( 
				'invalid_time_format', 
				/* translators: %d: session number */
				sprintf( __( 'Invalid time format for session %d. Please use the date picker.', 'confab' ), $i + 1 ) 
			);
		}
		
		// Convert correct time to UTC timestamp.
		$correct_timestamp = $correct_datetime->getTimestamp();
		
		// Calculate offset needed.
		$offset = $correct_timestamp - $current_timestamp;
		$offsets[] = $offset;
		
		$samples[] = array(
			'title'   => isset( $_POST[ 'session_' . $i . '_title' ] ) ? sanitize_text_field( wp_unslash( $_POST[ 'session_' . $i . '_title' ] ) ) : sprintf( /* translators: %d: session number */ __( 'Session %d', 'confab' ), $i + 1 ),
			'current' => $current_timestamp,
			'new'     => $correct_timestamp,
		);
	}
	
	if ( empty( $offsets ) ) {
		return new WP_Error( 'no_samples', __( 'No valid session samples provided.', 'confab' ) );
	}
	
	// Average the offsets from both samples (they should be the same).
	$average_offset = array_sum( $offsets ) / count( $offsets );
	
	// Count total sessions.
	$total_sessions = wp_count_posts( 'confab_session' )->publish;
	
	return array(
		'seconds'        => intval( $average_offset ),
		'hours'          => $average_offset / 3600,
		'samples'        => $samples,
		'total_sessions' => $total_sessions,
	);
}

/**
 * Apply time offset to all sessions
 *
 * @return int|WP_Error Number of updated sessions or error.
 */
function confab_apply_time_migration() {
	if ( ! isset( $_POST['confab_time_migration_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_time_migration_nonce'] ) ), 'confab_time_migration_action' ) ) {
		return new WP_Error( 'invalid_nonce', __( 'Security check failed.', 'confab' ) );
	}
	
	if ( ! isset( $_POST['offset_seconds'] ) ) {
		return new WP_Error( 'missing_offset', __( 'No offset provided.', 'confab' ) );
	}
	
	$offset = intval( sanitize_text_field( wp_unslash( $_POST['offset_seconds'] ) ) );
	
	// Get all sessions.
	$args = array(
		'post_type'      => 'confab_session',
		'posts_per_page' => -1,
		'post_status'    => 'any',
		'fields'         => 'ids',
	);
	
	$query = new WP_Query( $args );
	$updated = 0;
	
	if ( $query->have_posts() ) {
		foreach ( $query->posts as $session_id ) {
			// Update start time.
			$start_time = get_post_meta( $session_id, '_confab_session_time', true );
			if ( $start_time ) {
				$new_start = intval( $start_time ) + $offset;
				update_post_meta( $session_id, '_confab_session_time', $new_start );
			}
			
			// Update end time.
			$end_time = get_post_meta( $session_id, '_confab_session_end_time', true );
			if ( $end_time ) {
				$new_end = intval( $end_time ) + $offset;
				update_post_meta( $session_id, '_confab_session_end_time', $new_end );
			}
			
			$updated++;
		}
	}
	wp_reset_postdata();
	
	// Log the migration.
	confab_log_time_migration( $offset, $updated );
	
	return $updated;
}

/**
 * Log time migration for debugging
 *
 * @param int $offset      Offset in seconds.
 * @param int $sessions_count Number of sessions updated.
 */
function confab_log_time_migration( $offset, $sessions_count ) {
	$log_entry = array(
		'timestamp' => current_time( 'mysql' ),
		'offset'    => $offset,
		'sessions'  => $sessions_count,
		'user'      => get_current_user_id(),
	);
	
	$logs = get_option( 'confab_time_migration_logs', array() );
	$logs[] = $log_entry;
	update_option( 'confab_time_migration_logs', $logs, false );
}

/**
 * Render migration logs inline on the page
 */
function confab_render_migration_logs() {
	$logs = get_option( 'confab_time_migration_logs', array() );
	if ( empty( $logs ) ) {
		return;
	}
	?>
	<div class="confab-migration-logs">
		<h3><?php esc_html_e( 'Migration History', 'confab' ); ?></h3>
		<p class="description"><?php esc_html_e( 'Previous time migrations performed on this site:', 'confab' ); ?></p>
		<table class="widefat">
			<thead>
				<tr>
					<th><?php esc_html_e( 'Date', 'confab' ); ?></th>
					<th><?php esc_html_e( 'Offset', 'confab' ); ?></th>
					<th><?php esc_html_e( 'Sessions', 'confab' ); ?></th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( array_reverse( $logs ) as $log ) : ?>
					<tr>
						<td><?php echo esc_html( $log['timestamp'] ); ?></td>
						<td>
							<?php 
							$hours = $log['offset'] / 3600;
							if ( $log['offset'] > 0 ) {
								echo '+' . esc_html( number_format( $hours, 2 ) ) . ' ' . esc_html__( 'hours', 'confab' );
							} else {
								echo esc_html( number_format( $hours, 2 ) ) . ' ' . esc_html__( 'hours', 'confab' );
							}
							?>
						</td>
						<td><?php echo esc_html( number_format( $log['sessions'] ) ); ?></td>
					</tr>
				<?php endforeach; ?>
			</tbody>
		</table>
		
		<form method="post" class="confab-migration-logs-form">
			<?php wp_nonce_field( 'confab_clear_logs_action', 'confab_clear_logs_nonce' ); ?>
			<?php submit_button( __( 'Clear Migration History', 'confab' ), 'secondary', 'confab_clear_logs' ); ?>
		</form>
	</div>
	<?php
}

/**
 * Handle clear logs action
 */
function confab_handle_clear_logs() {
	if ( isset( $_POST['confab_clear_logs'] ) && isset( $_POST['confab_clear_logs_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_clear_logs_nonce'] ) ), 'confab_clear_logs_action' ) ) {
		delete_option( 'confab_time_migration_logs' );
		add_action( 'admin_notices', function() {
			?>
			<div class="notice notice-success is-dismissible">
				<p><?php esc_html_e( 'Migration history cleared.', 'confab' ); ?></p>
			</div>
			<?php
		});
	}
}
add_action( 'admin_init', 'confab_handle_clear_logs' );
