<?php

/**
 * Custom Taxonomy Registration
 * FILE: inc/taxonomies.php
 *
 * Registers confab_track and confab_location taxonomies for conference session organization.
 * Provides hierarchical categorization for session tracks and venue locations.
 *
 * @package confab
 * @subpackage Core\Taxonomies
 * @since 2.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Initialize custom taxonomy registration.
 *
 * @since 2.0.0
 */
function confab_init_taxonomies() {
	add_action( 'init', 'confab_register_taxonomies' );
}

// Initialize taxonomy functionality
confab_init_taxonomies();

/**
 * Register custom taxonomies for conference session organization.
 *
 * Creates confab_track and confab_location taxonomies with full WordPress integration
 * including REST API support and hierarchical structure for session categorization.
 *
 * @since 2.0.0
 */
function confab_register_taxonomies() {

	// Register session tracks taxonomy
	confab_register_tracks_taxonomy();
	
	// Register session locations taxonomy
	confab_register_locations_taxonomy();
}

/**
 * Register confab_track taxonomy for session track categorization.
 *
 * Creates hierarchical taxonomy for organizing sessions by conference tracks
 * such as "WordPress Development", "Design", "Business", etc.
 *
 * @since 2.0.0
 */
function confab_register_tracks_taxonomy() {
	
	// Track taxonomy labels for admin interface
	$track_labels = array(
		'name'                       => _x( 'Tracks', 'taxonomy general name', 'confab' ),
		'singular_name'              => _x( 'Track', 'taxonomy singular name', 'confab' ),
		'menu_name'                  => __( 'Tracks', 'confab' ),
		'all_items'                  => __( 'All Tracks', 'confab' ),
		'parent_item'                => __( 'Parent Track', 'confab' ),
		'parent_item_colon'          => __( 'Parent Track:', 'confab' ),
		'new_item_name'              => __( 'New Track Name', 'confab' ),
		'add_new_item'               => __( 'Add New Track', 'confab' ),
		'edit_item'                  => __( 'Edit Track', 'confab' ),
		'update_item'                => __( 'Update Track', 'confab' ),
		'view_item'                  => __( 'View Track', 'confab' ),
		'separate_items_with_commas' => __( 'Separate tracks with commas', 'confab' ),
		'add_or_remove_items'        => __( 'Add or remove tracks', 'confab' ),
		'choose_from_most_used'      => __( 'Choose from the most used tracks', 'confab' ),
		'popular_items'              => __( 'Popular Tracks', 'confab' ),
		'search_items'               => __( 'Search Tracks', 'confab' ),
		'not_found'                  => __( 'No tracks found.', 'confab' ),
		'no_terms'                   => __( 'No tracks', 'confab' ),
		'items_list'                 => __( 'Tracks list', 'confab' ),
		'items_list_navigation'      => __( 'Tracks list navigation', 'confab' ),
	);

	// Register tracks taxonomy with full WordPress integration
register_taxonomy(
	'confab_track',
	'confab_session',
	array(
		'labels'                => $track_labels,
		'description'           => __( 'Conference session tracks for organizing sessions by topic or theme.', 'confab' ),
		'hierarchical'          => true,
		'public'                => true,
		'publicly_queryable'    => true,
		'show_ui'               => true,
		'show_in_menu'          => true,
		'show_in_nav_menus'     => true,
		'show_in_rest'          => true,
		'rest_base'             => 'confab_track',           
		'rest_controller_class' => 'WP_REST_Terms_Controller',
		'show_tagcloud'         => true,
		'show_in_quick_edit'    => true,
		'show_admin_column'     => true,
		'query_var'             => 'track',
		'rewrite'               => array(
			'slug'                  => 'track',
			'with_front'            => true,
			'hierarchical'          => true,
		),
		'sort'                  => false,
	)
);
}

/**
 * Register confab_location taxonomy for session venue organization.
 *
 * Creates hierarchical taxonomy for organizing sessions by conference locations
 * such as "Main Auditorium", "Room A", "Workshop Space", etc.
 *
 * @since 2.0.0
 */
function confab_register_locations_taxonomy() {
	
	// Location taxonomy labels for admin interface
	$location_labels = array(
		'name'                       => _x( 'Locations', 'taxonomy general name', 'confab' ),
		'singular_name'              => _x( 'Location', 'taxonomy singular name', 'confab' ),
		'menu_name'                  => __( 'Locations', 'confab' ),
		'all_items'                  => __( 'All Locations', 'confab' ),
		'parent_item'                => __( 'Parent Location', 'confab' ),
		'parent_item_colon'          => __( 'Parent Location:', 'confab' ),
		'new_item_name'              => __( 'New Location Name', 'confab' ),
		'add_new_item'               => __( 'Add New Location', 'confab' ),
		'edit_item'                  => __( 'Edit Location', 'confab' ),
		'update_item'                => __( 'Update Location', 'confab' ),
		'view_item'                  => __( 'View Location', 'confab' ),
		'separate_items_with_commas' => __( 'Separate locations with commas', 'confab' ),
		'add_or_remove_items'        => __( 'Add or remove locations', 'confab' ),
		'choose_from_most_used'      => __( 'Choose from the most used locations', 'confab' ),
		'popular_items'              => __( 'Popular Locations', 'confab' ),
		'search_items'               => __( 'Search Locations', 'confab' ),
		'not_found'                  => __( 'No locations found.', 'confab' ),
		'no_terms'                   => __( 'No locations', 'confab' ),
		'items_list'                 => __( 'Locations list', 'confab' ),
		'items_list_navigation'      => __( 'Locations list navigation', 'confab' ),
	);

	// Register locations taxonomy with full WordPress integration
	register_taxonomy(
		'confab_location',
		'confab_session',
		array(
			'labels'                => $location_labels,
			'description'           => __( 'Conference session locations for organizing sessions by venue or room.', 'confab' ),
			'hierarchical'          => true,
			'public'                => true,
			'publicly_queryable'    => true,
			'show_ui'               => true,
			'show_in_menu'          => true,
			'show_in_nav_menus'     => true,
			'show_in_rest'          => true,
			'rest_base'             => 'session_locations',
			'show_tagcloud'         => true,
			'show_in_quick_edit'    => true,
			'show_admin_column'     => true,
			'query_var'             => 'location',
			'rewrite'               => array(
				'slug'                  => 'location',
				'with_front'            => true,
				'hierarchical'          => true,
			),
			'sort'                  => false,
		)
	);
}