<?php

/**
 * Admin Settings Interface
 * FILE: inc/settings.php
 *
 * WordPress admin settings page for ConFab plugin configuration with
 * hero banner, card-based layout, toggle switches, and dual-confirmation
 * data management system.
 *
 * @package confab
 * @subpackage Admin\Settings
 * @since 2.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Initialize WordPress admin settings interface with security validation and asset loading.
 *
 * Registers WordPress admin hooks for settings initialization, menu creation,
 * conditional asset loading, and admin notice system with proper capability checks 
 * and security validation.
 *
 * @since 2.0.0
 */
function confab_init_settings() {
	add_action( 'admin_init', 'confab_settings_init' );
	add_action( 'admin_menu', 'confab_options_page' );
	add_action( 'admin_enqueue_scripts', 'confab_enqueue_admin_styles' );
	add_action( 'admin_init', 'confab_handle_review_actions' );
	add_action( 'admin_init', 'confab_handle_donation_dismissal' );
	add_action( 'admin_notices', 'confab_display_review_notification' );
}

// Initialize settings functionality
confab_init_settings();


/**
 * Enqueue admin styles and scripts with optimized loading strategy.
 *
 * Loads full ConFab admin CSS only on settings page for performance, while
 * loading minimal notice CSS globally when review notifications should appear.
 * Prevents unnecessary CSS loading on admin pages that don't need it.
 *
 * @param string $hook_suffix Current admin page hook suffix for conditional loading
 * @since 2.0.0
 */
function confab_enqueue_admin_styles( $hook_suffix ) {
	$version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : CONFAB_VERSION;

	// Full admin interface CSS and JS - Settings page only
	if ( 'appearance_page_confab-settings' === $hook_suffix ) {
		wp_enqueue_style(
			'confab-admin-modern',
			CONFAB_URL . 'assets/css/confab-admin-modern.css',
			array(),
			$version
		);
		
		wp_enqueue_script(
			'confab-admin-settings',
			CONFAB_URL . 'assets/js/confab-admin-settings.js',
			array( 'jquery' ),
			$version,
			true
		);
		return;
	}

	// Minimal admin notice CSS - All other admin pages when review should show
	if ( is_admin() && confab_should_show_review_request() ) {
		wp_enqueue_style(
			'confab-admin-notices',
			CONFAB_URL . 'assets/css/confab-admin-notices.css',
			array(),
			$version
		);
	}
}

/**
 * Determine if review request should be displayed based on usage and time thresholds.
 *
 * Implements dual threshold logic where either sufficient usage OR sufficient time
 * since installation triggers review request, respecting user dismissal preferences.
 *
 * @return bool Whether review request should be shown
 * @since 2.0.0
 */
function confab_should_show_review_request() {
    $user_id = get_current_user_id();
    $review_dismissed = get_user_meta( $user_id, 'confab_review_dismissed', true );
    
    // Permanent dismissal
    if ( 'permanent' === $review_dismissed ) {
        return false;
    }
    
    // Temporary dismissal (7 day cooldown)
    if ( 'temporary' === $review_dismissed ) {
        $dismissed_time = get_user_meta( $user_id, 'confab_review_dismissed_time', true );
        if ( $dismissed_time && ( time() - $dismissed_time < 7 * DAY_IN_SECONDS ) ) {
            return false;
        }
    }
    
    // Check thresholds: 50 sessions created OR 100 schedule displays OR 14 days
    $install_date = get_option( 'confab_install_date', 0 );
    $stats = get_option( 'confab_usage_stats', array(
        'sessions_created' => 0,
        'schedules_displayed' => 0,
    ) );
    
    $days_since_install = $install_date ? ( time() - $install_date ) / DAY_IN_SECONDS : 0;
    
    return ( $stats['sessions_created'] >= 50 ) || 
           ( $stats['schedules_displayed'] >= 100 ) || 
           ( $days_since_install >= 14 );
}

/**
 * Render review request card with ConFab branding and usage statistics display.
 *
 * Creates review request interface matching existing card design with four-button
 * action system and professional ConFab styling integration.
 *
 * @since 2.0.0
 */
function confab_render_review_request_card() {
    if ( ! confab_should_show_review_request() ) {
        return;
    }
    
    $stats = get_option( 'confab_usage_stats', array(
        'sessions_created' => 0,
        'schedules_displayed' => 0,
    ) );
    
    $sessions_count = absint( $stats['sessions_created'] );
    $displays_count = absint( $stats['schedules_displayed'] );
    
    ?>
    <div class="confab-card confab-card-review">
        <h2>
            <span class="dashicons dashicons-star-filled confab-icon-review"></span>
            <?php esc_html_e( 'Love ConFab?', 'confab' ); ?>
        </h2>
        <p><?php 
        if ( $sessions_count > 0 && $displays_count > 0 ) {
			printf(
				/* translators: 1: number of sessions created, 2: number of schedules displayed */
				esc_html__( 'ConFab has helped you create %1$s conference sessions and displayed %2$s schedules! If you find it helpful, would you mind rating it?', 'confab' ),
				'<strong>' . number_format( $sessions_count ) . '</strong>',
				'<strong>' . number_format( $displays_count ) . '</strong>'
			);
		} elseif ( $sessions_count > 0 ) {
			printf(
				/* translators: %s: number of sessions created */
				esc_html__( 'ConFab has helped you create %s conference sessions! If you find it helpful, would you mind rating it?', 'confab' ),
				'<strong>' . number_format( $sessions_count ) . '</strong>'
			);
		} elseif ( $displays_count > 0 ) {
			printf(
				/* translators: %s: number of schedules displayed */
				esc_html__( 'ConFab has displayed %s conference schedules! If you find it helpful, would you mind rating it?', 'confab' ),
				'<strong>' . number_format( $displays_count ) . '</strong>'
			);

        } else {
            esc_html_e( 'You\'ve been using ConFab for conference scheduling! If you find it helpful, would you mind rating it?', 'confab' );
        }
        ?></p>
        
        <div class="confab-review-actions">
            <form method="post" class="confab-inline-form">
                <?php wp_nonce_field( 'confab_review_action', 'confab_review_nonce' ); ?>
                <button type="submit" name="confab_review_action" value="rate_now" 
                        class="button button-primary confab-review-primary">
                    <?php esc_html_e( '★★★★★ Rate on WordPress.org', 'confab' ); ?>
                </button>
            </form>
            
            <form method="post" class="confab-inline-form">
                <?php wp_nonce_field( 'confab_review_action', 'confab_review_nonce' ); ?>
                <button type="submit" name="confab_review_action" value="already_rated" 
                        class="button confab-review-secondary">
                    <?php esc_html_e( 'Already Rated', 'confab' ); ?>
                </button>
            </form>
            
            <form method="post" class="confab-inline-form">
                <?php wp_nonce_field( 'confab_review_action', 'confab_review_nonce' ); ?>
                <button type="submit" name="confab_review_action" value="ask_later" 
                        class="button confab-review-secondary">
                    <?php esc_html_e( 'Ask Me Later', 'confab' ); ?>
                </button>
            </form>
            
            <form method="post" class="confab-inline-form">
                <?php wp_nonce_field( 'confab_review_action', 'confab_review_nonce' ); ?>
                <button type="submit" name="confab_review_action" value="dont_ask" 
                        class="button confab-review-secondary">
                    <?php esc_html_e( 'Don\'t Ask Again', 'confab' ); ?>
                </button>
            </form>
        </div>
    </div>
    <?php
}

/**
 * Handle review action form submissions with WordPress security validation.
 *
 * Processes user review actions including rating redirect, dismissal preferences,
 * and positive action tracking for donation system integration.
 *
 * @since 2.0.0
 */
function confab_handle_review_actions() {
    if ( ! isset( $_POST['confab_review_action'] ) ) {
        return;
    }
    
    if ( ! isset( $_POST['confab_review_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_review_nonce'] ) ), 'confab_review_action' ) ) {
        return;
    }
    
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    $user_id = get_current_user_id();
    $action = sanitize_text_field( wp_unslash( $_POST['confab_review_action'] ) );
    
    switch ( $action ) {
        case 'rate_now':
            update_user_meta( $user_id, 'confab_positive_review_action', true );
            update_user_meta( $user_id, 'confab_review_dismissed', 'permanent' );
            wp_safe_redirect( 'https://wordpress.org/support/plugin/confab/reviews/#new-post' );
            exit;
            
        case 'already_rated':
            update_user_meta( $user_id, 'confab_positive_review_action', true );
            update_user_meta( $user_id, 'confab_review_dismissed', 'permanent' );
            break;
            
        case 'ask_later':
            update_user_meta( $user_id, 'confab_review_dismissed', 'temporary' );
            update_user_meta( $user_id, 'confab_review_dismissed_time', time() );
            break;
            
        case 'dont_ask':
            update_user_meta( $user_id, 'confab_review_dismissed', 'permanent' );
            break;
    }
    
    wp_safe_redirect( admin_url( 'themes.php?page=confab-settings' ) );
    exit;
}

/**
 * Determine if donation request should be displayed based on positive review actions.
 *
 * Shows donation request only after user has taken positive review action,
 * ensuring we've earned the right to ask for support through demonstrated value.
 *
 * @return bool Whether donation request should be shown
 * @since 2.0.0
 */
function confab_should_show_donation_request() {
    $user_id = get_current_user_id();
    $donation_shown = get_user_meta( $user_id, 'confab_donation_shown', true );
    $review_action = get_user_meta( $user_id, 'confab_positive_review_action', true );
    
    // Only show donation after review is dismissed AND user took positive action
    $review_should_show = confab_should_show_review_request();
    
    return $review_action && ! $donation_shown && ! $review_should_show;
}

/**
 * Render donation request card with ConFab branding and ethical messaging.
 *
 * Displays donation request only after positive review action with professional
 * messaging and easy dismissal options aligned with WordPress community values.
 *
 * @since 2.0.0
 */
function confab_render_donation_request_card() {
    if ( ! confab_should_show_donation_request() ) {
        return;
    }
    
    ?>
    <div class="confab-card confab-card-donation">
        <h2>
            <span class="dashicons dashicons-heart confab-icon-donation"></span>
            <?php esc_html_e( 'Support Development', 'confab' ); ?>
        </h2>
        <p><?php esc_html_e( 'Thanks for rating ConFab! Development and maintenance take time. If ConFab saves you time, consider supporting continued development.', 'confab' ); ?></p>
        
        <div class="confab-donation-actions">
            <a href="https://shop.low.li/downloads/help-the-legend-sail-on/" target="_blank" 
               class="button button-primary confab-donation-button">
                <?php esc_html_e( '💝 Donate via PayPal', 'confab' ); ?>
            </a>
            
            <form method="post" class="confab-inline-form">
                <?php wp_nonce_field( 'confab_donation_action', 'confab_donation_nonce' ); ?>
                <button type="submit" name="confab_donation_action" value="donated" 
                        class="button confab-donation-secondary">
                    <?php esc_html_e( 'Already Donated', 'confab' ); ?>
                </button>
            </form>
            
            <form method="post" class="confab-inline-form">
                <?php wp_nonce_field( 'confab_donation_action', 'confab_donation_nonce' ); ?>
                <button type="submit" name="confab_donation_action" value="dismiss" 
                        class="button confab-donation-secondary">
                    <?php esc_html_e( 'Maybe Later', 'confab' ); ?>
                </button>
            </form>
        </div>
    </div>
    <?php
}

/**
 * Handle donation dismissal form submissions with WordPress security validation.
 *
 * Processes donation action responses ensuring proper user preference tracking
 * and preventing repeat donation requests after dismissal.
 *
 * @since 2.0.0
 */
function confab_handle_donation_dismissal() {
    if ( ! isset( $_POST['confab_donation_action'] ) ) {
        return;
    }
    
    if ( ! isset( $_POST['confab_donation_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_donation_nonce'] ) ), 'confab_donation_action' ) ) {
        return;
    }
    
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    $user_id = get_current_user_id();
    $action = sanitize_text_field( wp_unslash( $_POST['confab_donation_action'] ) );
    
    if ( 'dismiss' === $action || 'donated' === $action ) {
        update_user_meta( $user_id, 'confab_donation_shown', true );
    }
    
    wp_safe_redirect( admin_url( 'themes.php?page=confab-settings' ) );
    exit;
}

/**
 * Display ConFab review notification in WordPress admin with logo branding.
 *
 * Shows subtle admin notice with ConFab logo and link to settings page when
 * review thresholds are met, excluding the ConFab settings page itself.
 *
 * @since 2.0.0
 */
function confab_display_review_notification() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    if ( ! confab_should_show_review_request() ) {
        return;
    }
    
    // Don't show on ConFab settings page (handled there)
    $screen = get_current_screen();
    if ( $screen && 'appearance_page_confab-settings' === $screen->id ) {
        return;
    }
    
    ?>
    <div class="notice notice-info is-dismissible confab-review-notice">
        <div class="confab-notice-content">
            <img src="<?php echo esc_url( CONFAB_URL . 'assets/images/confab-logo.svg' ); ?>" 
                 alt="<?php esc_attr_e( 'ConFab Logo', 'confab' ); ?>" 
                 class="confab-notice-logo">
            <strong><?php esc_html_e( 'ConFab Notification', 'confab' ); ?></strong>
            <span class="confab-notification-badge">1</span>
            <a href="<?php echo esc_url( admin_url( 'themes.php?page=confab-settings' ) ); ?>" 
               class="button confab-notice-button">
                <?php esc_html_e( 'Go To Settings', 'confab' ); ?>
            </a>
        </div>
    </div>
    <?php
}

/**
 * Register WordPress settings sections and fields with validation callbacks.
 *
 * Creates admin interface sections for plugin configuration with proper sanitization
 * callbacks and security validation for all user inputs.
 *
 * @since 2.0.0
 */
function confab_settings_init() {
	
	// Register settings with sanitization callbacks
	register_setting(
		'confab_settings_group',
		'confab_field_byline',
		array(
			'type'              => 'boolean',
			'description'       => __( 'Display credit link at bottom of schedule', 'confab' ),
			'sanitize_callback' => 'confab_sanitize_byline_setting',
			'default'           => true,
		)
	);

	register_setting(
		'confab_settings_group',
		'confab_field_schedule_page_url',
		array(
			'type'              => 'string',
			'description'       => __( 'URL of page containing conference schedule', 'confab' ),
			'sanitize_callback' => 'confab_sanitize_url_setting',
			'default'           => '',
		)
	);

	register_setting(
		'confab_settings_group',
		'confab_excerpt_length',
		array(
			'type'              => 'integer',
			'description'       => __( 'Maximum character length for session excerpts', 'confab' ),
			'sanitize_callback' => 'confab_sanitize_excerpt_length',
			'default'           => 150,
		)
	);

	register_setting(
		'confab_settings_group',
		'confab_delete_initial_confirmed',
		array(
			'type'              => 'boolean',
			'description'       => __( 'Initial confirmation to delete ConFab data on uninstall', 'confab' ),
			'sanitize_callback' => 'confab_sanitize_dual_confirmation',
			'default'           => false,
		)
	);

	register_setting(
		'confab_settings_group',
		'confab_delete_final_confirmed',
		array(
			'type'              => 'boolean',
			'description'       => __( 'Final confirmation to delete ConFab data on uninstall', 'confab' ),
			'sanitize_callback' => 'confab_sanitize_dual_confirmation',
			'default'           => false,
		)
	);

	// Register general settings section
	add_settings_section(
		'confab_section_settings',
		__( 'General Settings', 'confab' ),
		'confab_section_settings_callback',
		'confab_settings'
	);

	// Register byline setting field with proper labeling
	add_settings_field(
		'confab_field_byline',
		__( 'Show Credit Link', 'confab' ),
		'confab_field_byline_callback',
		'confab_settings',
		'confab_section_settings',
		array(
			'label_for' => 'confab_field_byline',
			'class'     => 'confab-setting-row',
		)
	);

	// Register schedule page URL field with validation
	add_settings_field(
		'confab_field_schedule_page_url',
		__( 'Schedule Page URL', 'confab' ),
		'confab_field_schedule_page_url_callback',
		'confab_settings',
		'confab_section_settings',
		array(
			'label_for' => 'confab_field_schedule_page_url',
			'class'     => 'confab-setting-row',
		)
	);

	// Register excerpt length field
	add_settings_field(
		'confab_excerpt_length',
		__( 'Excerpt Length', 'confab' ),
		'confab_field_excerpt_length_callback',
		'confab_settings',
		'confab_section_settings',
		array(
			'label_for' => 'confab_excerpt_length',
			'class'     => 'confab-setting-row',
		)
	);

	add_settings_field(
		'confab_field_delete_data_on_uninstall',
		__( 'Delete Data on Uninstall', 'confab' ),
		'confab_field_delete_data_callback',
		'confab_settings',
		'confab_section_settings',
		array(
			'label_for' => 'confab_field_delete_data_on_uninstall',
			'class'     => 'confab-setting-row confab-setting-danger',
		)
	);
}

/**
 * Sanitize initial confirmation setting with dual-confirmation requirement.
 *
 * Validates initial delete confirmation toggle and triggers final confirmation
 * clearing when user unchecks the initial setting to maintain data safety.
 *
 * @param mixed $input Raw input value from WordPress settings form submission
 * @return bool Validated boolean value for initial confirmation state
 * @since 2.0.0
 */
function confab_sanitize_initial_confirmation( $input ) {
	$initial = (bool) $input;
	
	// If initial is unchecked, ensure final is also false via action hook
	if ( ! $initial ) {
		add_action( 'updated_option_confab_delete_initial_confirmed', 'confab_clear_final_confirmation_on_initial_false' );
	}
	
	return $initial;
}

/**
 * Sanitize final confirmation setting with dual-confirmation validation.
 *
 * Validates final delete confirmation toggle and enforces requirement that
 * initial confirmation must be true before final confirmation can be enabled.
 *
 * @param mixed $input Raw input value from WordPress settings form submission
 * @return bool Validated boolean value for final confirmation state
 * @since 2.0.0
 */
function confab_sanitize_final_confirmation( $input ) {
	$final = (bool) $input;
	$initial = get_option( 'confab_delete_initial_confirmed', false );
	
	// Final can only be true if initial is also true
	if ( $final && ! $initial ) {
		return false;
	}
	
	return $final;
}

/**
 * Sanitize byline setting with boolean validation.
 *
 * Converts input to boolean for credit link display preference while ensuring
 * type safety for WordPress options storage.
 *
 * @param mixed $input Raw input value from WordPress settings form
 * @return bool Validated boolean value for byline display preference
 * @since 2.0.0
 */
function confab_sanitize_byline_setting( $input ) {
    $result = (bool) $input;
    return $result;
}
/**
 * Sanitize schedule page URL with validation and security filtering.
 *
 * Validates and sanitizes URL input for schedule page setting with error handling
 * for invalid URLs and preservation of existing valid values on validation failure.
 *
 * @param mixed $input Raw URL input from WordPress settings form
 * @return string Sanitized and validated URL or empty string on validation failure
 * @since 2.0.0
 */
function confab_sanitize_url_setting( $input ) {
	if ( empty( $input ) ) {
		return '';
	}

	$url = sanitize_text_field( $input );
	$validated_url = esc_url_raw( $url );
	
	if ( ! filter_var( $validated_url, FILTER_VALIDATE_URL ) ) {
		add_settings_error(
			'confab_field_schedule_page_url',
			'invalid_url',
			__( 'Please enter a valid URL (including http:// or https://)', 'confab' ),
			'error'
		);
		
		return get_option( 'confab_field_schedule_page_url', '' );
	}

	return $validated_url;
}

/**
 * Sanitize excerpt length setting with range validation.
 *
 * Validates excerpt length is within acceptable range (50-500 characters)
 * and provides fallback to default value on validation failure.
 *
 * @param mixed $input Raw input value from WordPress settings form
 * @return int Validated integer value within acceptable range
 * @since 2.1.0
 */
function confab_sanitize_excerpt_length( $input ) {    
    if ( empty( $input ) ) {
        return 150;
    }
    
    $length = absint( $input );
    
    if ( $length < 50 || $length > 500 ) {
        add_settings_error(
            'confab_excerpt_length',
            'invalid_length',
            __( 'Excerpt length must be between 50 and 500 characters. Using default value of 150.', 'confab' ),
            'error'
        );
        return 150;
    }
    
    return $length;
}

/**
 * Sanitize dual-confirmation settings with both-or-neither validation logic.
 *
 * Implements data deletion safety requirement where both confirmation toggles
 * must be checked for either to save as true. This prevents accidental data
 * deletion by requiring explicit dual confirmation from the user.
 *
 * @param mixed $input Raw input value from WordPress settings form submission
 * @return bool True only if both confirmations checked, false otherwise
 * @since 2.1.0
 */
function confab_sanitize_dual_confirmation( $input ) {
    // Read both confirmations directly from POST
    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WordPress Settings API handles nonce verification
    $initial = isset( $_POST['confab_delete_initial_confirmed'] ) ? true : false;
    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WordPress Settings API handles nonce verification  
    $final = isset( $_POST['confab_delete_final_confirmed'] ) ? true : false;
    
    // Both-or-neither: both must be checked for either to save as true
    return ( $initial && $final );
}

/**
 * Render general settings section with accessibility markup.
 *
 * Provides section description container for WordPress settings API with
 * proper accessibility structure for screen readers and keyboard navigation.
 *
 * @param array $args Section arguments from WordPress settings API
 * @since 2.0.0
 */
function confab_section_settings_callback( $args ) {
	// Section description is now handled in individual cards
}

/**
 * Render byline toggle field with switch component and accessibility features.
 *
 * Creates credit link display toggle with toggle switch styling,
 * proper ARIA attributes, and user guidance text.
 *
 * @param array $args Field arguments from WordPress settings API
 * @since 2.0.0
 */
function confab_field_byline_callback( $args ) {
	$option_value = get_option( 'confab_field_byline', false );
	?>
	<div class="confab-setting-item">
		<label class="confab-toggle-wrapper" for="<?php echo esc_attr( $args['label_for'] ); ?>">
			<input 
				type="checkbox" 
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="confab_field_byline" 
				class="confab-toggle-input"
				value="1" 
				<?php checked( 1, $option_value ); ?>
				aria-describedby="confab-byline-description"
			/>
			<span class="confab-toggle-slider"></span>
		</label>
		<div class="confab-setting-content">
			<label for="<?php echo esc_attr( $args['label_for'] ); ?>" class="confab-setting-label">
				<?php esc_html_e( 'Display "Powered by ConFab" Link', 'confab' ); ?>
			</label>
			<p id="confab-byline-description" class="confab-setting-description">
				<?php
				esc_html_e(
					'Shows a small credit link at the bottom of your conference schedule to support plugin development.',
					'confab'
				);
				?>
			</p>
		</div>
	</div>
	<?php
}

/**
 * Render schedule page URL field with validation and accessibility features.
 *
 * Creates URL input field for schedule page setting with validation,
 * placeholder text, and user guidance for navigation link functionality.
 *
 * @param array $args Field arguments from WordPress settings API
 * @since 2.0.0
 */
function confab_field_schedule_page_url_callback( $args ) {
	$option_value = get_option( 'confab_field_schedule_page_url', '' );
	?>
	<div class="confab-setting-item" style="display: block;">
		<div class="confab-setting-content">
			<label for="<?php echo esc_attr( $args['label_for'] ); ?>" class="confab-setting-label">
				<?php esc_html_e( 'Schedule Page URL', 'confab' ); ?>
			</label>
			<input 
				type="url" 
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="confab_field_schedule_page_url" 
				value="<?php echo esc_attr( $option_value ); ?>" 
				class="regular-text"
				placeholder="<?php esc_attr_e( 'https://example.com/schedule', 'confab' ); ?>"
				aria-describedby="confab-url-description"
				style="width: 100%; max-width: 400px; margin: 8px 0;"
			/>
			<p id="confab-url-description" class="confab-setting-description">
				<?php
				esc_html_e(
					'The URL of the page where your conference schedule is displayed. Used for "Return to Schedule" links on individual session pages.',
					'confab'
				);
				?>
			</p>
		</div>
	</div>
	<?php
}

/**
 * Render excerpt length field with number input and range validation.
 *
 * Creates number input for excerpt character limit with visual guidance
 * and explanation of paragraph-aware behavior.
 *
 * @param array $args Field arguments from WordPress settings API
 * @since 2.1.0
 */
function confab_field_excerpt_length_callback( $args ) {
	$option_value = get_option( 'confab_excerpt_length', 150 );
	?>
	<div class="confab-setting-item" style="display: block;">
		<div class="confab-setting-content">
			<label for="<?php echo esc_attr( $args['label_for'] ); ?>" class="confab-setting-label">
				<?php esc_html_e( 'Session Excerpt Length (Characters)', 'confab' ); ?>
			</label>
			<input 
				type="text" 
				id="<?php echo esc_attr( $args['label_for'] ); ?>"
				name="confab_excerpt_length" 
				value="<?php echo esc_attr( $option_value ); ?>" 
				class="small-text"
				aria-describedby="confab-excerpt-description"
				style="margin: 8px 0;"
			/>
			<span class="confab-setting-hint"><?php esc_html_e( '(50-500 characters)', 'confab' ); ?></span>
			<p id="confab-excerpt-description" class="confab-setting-description">
				<?php
				esc_html_e(
					'Maximum character length for session descriptions in schedule displays. The system respects paragraph boundaries, showing complete first paragraphs when they fit within this limit.',
					'confab'
				);
				?>
			</p>
		</div>
	</div>
	<?php
}

/**
 * Render data deletion warning field with dual-confirmation interface and danger styling.
 *
 * Creates dangerous action toggle with progressive disclosure interface, detailed
 * explanation of data removal consequences, warning system, and safety-first
 * default behavior with dual-confirmation requirement.
 *
 * @param array $args Field arguments from WordPress settings API registration
 * @since 2.0.0
 */
function confab_field_delete_data_callback( $args ) {
	$initial_confirmed = (bool) get_option( 'confab_delete_initial_confirmed', false );
	$final_confirmed = (bool) get_option( 'confab_delete_final_confirmed', false );
	?>
	
	<?php if ( ! $initial_confirmed ) : ?>
		<!-- Normal state - initial toggle -->
		<div class="confab-setting-item">
			<label class="confab-toggle-wrapper" for="confab_delete_initial_confirmed">
				<input 
					type="checkbox" 
					id="confab_delete_initial_confirmed"
					name="confab_delete_initial_confirmed"
					class="confab-toggle-input"
					value="1"
					aria-describedby="confab-delete-initial-description"
				/>
				<span class="confab-toggle-slider"></span>
			</label>
			<div class="confab-setting-content">
				<label for="confab_delete_initial_confirmed" class="confab-setting-label">
					<?php esc_html_e( 'Delete All Data on Uninstall', 'confab' ); ?>
				</label>
				<p id="confab-delete-initial-description" class="confab-setting-description">
					<?php esc_html_e( 'Configure what happens to your conference data when ConFab is uninstalled.', 'confab' ); ?>
				</p>
			</div>
		</div>
	<?php else : ?>
		<!-- Danger state - red toggle with alert -->
		<div class="confab-setting-item confab-danger-active">
			<label class="confab-toggle-wrapper confab-toggle-danger" for="confab_delete_initial_confirmed">
				<input 
					type="checkbox" 
					id="confab_delete_initial_confirmed"
					name="confab_delete_initial_confirmed"
					class="confab-toggle-input"
					value="1"
					checked
				/>
				<span class="confab-toggle-slider confab-toggle-slider--danger"></span>
			</label>
			<div class="confab-setting-content">
				<label for="confab_delete_initial_confirmed" class="confab-setting-label confab-label-danger">
					<?php esc_html_e( 'Delete All Data on Uninstall', 'confab' ); ?>
				</label>
				<div class="confab-danger-indicator" id="confab-danger-summary">
					<span class="confab-warning-icon">⚠</span>
					<span class="confab-text-danger"><?php esc_html_e( 'All data will be deleted', 'confab' ); ?></span>
				</div>
			</div>
		</div>
	<?php endif; ?>

	<!-- Warning details section - ALWAYS RENDERED, CSS-controlled visibility -->
	<div class="confab-warning-details" id="confab-warning-details">
		<div class="confab-warning-box">
			<p class="confab-setting-description confab-text-danger">
				<strong><?php esc_html_e( 'Warning:', 'confab' ); ?></strong>
				<?php esc_html_e( 'This will permanently delete ALL ConFab data when the plugin is uninstalled, including all sessions, tracks, locations, and settings. This action cannot be undone.', 'confab' ); ?>
			</p>
			
			<div class="confab-danger-details">
				<h4><?php esc_html_e( 'Data that will be permanently deleted:', 'confab' ); ?></h4>
				<ul>
					<li><?php esc_html_e( 'All conference sessions and session content', 'confab' ); ?></li>
					<li><?php esc_html_e( 'All track and location taxonomy terms', 'confab' ); ?></li>
					<li><?php esc_html_e( 'All plugin settings and configuration', 'confab' ); ?></li>
					<li><?php esc_html_e( 'All session metadata (dates, times, speakers)', 'confab' ); ?></li>
				</ul>
			</div>

			<!-- Final Confirmation Toggle -->
			<div class="confab-final-confirmation">
				<label class="confab-toggle-wrapper confab-toggle-danger" for="confab_delete_final_confirmed">
					<input 
						type="checkbox" 
						id="confab_delete_final_confirmed"
						name="confab_delete_final_confirmed"
						class="confab-toggle-input"
						value="1"
						<?php checked( 1, $final_confirmed ); ?>
					/>
					<span class="confab-toggle-slider confab-toggle-slider--danger"></span>
				</label>
				<div class="confab-confirmation-text">
					<label for="confab_delete_final_confirmed" class="confab-setting-label confab-label-danger">
						<?php esc_html_e( 'I accept the consequences of my actions. I will blame nobody else for deleted data.', 'confab' ); ?>
					</label>
				</div>
			</div>
		</div>
	</div>
	<?php
}

/**
 * Register WordPress admin menu page with capability checks.
 *
 * Creates ConFab settings page in WordPress admin with appropriate permissions,
 * menu positioning, and callback function registration.
 *
 * @since 2.0.0
 */
function confab_options_page() {
	add_theme_page(
		__( 'ConFab Settings', 'confab' ),           // Page title
		__( 'ConFab', 'confab' ),                    // Menu title
		'manage_options',                            // Capability
		'confab-settings',                           // Menu slug
		'confab_options_page_html'                   // Callback
	);
}

/**
 * Render admin settings page with hero banner, card layout, and documentation.
 *
 * Displays ConFab settings form with hero branding, card-based layout,
 * how-to documentation, attribution section, and interface design.
 *
 * @since 2.0.0
 */
function confab_options_page_html() {
	// Check user capabilities with WordPress security
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die(
			esc_html__( 'You do not have sufficient permissions to access this page.', 'confab' ),
			esc_html__( 'Permission Denied', 'confab' ),
			array( 'response' => 403 )
		);
	}

	// Display settings update messages
	settings_errors( 'confab_settings_group' );

	?>
	<div class="wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
	</div>
	
	<div class="confab-admin-wrapper">
		
		<!-- Modern Hero Banner -->
			<div class="confab-hero">
				<div class="confab-hero-content">
					<div class="confab-hero-branding">
						<img src="<?php echo esc_url( CONFAB_URL . 'assets/images/confab-logo.svg' ); ?>" 
							 alt="<?php esc_attr_e( 'ConFab Logo', 'confab' ); ?>" 
							 class="confab-hero-logo">
						<div class="confab-hero-text">
							<h1 class="confab-hero-title"><?php esc_html_e( 'ConFab', 'confab' ); ?></h1>
							<p class="confab-hero-subtitle"><?php esc_html_e( 'Professional Conference Scheduling', 'confab' ); ?></p>
							<div class="confab-hero-version">
								<?php
								printf(
									/* translators: %s: plugin version number */
									esc_html__( 'Version %s', 'confab' ),
									esc_html( CONFAB_VERSION )
								);
								?>
							</div>
						</div>
					</div>
				</div>
			</div>

			<?php confab_render_review_request_card(); ?>
			<?php confab_render_donation_request_card(); ?>

			<!-- Settings Form with Modern Layout -->
			<div class="confab-settings-container">
				<form action="options.php" method="post">
					<?php settings_fields( 'confab_settings_group' ); ?>
					
					<!-- Action Card for Save Button -->
					<div class="confab-card confab-settings-action-card">
						<div class="confab-settings-action-content">
							<?php submit_button( __( 'Save Settings', 'confab' ), 'confab-button-primary', 'submit', false ); ?>
						</div>
					</div>
					
					<!-- Settings Grid Layout -->
					<div class="confab-settings-grid">


						
						<!-- Credit Link Card -->
						<div class="confab-card">
							<h2>
								<span class="dashicons dashicons-heart confab-icon"></span>
								<?php esc_html_e( 'Support Development', 'confab' ); ?>
							</h2>
							<?php confab_field_byline_callback( array( 'label_for' => 'confab_field_byline' ) ); ?>
						</div>

						<!-- Schedule Page URL Card -->
						<div class="confab-card">
							<h2>
								<span class="dashicons dashicons-admin-links confab-icon"></span>
								<?php esc_html_e( 'Navigation Settings', 'confab' ); ?>
							</h2>
							<?php confab_field_schedule_page_url_callback( array( 'label_for' => 'confab_field_schedule_page_url' ) ); ?>
						</div>

					<!-- Excerpt Display Card -->
					<div class="confab-card">
						<h2>
							<span class="dashicons dashicons-text-page confab-icon"></span>
							<?php esc_html_e( 'Content Display', 'confab' ); ?>
						</h2>
					<?php confab_field_excerpt_length_callback( array( 'label_for' => 'confab_excerpt_length' ) ); ?>
				</div>

				<!-- Time Migration Card -->
				<div class="confab-card">
					<h2>
						<span class="dashicons dashicons-migrate confab-icon"></span>
						<?php esc_html_e( 'Time Migration', 'confab' ); ?>
					</h2>
					<div class="confab-setting-item">
						<div class="confab-setting-content">
							<p class="confab-setting-description">
								<?php esc_html_e( 'If your session times are displaying incorrectly after migration, use the Time Migration Tool to calculate and apply the correct offset to all sessions.', 'confab' ); ?>
							</p>
							<p style="margin-top: 15px;">
								<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=confab_session&page=confab-time-migration' ) ); ?>" class="button button-secondary">
									<?php esc_html_e( 'Open Time Migration Tool', 'confab' ); ?>
								</a>
							</p>
						</div>
					</div>
				</div>

				<!-- Data Management Card -->
						<div class="confab-card">
							<h2>
								<span class="dashicons dashicons-admin-tools confab-icon"></span>
								<?php esc_html_e( 'Data Management', 'confab' ); ?>
							</h2>
							
							<?php confab_field_delete_data_callback( array( 'label_for' => 'confab_field_delete_data_on_uninstall' ) ); ?>
						</div>
						
						<!-- How to Use Documentation Card -->
						<div class="confab-card" style="grid-column: 1 / -1;">
							<h2>
								<span class="dashicons dashicons-info confab-icon"></span>
								<?php esc_html_e( 'How to Use ConFab', 'confab' ); ?>
							</h2>
							<div class="confab-accordion">
								
								<!-- New Installation Timezone Setup -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="new-install-timezone">
										<?php esc_html_e( 'IMPORTANT: New Installation Setup', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="new-install-timezone">
										<p style="background: #d4edda; border-left: 4px solid #28a745; padding: 15px; margin-bottom: 20px;">
											<strong><?php esc_html_e( '✅ Read This First!', 'confab' ); ?></strong> <?php esc_html_e( 'Before creating ANY sessions, you must set your WordPress timezone correctly. This is the most important step for ensuring your session times display correctly.', 'confab' ); ?>
										</p>

										<h4><?php esc_html_e( 'CRITICAL First Step: Set Your Timezone', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'Before creating your first session, follow these steps:', 'confab' ); ?></p>
										<ol>
											<li>
												<strong><?php esc_html_e( 'Go to Settings > General in WordPress admin', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Find this in your main WordPress admin sidebar (not ConFab settings)', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Scroll to the Timezone setting', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Choose a city in your timezone (e.g., "Los Angeles" for Pacific Time)', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Click Save Changes', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Verify it saved correctly by checking the page again', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'NOW create your first session', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Your session times will now be stored and displayed correctly', 'confab' ); ?>
											</li>
										</ol>

										<h4><?php esc_html_e( 'Why This Matters:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'ConFab uses your WordPress timezone to store and display all session times', 'confab' ); ?></li>
											<li><?php esc_html_e( 'If you change your timezone AFTER creating sessions, all existing sessions will display in the NEW timezone', 'confab' ); ?></li>
											<li><?php esc_html_e( 'For example: A session created at 9:00 AM might display as 4:00 PM if you change timezones later', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Setting it correctly FIRST prevents all timezone-related problems', 'confab' ); ?></li>
										</ul>

										<h4><?php esc_html_e( 'What If I Already Created Sessions?', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'If you already have sessions and the times are displaying incorrectly:', 'confab' ); ?></p>
										<ol>
											<li><?php esc_html_e( 'First, set your WordPress timezone to the CORRECT value', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Then use the Time Migration Tool (see the "Migrating Session Times" section below)', 'confab' ); ?></li>
											<li><?php esc_html_e( 'The tool will adjust all existing sessions to display correctly', 'confab' ); ?></li>
										</ol>

										<p style="background: #fef5e6; padding: 15px; margin-top: 20px; border-left: 4px solid #ff9800;">
											<strong><?php esc_html_e( 'Remember:', 'confab' ); ?></strong> <?php esc_html_e( 'Set timezone FIRST, create sessions SECOND. This simple order prevents hours of troubleshooting later.', 'confab' ); ?>
										</p>
									</div>
								</div>

								<!-- Block Editor Method -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="block-method">
										<?php esc_html_e( 'Using the Block Editor (Recommended)', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="block-method">
										<h4><?php esc_html_e( 'Adding a Schedule Block', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'The easiest way to display your conference schedule is using the ConFab block in the WordPress block editor:', 'confab' ); ?></p>
										<ol>
											<li><?php esc_html_e( 'Create or edit a page where you want to display the schedule', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Click the "+" button to add a new block', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Search for "ConFab" or "Conference Schedule"', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Configure the block settings in the sidebar panel', 'confab' ); ?></li>
											<li><?php esc_html_e( 'See a live preview as you adjust settings', 'confab' ); ?></li>
										</ol>
										
										<h4><?php esc_html_e( 'Block Settings Panel', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'The block includes these customization options:', 'confab' ); ?></p>
										<ul>
											<li><strong><?php esc_html_e( 'Date Selection:', 'confab' ); ?></strong> <?php esc_html_e( 'Choose specific dates or multiple dates for tabbed navigation', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Track Filtering:', 'confab' ); ?></strong> <?php esc_html_e( 'Display all tracks or select specific ones', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Color Scheme:', 'confab' ); ?></strong> <?php esc_html_e( 'Choose from 12 professional color schemes including light, dark, earthtones, jeweltones, pastels, ocean, and forest themes (each available in light and dark variants)', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Table Style:', 'confab' ); ?></strong> <?php esc_html_e( 'Select bordered (traditional), borderless (clean minimal), or rounded (modern cards). Note: Applies only to table layout, not grid layout.', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Layout Options:', 'confab' ); ?></strong> <?php esc_html_e( 'Choose table (traditional time-based grid) or grid (modern card-based display)', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Content Display:', 'confab' ); ?></strong> <?php esc_html_e( 'Show title only, title with excerpt, or full session content', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Session Links:', 'confab' ); ?></strong> <?php esc_html_e( 'Link to permalinks, page anchors, or no links', 'confab' ); ?></li>
										</ul>
									</div>
								</div>

								<!-- Shortcode Method -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="shortcode-method">
										<?php esc_html_e( 'Using Shortcodes', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="shortcode-method">
										<h4><?php esc_html_e( 'Basic Usage Examples', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'Use the shortcode in any post, page, or widget area:', 'confab' ); ?></p>
										<pre><code>[confab_schedule]</code></pre>
										
										<p><?php esc_html_e( 'Display today\'s schedule:', 'confab' ); ?></p>
										<pre><code>[confab_schedule date="<?php echo esc_html( gmdate( 'Y-m-d' ) ); ?>"]</code></pre>
										
										<p><?php esc_html_e( 'Show only specific tracks:', 'confab' ); ?></p>
										<pre><code>[confab_schedule tracks="keynote,workshop"]</code></pre>

										<h4><?php esc_html_e( 'Layout Examples', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'Modern grid layout with card-based display:', 'confab' ); ?></p>
										<pre><code>[confab_schedule layout="grid" color_scheme="ocean"]</code></pre>
										
										<p><?php esc_html_e( 'Traditional table with borderless style:', 'confab' ); ?></p>
										<pre><code>[confab_schedule layout="table" table_style="borderless"]</code></pre>

										<h4><?php esc_html_e( 'Color Scheme Examples', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'Professional ocean theme for coastal conferences:', 'confab' ); ?></p>
										<pre><code>[confab_schedule color_scheme="ocean"]</code></pre>
										
										<p><?php esc_html_e( 'Elegant dark jeweltones for evening events:', 'confab' ); ?></p>
										<pre><code>[confab_schedule color_scheme="jeweltones-dark"]</code></pre>

										<h4><?php esc_html_e( 'Content Display Examples', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'Show session excerpts (length controlled in settings above):', 'confab' ); ?></p>
										<pre><code>[confab_schedule content="excerpt"]</code></pre>
										
										<p><?php esc_html_e( 'Multi-day schedule with tabbed navigation:', 'confab' ); ?></p>
										<pre><code>[confab_schedule date="2024-10-15,2024-10-16,2024-10-17"]</code></pre>

										<h4><?php esc_html_e( 'Complete Customization Example', 'confab' ); ?></h4>
										<pre><code>[confab_schedule date="2024-10-15" tracks="keynote,technical" layout="grid" color_scheme="forest" content="excerpt" session_link="permalink"]</code></pre>
									</div>
								</div>

								<!-- Configuration Parameters -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="parameters">
										<?php esc_html_e( 'Available Parameters', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="parameters">
										<h4><?php esc_html_e( 'Complete Parameter Reference', 'confab' ); ?></h4>
										<ul>
											<li><code>date</code> - <?php esc_html_e( 'Specific date (Y-m-d) or comma-separated dates for tabbed navigation. Leave empty for all dates.', 'confab' ); ?></li>
											<li><code>tracks</code> - <?php esc_html_e( 'Track slugs (comma-separated) or "all". Example: "keynote,workshop"', 'confab' ); ?></li>
											<li><code>layout</code> - <?php esc_html_e( '"table" (default - traditional time-based table with track columns) or "grid" (modern card-based display with time slot grouping)', 'confab' ); ?></li>
											<li><code>color_scheme</code> - <?php esc_html_e( 'Choose from: "light" (default), "dark", "earthtones", "earthtones-dark", "jeweltones", "jeweltones-dark", "pastels", "pastels-dark", "ocean", "ocean-dark", "forest", "forest-dark"', 'confab' ); ?></li>
											<li><code>table_style</code> - <?php esc_html_e( '"bordered" (default), "borderless", or "rounded". Note: Only affects table layout, not grid layout.', 'confab' ); ?></li>
											<li><code>content</code> - <?php esc_html_e( '"none" (default - title only), "excerpt" (title with excerpt), or "full" (title with complete content)', 'confab' ); ?></li>
											<li><code>session_link</code> - <?php esc_html_e( '"permalink" (default - links to session page), "anchor" (same-page anchor links), or "none" (no links)', 'confab' ); ?></li>
											<li><code>row_height</code> - <?php esc_html_e( '"match" (default - uniform row heights) or "auto" (dynamic heights). Table layout only.', 'confab' ); ?></li>
										</ul>
									</div>
								</div>

								<!-- Track Order Management -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="track-order">
										<?php esc_html_e( 'Managing Track Display Order', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="track-order">
										<p><?php esc_html_e( 'ConFab displays tracks in alphabetical order by their slug (URL-friendly name). This gives you control over track ordering:', 'confab' ); ?></p>
										
										<h4><?php esc_html_e( 'Default Behavior:', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'When you create tracks with standard names, WordPress automatically generates slugs:', 'confab' ); ?></p>
										<ul>
											<li>"Track A" → slug: <code>track-a</code></li>
											<li>"Track B" → slug: <code>track-b</code></li>
											<li>"Track C" → slug: <code>track-c</code></li>
											<li><?php esc_html_e( 'Displayed in order: Track A, Track B, Track C', 'confab' ); ?></li>
										</ul>
										
										<h4><?php esc_html_e( 'Custom Ordering:', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'For non-alphabetical track names, add numeric prefixes to slugs for precise control:', 'confab' ); ?></p>
										<ul>
											<li>"Business Track" → slug: <code>1-business-track</code></li>
											<li>"Community Track" → slug: <code>2-community-track</code></li>
											<li>"Education Track" → slug: <code>3-education-track</code></li>
											<li><?php esc_html_e( 'Displayed in order: Main Stage, Room 203, Workshop Area', 'confab' ); ?></li>
										</ul>
										
										<h4><?php esc_html_e( 'How to Edit Track Slugs:', 'confab' ); ?></h4>
										<ol>
											<li><?php esc_html_e( 'Go to Sessions > Tracks in your WordPress admin', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Click on a track to edit it', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Find the "Slug" field and modify it', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Update the track to save your changes', 'confab' ); ?></li>
										</ol>
										
										<p><strong><?php esc_html_e( 'Note:', 'confab' ); ?></strong> <?php esc_html_e( 'Track slugs remain stable even if you later change the track name, preserving your custom ordering.', 'confab' ); ?></p>
									</div>
								</div>

								<!-- Grid Layout -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="grid-layout">
										<?php esc_html_e( 'Grid Layout Display', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="grid-layout">
										<p><?php esc_html_e( 'Grid layout provides a modern, card-based alternative to traditional table display:', 'confab' ); ?></p>
										<pre><code>[confab_schedule layout="grid" date="2024-10-15"]</code></pre>
										
										<h4><?php esc_html_e( 'Grid Layout Features:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'Time slot grouping with visible time labels', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Automatic detection of global sessions (spanning all tracks)', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Responsive column layout for concurrent sessions', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Card-based design with hover effects', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Mobile-optimized single-column stacking', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Professional empty state display when no sessions scheduled', 'confab' ); ?></li>
										</ul>
										
										<h4><?php esc_html_e( 'When to Use Grid vs Table:', 'confab' ); ?></h4>
										<ul>
											<li><strong><?php esc_html_e( 'Grid:', 'confab' ); ?></strong> <?php esc_html_e( 'Better for modern designs, fewer concurrent sessions, mobile-first sites', 'confab' ); ?></li>
											<li><strong><?php esc_html_e( 'Table:', 'confab' ); ?></strong> <?php esc_html_e( 'Better for dense schedules with many concurrent tracks, traditional conference formats', 'confab' ); ?></li>
										</ul>
										
										<p><?php esc_html_e( 'Note: Table style variations (bordered/borderless/rounded) do not affect grid layout, which uses a consistent card-based design.', 'confab' ); ?></p>
									</div>
								</div>

								<!-- Multi-Date Navigation -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="multi-date">
										<?php esc_html_e( 'Multi-Date Tab Navigation', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="multi-date">
										<p><?php esc_html_e( 'ConFab automatically creates tabbed navigation when you specify multiple dates:', 'confab' ); ?></p>
										<pre><code>[confab_schedule date="2024-10-15,2024-10-16,2024-10-17"]</code></pre>
										<h4><?php esc_html_e( 'Features Include:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'Automatic tab generation for each date', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Client-side JavaScript tab switching for instant navigation', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Accessible keyboard navigation (left/right arrow keys)', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Mobile-responsive tab layout', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Inherits color scheme from shortcode/block settings', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Professional empty state message when no sessions scheduled', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Works with both table and grid layouts', 'confab' ); ?></li>
										</ul>
										<p><?php esc_html_e( 'Each tab displays the schedule for that specific date, making it easy for attendees to navigate multi-day conferences.', 'confab' ); ?></p>
									</div>
								</div>

								<!-- Excerpt Length Setting -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="excerpt-length">
										<?php esc_html_e( 'Session Excerpt Length', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="excerpt-length">
										<p><?php esc_html_e( 'Control how much of each session description appears in your schedule displays using the "Excerpt Length" setting above.', 'confab' ); ?></p>
										
										<h4><?php esc_html_e( 'How It Works:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'Set the maximum character length (50-500 characters) in the Content Display settings above', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Default is 150 characters, which works well for most conference schedules', 'confab' ); ?></li>
											<li><?php esc_html_e( 'The system respects paragraph boundaries - if the first paragraph fits within your limit, it displays the complete paragraph', 'confab' ); ?></li>
											<li><?php esc_html_e( 'When the first paragraph exceeds your limit, ConFab truncates it intelligently at word boundaries', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Manual excerpts you write in the session editor are also enforced to this length', 'confab' ); ?></li>
										</ul>
										
										<h4><?php esc_html_e( 'Paragraph-Aware Behavior:', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'ConFab\'s excerpt system is designed to create clean, readable session previews:', 'confab' ); ?></p>
										<ul>
											<li><?php esc_html_e( 'If your first paragraph is 140 characters and your limit is 150, the full paragraph appears', 'confab' ); ?></li>
											<li><?php esc_html_e( 'If your first paragraph is 200 characters and your limit is 150, it truncates at 150 with an ellipsis', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Truncation always happens at word boundaries to avoid cutting words in half', 'confab' ); ?></li>
											<li><?php esc_html_e( 'The system maintains at least one-third of your target length to prevent overly aggressive cutting', 'confab' ); ?></li>
										</ul>
										
										<h4><?php esc_html_e( 'Usage Tips:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'Use shorter lengths (50-100) for dense multi-track schedules', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Use longer lengths (200-300) for schedules with few concurrent sessions', 'confab' ); ?></li>
											<li><?php esc_html_e( 'The excerpt length applies everywhere content="excerpt" is used in shortcodes or blocks', 'confab' ); ?></li>
										</ul>
									</div>
								</div>

								<!-- Schedule Page URL -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="schedule-url">
										<?php esc_html_e( 'Schedule Page URL Setting', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="schedule-url">
										<p><?php esc_html_e( 'The Schedule Page URL setting creates "Return to Schedule" links on individual session pages.', 'confab' ); ?></p>
										<h4><?php esc_html_e( 'How It Works:', 'confab' ); ?></h4>
										<ol>
											<li><?php esc_html_e( 'Enter the URL of your main schedule page in the settings above', 'confab' ); ?></li>
											<li><?php esc_html_e( 'When visitors view individual session pages, they\'ll see a "Return to Schedule" link', 'confab' ); ?></li>
											<li><?php esc_html_e( 'This improves navigation and user experience for conference attendees', 'confab' ); ?></li>
										</ol>
										<p><?php esc_html_e( 'Example: If your schedule is at https://yoursite.com/conference-schedule, enter that URL in the setting.', 'confab' ); ?></p>
										<p><?php esc_html_e( 'Note: This setting links all sessions—whether displayed in a schedule block or shortcode—back to the main Schedule Page URL. It is designed for simple sites with only one conference schedule. If your site has multiple schedules, leave this field blank and instead add links on your session pages that point to the appropriate schedule.', 'confab' ); ?></p>
									</div>
								</div>

								<!-- Timezone Troubleshooting -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="timezone-help">
										<?php esc_html_e( 'Session Times and Timezone Settings', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="timezone-help">
										<p><?php esc_html_e( 'ConFab uses your WordPress timezone setting to display session times consistently across your site.', 'confab' ); ?></p>
										
										<h4><?php esc_html_e( 'How Session Times Work:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'Times you enter in the session editor are stored based on your site\'s timezone', 'confab' ); ?></li>
											<li><?php esc_html_e( 'All visitors see the same times regardless of their location', 'confab' ); ?></li>
											<li><?php esc_html_e( 'The time you enter (e.g., 9:00 AM) is the time visitors will see', 'confab' ); ?></li>
										</ul>
										
										<h4><?php esc_html_e( 'Verifying Your Timezone:', 'confab' ); ?></h4>
										<ol>
											<li><?php esc_html_e( 'Go to Settings > General in WordPress admin', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Check the "Timezone" setting', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Make sure it matches your conference location (e.g., "Los Angeles" for Pacific Time)', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Save changes if you updated the timezone', 'confab' ); ?></li>
										</ol>
										
										<h4><?php esc_html_e( 'Important Notes:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'ConFab does NOT automatically adjust times based on visitor location', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Include a note on your schedule like "All times shown are Pacific Time" to help attendees', 'confab' ); ?></li>
											<li><?php esc_html_e( 'If you change your WordPress timezone, existing session times will display differently', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Set your timezone correctly BEFORE creating sessions to avoid confusion', 'confab' ); ?></li>
										</ul>

									</div>
								</div>

								<!-- Time Migration Recipe -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="time-migration-recipe">
										<?php esc_html_e( 'Migrating Session Times (Unsupported)', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="time-migration-recipe">
										<p style="background: #fff3cd; border-left: 4px solid #ff9800; padding: 15px; margin-bottom: 20px;">
											<strong><?php esc_html_e( '⚠️ Important:', 'confab' ); ?></strong> <?php esc_html_e( 'Migration from WP Conference Schedule is completely unsupported. The Time Migration Tool is provided as-is with no guarantees and no support. Test on a staging site first. Back up your database. Use at your own risk.', 'confab' ); ?>
										</p>

										<h4><?php esc_html_e( 'CRITICAL: Set Your Timezone First!', 'confab' ); ?></h4>
										<p><?php esc_html_e( 'Before creating ANY sessions in a new ConFab installation:', 'confab' ); ?></p>
										<ol>
											<li><strong><?php esc_html_e( 'Go to Settings > General > Timezone', 'confab' ); ?></strong></li>
											<li><strong><?php esc_html_e( 'Set it to your conference location (e.g., "Los Angeles" for Pacific Time)', 'confab' ); ?></strong></li>
											<li><strong><?php esc_html_e( 'Save and verify it\'s correct', 'confab' ); ?></strong></li>
											<li><strong><?php esc_html_e( 'THEN create your first session', 'confab' ); ?></strong></li>
										</ol>
										<p style="background: #fef5e6; padding: 10px; margin: 15px 0;">
											<?php esc_html_e( 'If you change your WordPress timezone AFTER creating sessions, all existing sessions will display in the new timezone. This is expected behavior, but can be confusing if you\'ve already published your schedule.', 'confab' ); ?>
										</p>
										
										<h4><?php esc_html_e( 'Why Session Times Might Be Wrong:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'You migrated from WP Conference Schedule (times stored differently)', 'confab' ); ?></li>
											<li><?php esc_html_e( 'You upgraded from ConFab 2.1.3 or earlier (old timezone system)', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Your WordPress timezone changed after creating sessions', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Your server timezone doesn\'t match WordPress timezone', 'confab' ); ?></li>
										</ul>

										<h4><?php esc_html_e( 'Time Migration Tool Recipe (Use At Your Own Risk):', 'confab' ); ?></h4>
										<ol>
											<li>
												<strong><?php esc_html_e( 'Verify Your WordPress Timezone:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Go to Settings > General > Timezone and confirm it matches your conference location. If you need to change it, do that first and save.', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Open the Time Migration Tool:', 'confab' ); ?></strong><br>
												<?php
												printf(
													/* translators: %s: URL to Time Migration Tool */
													esc_html__( 'Click the %s button in the Time Migration card above. This opens a guided wizard.', 'confab' ),
													'<strong>' . esc_html__( '"Open Time Migration Tool"', 'confab' ) . '</strong>'
												);
												?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Pick a Date with Sessions:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Choose any conference date that has sessions. The tool will show you sessions from that day.', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Compare Times:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Look at what times are currently showing vs. what they should be. For example, if a 9:00 AM session shows as 4:00 PM, note the difference.', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Enter Correct Times:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'For 2-3 sample sessions, type in what the correct times should be. The tool calculates the offset automatically.', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Preview the Change:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'The tool shows you "Current → Will Become" for all sessions. Verify this looks right before applying.', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Apply the Migration:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Check the confirmation box and click "Apply Time Migration." This updates ALL sessions at once.', 'confab' ); ?>
											</li>
											<li>
												<strong><?php esc_html_e( 'Verify Your Schedule:', 'confab' ); ?></strong><br>
												<?php esc_html_e( 'Check your published conference schedule to confirm all times now display correctly.', 'confab' ); ?>
											</li>
										</ol>

										<h4><?php esc_html_e( 'Important Safety Notes:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'The Time Migration Tool modifies your session data - back up your database first', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Test on a staging site first if you have many sessions', 'confab' ); ?></li>
											<li><?php esc_html_e( 'The migration history log tracks all changes made by the Time Migration Tool', 'confab' ); ?></li>
										</ul>
									</div>
								</div>

								<!-- Credit Link -->
								<div class="confab-accordion-item">
									<button class="confab-accordion-header" aria-expanded="false" data-target="credit-link">
										<?php esc_html_e( 'Display Credit Link', 'confab' ); ?>
										<span class="confab-accordion-toggle dashicons dashicons-arrow-down"></span>
									</button>
									<div class="confab-accordion-content" id="credit-link">
										<p><?php esc_html_e( 'When enabled, this setting adds a small "Powered by ConFab" link at the bottom of your schedule displays.', 'confab' ); ?></p>
										<h4><?php esc_html_e( 'Supporting Development:', 'confab' ); ?></h4>
										<ul>
											<li><?php esc_html_e( 'Helps other conference organizers discover ConFab', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Shows appreciation for free, open-source software', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Takes up minimal space and matches your site\'s design', 'confab' ); ?></li>
											<li><?php esc_html_e( 'Can be disabled anytime without affecting functionality', 'confab' ); ?></li>
										</ul>
										<p><?php esc_html_e( 'This is completely optional and will not impact your schedule\'s functionality whether enabled or disabled.', 'confab' ); ?></p>
									</div>
								</div>

							</div>
						</div>
						
					</div>
					
				</form>
			</div>
			
			<!-- Attribution (Small Print) -->
			<div class="confab-card confab-card--spaced" style="background: #f9f9f9; border: 1px solid #e5e5e5; font-size: 0.9em; color: #666;">
				<h2 style="font-size: 1.1em; color: #666;"><?php esc_html_e( 'Attribution', 'confab' ); ?></h2>
				<p style="line-height: 1.6;">
					<?php
					printf(
						/* translators: 1: Original plugin link, 2: Amber Hinds profile, 3: Steve Jones profile, 4: Road Warrior Creative profile, 5: The Events Calendar profile */
						esc_html__( 'ConFab builds upon the original %1$s plugin created by %2$s, %3$s, %4$s, and %5$s, now modernized for current WordPress compatibility with enhanced security, accessibility, and features.', 'confab' ),
						'<a href="' . esc_url( 'https://wordpress.org/plugins/wp-conference-schedule/' ) . '" target="_blank" rel="noopener">' . esc_html__( 'WP Conference Schedule', 'confab' ) . '</a>',
						'<a href="' . esc_url( 'https://profiles.wordpress.org/alh0319/' ) . '" target="_blank" rel="noopener">Amber Hinds</a>',
						'<a href="' . esc_url( 'https://profiles.wordpress.org/stevejonesdev/' ) . '" target="_blank" rel="noopener">Steve Jones</a>',
						'<a href="' . esc_url( 'https://profiles.wordpress.org/roadwarriorwp/' ) . '" target="_blank" rel="noopener">Road Warrior Creative</a>',
						'<a href="' . esc_url( 'https://profiles.wordpress.org/theeventscalendar/' ) . '" target="_blank" rel="noopener">The Events Calendar</a>'
					);
					?>
				</p>
			</div>
			
			<!-- Footer -->
			<div class="confab-footer" style="text-align: center; padding: 30px 20px; margin-top: 40px; border-top: 1px solid #ddd; color: #666;">
				<a href="<?php echo esc_url( 'https://wordpress.org/plugins/confab/' ); ?>" 
				   target="_blank" 
				   rel="noopener noreferrer"
				   style="color: #2271b1; text-decoration: none;">
					<?php esc_html_e( 'ConFab on WordPress', 'confab' ); ?>
				</a>
				<span class="confab-footer-sep" style="margin: 0 8px; color: #999;"> • </span>
				<a href="<?php echo esc_url( 'https://wordpress.org/plugins/confab/#reviews' ); ?>" 
				   target="_blank" 
				   rel="noopener noreferrer"
				   style="color: #2271b1; text-decoration: none;">
					<?php esc_html_e( 'Rate Plugin', 'confab' ); ?>
				</a>
				<span class="confab-footer-sep" style="margin: 0 8px; color: #999;"> • </span>
				<a href="<?php echo esc_url( 'https://shop.low.li/downloads/help-the-legend-sail-on/' ); ?>" 
				   target="_blank" 
				   rel="noopener noreferrer"
				   style="color: #2271b1; text-decoration: none;">
					💝 <?php esc_html_e( 'Support ConFab', 'confab' ); ?>
				</a>
			</div>
			
		</div>
	
	<?php
}