<?php
/**
 * Process schedule display requests from shortcode or block with multi-date tab support.
 *
 * Renders conference sessions in table or grid layouts with proper color scheme theming,
 * track filtering, and empty state handling. Coordinates tab navigation for multi-date
 * schedules and tracks usage statistics for review system integration.
 *
 * @param array $props Raw display properties from shortcode or block
 * @return string Sanitized HTML output for schedule display
 * @since 2.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Default excerpt length for session descriptions in schedule displays
 *
 * @since 2.0.0
 */
if ( ! defined( 'CONFAB_EXCERPT_LENGTH' ) ) {
	define( 'CONFAB_EXCERPT_LENGTH', get_option( 'confab_excerpt_length', 150 ) );
}

/**
 * Retrieve filtered track terms for schedule display with Plugin Check compliant validation.
 *
 * Returns associative array of track objects filtered by selection criteria.
 * Handles both explicit track selection and 'all' tracks functionality.
 *
 * @param string $selected_tracks Comma-separated track slugs or 'all' for all tracks
 * @return array Associative array of track objects keyed by term_id
 * @since 2.0.0
 */
function confab_get_schedule_tracks( $selected_tracks ) {
	$tracks = array();
	
	// Sanitize input parameter
	$selected_tracks = sanitize_text_field( $selected_tracks );
	
	if ( 'all' === $selected_tracks ) {
		// Include all available tracks
		$all_tracks = get_terms( array(
			'taxonomy'   => 'confab_track',
			'hide_empty' => false,
		) );
		
		if ( ! is_wp_error( $all_tracks ) && is_array( $all_tracks ) ) {
			foreach ( $all_tracks as $track ) {
				$tracks[ $track->term_id ] = $track;
			}
		}
	} else {
		// Process comma-separated track slugs
		$term_slugs = array_map( 'trim', explode( ',', $selected_tracks ) );
		
		foreach ( $term_slugs as $term_slug ) {
			$term_slug = sanitize_title( $term_slug );
			if ( empty( $term_slug ) ) {
				continue;
			}
			
			$term = get_term_by( 'slug', $term_slug, 'confab_track' );
			if ( $term && ! is_wp_error( $term ) ) {
				$tracks[ $term->term_id ] = $term;
			}
		}
	}

	if ( ! empty( $tracks ) ) {
        uasort( $tracks, function( $a, $b ) {
            return strcmp( $a->slug, $b->slug );
        } );
    }


	return $tracks;
}

/**
 * Retrieve time-sorted session data for schedule display with security validation.
 *
 * Returns sessions organized by timestamp and track for schedule rendering.
 * Includes comprehensive input validation and secure query construction.
 *
 * @param string $schedule_date               Target date for session retrieval (Y-m-d format)
 * @param bool   $tracks_explicitly_specified Whether tracks were explicitly specified
 * @param array  $tracks                      Array of track term objects from confab_get_schedule_tracks()
 * @return array Multi-dimensional array: [timestamp][track_id] = session_id
 * @since 2.0.0
 */
function confab_get_schedule_sessions( $schedule_date, $tracks_explicitly_specified, $tracks ) {
	$query_args = array(
		'post_type'      => 'confab_session',
		'posts_per_page' => -1,
		'post_status'    => 'publish',
		'meta_query'     => array(
			'relation' => 'AND',
			array(
				'key'     => '_confab_session_time',
				'compare' => 'EXISTS',
			),
		),
	);

	// Add date filtering with UTC timestamp conversion
	if ( $schedule_date && strtotime( $schedule_date ) ) {
		// Create DateTime objects in WordPress timezone for the full day
		$wp_timezone = wp_timezone();
		$dt_start_local = DateTime::createFromFormat( 'Y-m-d H:i:s', $schedule_date . ' 00:00:00', $wp_timezone );
		$dt_end_local = DateTime::createFromFormat( 'Y-m-d H:i:s', $schedule_date . ' 23:59:59', $wp_timezone );
		
		// Convert to UTC timestamps for querying
		if ( $dt_start_local && $dt_end_local ) {
			$query_args['meta_query'][] = array(
				'key'     => '_confab_session_time',
				'value'   => array( $dt_start_local->getTimestamp(), $dt_end_local->getTimestamp() ),
				'compare' => 'BETWEEN',
				'type'    => 'NUMERIC',
			);
		}
	}

	// Add track filtering with taxonomy validation
	if ( $tracks_explicitly_specified && ! empty( $tracks ) ) {
		$track_ids = array();
		foreach ( $tracks as $track ) {
			if ( isset( $track->term_id ) && is_numeric( $track->term_id ) ) {
				$track_ids[] = absint( $track->term_id );
			}
		}
		
		if ( ! empty( $track_ids ) ) {
			$query_args['tax_query'] = array(
				array(
					'taxonomy' => 'confab_track',
					'field'    => 'term_id',
					'terms'    => $track_ids,
				),
			);
		}
	}

	// Execute query and organize results
	$sessions       = array();
	$sessions_query = new WP_Query( $query_args );

	if ( $sessions_query->have_posts() ) {
		foreach ( $sessions_query->posts as $session ) {
			$session_time = absint( confab_get_session_meta_compat( $session->ID, '_confab_session_time' ) );
			if ( ! $session_time ) {
				continue;
			}
			
			$session_tracks = get_the_terms( $session->ID, 'confab_track' );
			if ( ! isset( $sessions[ $session_time ] ) ) {
				$sessions[ $session_time ] = array();
			}

			// Handle sessions without tracks
			if ( empty( $session_tracks ) || is_wp_error( $session_tracks ) ) {
				$sessions[ $session_time ][0] = $session->ID;
			} else {
				// Assign session to each of its tracks
				foreach ( $session_tracks as $track ) {
					if ( isset( $track->term_id ) ) {
						$sessions[ $session_time ][ $track->term_id ] = $session->ID;
					}
				}
			}
		}
	}

	// Sort sessions by timestamp for chronological display
	ksort( $sessions );

	return $sessions;
}

/**
 * Generate column structure for schedule table display with track filtering.
 *
 * Determines which columns should be displayed based on track configuration
 * and removes empty columns unless explicitly specified.
 *
 * @param array $tracks                      Track term objects from confab_get_schedule_tracks()
 * @param array $sessions                    Session data from confab_get_schedule_sessions()
 * @param bool  $tracks_explicitly_specified Whether tracks were explicitly specified
 * @return array Array of column identifiers (term_ids) for display
 * @since 2.0.0
 */
function confab_get_schedule_columns( $tracks, $sessions, $tracks_explicitly_specified ) {
	$columns = array();

	// Build initial column structure from tracks
	if ( ! empty( $tracks ) ) {
		foreach ( $tracks as $track ) {
			if ( isset( $track->term_id ) ) {
				$columns[ $track->term_id ] = $track->term_id;
			}
		}
	} else {
		// Default column for sessions without tracks
		$columns[0] = 0;
	}

	// Remove empty columns unless tracks explicitly specified
	if ( ! $tracks_explicitly_specified ) {
		$used_terms = array();

		// Identify which tracks actually have sessions
		foreach ( $sessions as $time_slot ) {
			if ( is_array( $time_slot ) ) {
				foreach ( $time_slot as $term_id => $session_id ) {
					$used_terms[ $term_id ] = $term_id;
				}
			}
		}

		// Keep only columns that have sessions
		$columns = array_intersect( $columns, $used_terms );
	}

	return $columns;
}

/**
 * Sanitize and validate schedule display attributes with Plugin Check compliance.
 *
 * Processes shortcode/block attributes ensuring security validation and providing
 * appropriate defaults for all display parameters.
 *
 * @param array $props Raw attributes from shortcode or block
 * @return array Sanitized and validated attributes array
 * @since 2.0.0
 */
/**
 * Sanitize and validate schedule display attributes with Plugin Check compliance.
 *
 * Processes shortcode/block attributes ensuring security validation and providing
 * appropriate defaults for all display parameters.
 *
 * @param array $props Raw attributes from shortcode or block
 * @return array Sanitized and validated attributes array
 * @since 2.0.0
 */
function confab_preprocess_schedule_attributes( $props ) {
	// Default attribute values with security in mind
	$attr = array(
		'date'         => '',
		'tracks'       => 'all',
		'session_link' => 'permalink',
		'color_scheme' => 'light',
		'table_style'  => 'bordered',
		'align'        => '',
		'layout'       => 'table',
		'row_height'   => 'match',
		'content'      => 'none',
	);

	// Validate props parameter
	if ( ! is_array( $props ) ) {
		return $attr;
	}

	// Process date with validation
	if ( isset( $props['date'] ) ) {
		$date_value = sanitize_text_field( $props['date'] );
		// Basic date format validation
		if ( ! empty( $date_value ) ) {
			$attr['date'] = $date_value;
		}
	}

	// Process color scheme with allowlist validation
	if ( isset( $props['color_scheme'] ) ) {
		$color_scheme = strtolower( sanitize_text_field( $props['color_scheme'] ) );
		$allowed_color_schemes = array( 
			'light', 'dark', 
			'earthtones', 'earthtones-dark', 
			'jeweltones', 'jeweltones-dark', 
			'pastels', 'pastels-dark', 
			'ocean', 'ocean-dark', 
			'forest', 'forest-dark',
			'autumn', 'autumn-dark',
			'terracotta', 'terracotta-dark',
			'burgundy', 'burgundy-dark',
			'honey', 'honey-dark',
			'sunrise', 'sunrise-dark',
			'lagoon', 'lagoon-dark',
			'sanguine', 'sanguine-dark'
		);
		if ( in_array( $color_scheme, $allowed_color_schemes, true ) ) {
			$attr['color_scheme'] = $color_scheme;
		}
	}

	// Process table style with allowlist validation
	if ( isset( $props['table_style'] ) ) {
		$table_style = strtolower( sanitize_text_field( $props['table_style'] ) );
		if ( in_array( $table_style, array( 'bordered', 'borderless', 'rounded' ), true ) ) {
			$attr['table_style'] = $table_style;
		}
	}

	// Process layout with allowlist validation
	if ( isset( $props['layout'] ) ) {
		$layout = strtolower( sanitize_text_field( $props['layout'] ) );
		if ( in_array( $layout, array( 'table', 'grid' ), true ) ) {
			$attr['layout'] = $layout;
		}
	}

	// Process row height with allowlist validation
	if ( isset( $props['row_height'] ) ) {
		$row_height = strtolower( sanitize_text_field( $props['row_height'] ) );
		if ( in_array( $row_height, array( 'match', 'auto' ), true ) ) {
			$attr['row_height'] = $row_height;
		}
	}

	// Process content display with allowlist validation
	if ( isset( $props['content'] ) ) {
		$content = strtolower( sanitize_text_field( $props['content'] ) );
		if ( in_array( $content, array( 'none', 'excerpt', 'full' ), true ) ) {
			$attr['content'] = $content;
		}
	}

	// Process session link with allowlist validation
	if ( isset( $props['session_link'] ) ) {
		$session_link = strtolower( sanitize_text_field( $props['session_link'] ) );
		if ( in_array( $session_link, array( 'permalink', 'anchor', 'none' ), true ) ) {
			$attr['session_link'] = $session_link;
		}
	}

	// Process alignment with allowlist validation
	if ( isset( $props['align'] ) ) {
		$align = strtolower( sanitize_text_field( $props['align'] ) );
		if ( 'wide' === $align ) {
			$attr['align'] = 'alignwide';
		} elseif ( 'full' === $align ) {
			$attr['align'] = 'alignfull';
		}
	}

	// Process tracks with sanitization
	if ( isset( $props['tracks'] ) ) {
		$tracks = sanitize_text_field( $props['tracks'] );
		if ( ! empty( $tracks ) ) {
			$attr['tracks'] = $tracks;
		}
	}

	// Ensure critical attributes are never empty - fallback to defaults
	if ( empty( $attr['color_scheme'] ) ) {
		$attr['color_scheme'] = 'light';
	}
	
	if ( empty( $attr['table_style'] ) ) {
		$attr['table_style'] = 'bordered';
	}
	
	if ( empty( $attr['layout'] ) ) {
		$attr['layout'] = 'table';
	}

	return $attr;
}

/**
 * Render multi-date tab navigation with Plugin Check compliant security validation.
 *
 * Creates accessible tab interface for multi-day conference schedules with proper
 * nonce verification and user input sanitization.
 *
 * @param array $dates Array of date strings for tab creation
 * @return string Sanitized HTML output for tab navigation
 * @since 2.0.0
 */
function confab_render_schedule_tabs( $dates, $color_scheme = 'light' ) {
	if ( ! is_array( $dates ) || count( $dates ) <= 1 ) {
		return '';
	}

	// Plugin Check compliant validation for tab parameter
	// This parameter only affects front-end display. No nonce needed per WP security guidelines.
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- display-only, sanitized input
	$current_tab = isset( $_GET['confab-tab'] ) ? absint( wp_unslash( $_GET['confab-tab'] ) ) : 1;
	
	// Validate tab number is within range
	if ( $current_tab < 1 || $current_tab > count( $dates ) ) {
		$current_tab = 1;
	}

	$output = '<div class="confab-tabs tabs confab-color-scheme-' . esc_attr( $color_scheme ) . '">';
	$output .= '<div class="confab-tabs-list" role="tablist" aria-label="' . esc_attr__( 'Conference Schedule Tabs', 'confab' ) . '">';
	
	$tab_count = 1;
	foreach ( $dates as $date ) {
		$date = sanitize_text_field( $date );
		$timestamp = strtotime( $date );
		
		if ( ! $timestamp ) {
			continue;
		}
		
		$is_selected = ( $tab_count === $current_tab );
		$tabindex    = $is_selected ? '0' : '-1';
		$selected    = $is_selected ? 'true' : 'false';
		
		$output .= sprintf(
			'<button class="confab-tabs-list-button" role="tab" aria-selected="%s" aria-controls="confab-panel-%d" id="tab-%d" data-id="%d" tabindex="%s">%s</button>',
			esc_attr( $selected ),
			absint( $tab_count ),
			absint( $tab_count ),
			absint( $tab_count ),
			esc_attr( $tabindex ),
			esc_html( wp_date( 'l, F j', $timestamp ) )
		);
		
		$tab_count++;
	}
	
	$output .= '</div>';
	
	return $output;
}

/**
 * Generate table layout HTML for conference schedule display.
 *
 * Creates accessible table structure with proper escaping and semantic markup
 * for screen readers and responsive design.
 *
 * @param array $attr      Sanitized display attributes
 * @param array $tracks    Track term objects
 * @param array $sessions  Session data organized by time and track
 * @param array $columns   Column structure for table display
 * @return string Sanitized HTML table output
 * @since 2.0.0
 */
function confab_render_table_layout( $attr, $tracks, $sessions, $columns ) {
	$time_format = get_option( 'time_format', 'g:i a' );
	
	// Build CSS classes with color scheme and table style
	$table_classes = array(
		'confab-schedule',
		'confab-color-scheme-' . esc_attr( $attr['color_scheme'] ),
		'confab-layout-' . esc_attr( $attr['layout'] ),
		'confab-table-style-' . esc_attr( $attr['table_style'] )
	);
	
	$html = '<div class="confab-schedule-wrapper ' . esc_attr( $attr['align'] ) . '">';
	$html .= '<table class="' . esc_attr( implode( ' ', $table_classes ) ) . '" border="0">';
	
	// Rest of function remains the same...
	// Table header
	$html .= '<thead><tr>';
	$html .= '<th class="confab-col-time">' . esc_html__( 'Time', 'confab' ) . '</th>';
	
	foreach ( $columns as $term_id ) {
		$track = get_term( $term_id, 'confab_track' );
		$track_name = ( $track && ! is_wp_error( $track ) ) ? $track->name : '';
		$track_desc = ( $track && ! is_wp_error( $track ) ) ? $track->description : '';
		
		$html .= sprintf(
			'<th class="confab-col-track"><span class="confab-track-name">%s</span> <span class="confab-track-description">%s</span></th>',
			esc_html( $track_name ),
			esc_html( $track_desc )
		);
	}
	
	$html .= '</tr></thead>';
	
	// Table body
	$html .= '<tbody>';
	
	foreach ( $sessions as $time => $entry ) {
		$html .= confab_render_table_row( $time, $entry, $columns, $attr, $time_format );
	}
	
	$html .= '</tbody></table>';
	
	// Optional credit link
	if ( get_option( 'confab_field_byline' ) ) {
		$html .= '<div class="confab-promo"><small>' . 
				sprintf(
					/* translators: %s: plugin website URL */
					__( 'Powered by %s', 'confab' ),
					'<a href="' . esc_url( 'https://low.li' ) . '" target="_blank">ConFab</a>'
				) . 
				'</small></div>';
	}
	
	$html .= '</div>';
	
	return $html;
}

/**
 * Render individual table row with session data and security validation.
 *
 * @param int    $time        Session timestamp
 * @param array  $entry       Session data for this time slot
 * @param array  $columns     Column structure
 * @param array  $attr        Display attributes
 * @param string $time_format WordPress time format
 * @return string Sanitized HTML table row
 * @since 2.0.0
 */
function confab_render_table_row( $time, $entry, $columns, $attr, $time_format ) {
	$skip_next = 0;
	$columns_html = '';
	
	foreach ( $columns as $key => $term_id ) {
		// Skip cells when colspan is active
		if ( $skip_next > 0 ) {
			$skip_next--;
			continue;
		}

		// Handle empty cells
		if ( empty( $entry[ $term_id ] ) ) {
			$columns_html .= '<td class="confab-session-empty"></td>';
			continue;
		}

		// Handle custom text entries
		if ( is_string( $entry[ $term_id ] ) ) {
			$columns_html .= sprintf(
				'<td colspan="%d" class="confab-session-custom">%s</td>',
				count( $columns ),
				esc_html( $entry[ $term_id ] )
			);
			break;
		}

		// Process session data
		$session_data = confab_get_session_data( $entry[ $term_id ] );
		if ( ! $session_data ) {
			continue;
		}

		// Calculate colspan for multi-track sessions
		$colspan = confab_calculate_colspan( $entry, $columns, $key, $session_data['session']->ID );
		$skip_next = $colspan - 1;

		// Generate session cell content
		$cell_content = confab_render_session_cell( $session_data, $attr );
		
		$columns_html .= sprintf(
			'<td colspan="%d" class="%s" data-track-title="%s" data-session-id="%s">%s</td>',
			absint( $colspan ),
			esc_attr( implode( ' ', $session_data['classes'] ) ),
			esc_attr( $session_data['track_titles'] ),
			esc_attr( $session_data['session']->ID ),
			$cell_content
		);
	}

	// Generate row classes
	$row_classes = array();
	$row_classes[] = sanitize_html_class( 'confab-time-' . wp_date( $time_format, $time ) );
	
	if ( isset( $session_data ) ) {
		$colspan_count = isset( $colspan ) ? $colspan : 1;
		if ( $colspan_count === count( $columns ) ) {
			$row_classes[] = 'confab-global-session';
			$row_classes[] = 'confab-global-session-' . esc_attr( $session_data['session_type'] );
			$row_classes[] = sanitize_html_class( sanitize_title_with_dashes( $session_data['session']->post_title ) );
		}
	}

	$html = sprintf( '<tr class="%s">', esc_attr( implode( ' ', $row_classes ) ) );
	$html .= sprintf(
		'<td class="confab-time">%s</td>',
		str_replace( ' ', '&nbsp;', esc_html( wp_date( $time_format, $time ) ) )
	);
	$html .= $columns_html;
	$html .= '</tr>';

	return $html;
}

/**
 * Retrieve and validate session data with security checks.
 *
 * @param int $session_id WordPress post ID for session
 * @return array|false Session data array or false on failure
 * @since 2.0.0
 */
function confab_get_session_data( $session_id ) {
    $session = get_post( $session_id );
    
    if ( ! $session || 'confab_session' !== $session->post_type || 'publish' !== $session->post_status ) {
        return false;
    }

    $session_tracks = get_the_terms( $session->ID, 'confab_track' );
    $session_type = confab_get_session_meta_compat( $session->ID, '_confab_session_type' );
    $speakers = confab_get_session_meta_compat( $session->ID, '_confab_session_speakers' );

    // Validate session type - only accept new names
    $valid_types = array( 'normal', 'featured', 'alternate' );
    
    if ( ! in_array( $session_type, $valid_types, true ) ) {
        $session_type = 'normal';
    }

    // Build CSS classes
    $classes = array();
    if ( is_array( $session_tracks ) ) {
        foreach ( $session_tracks as $track ) {
            $classes[] = 'confab-track-' . $track->slug;
        }
    }
    
    $classes[] = 'confab-session-type-' . $session_type;
    $classes[] = 'confab-session-' . $session->post_name;

    // Build track titles string
    $track_titles = '';
    if ( is_array( $session_tracks ) ) {
        $track_titles = implode( ', ', wp_list_pluck( $session_tracks, 'name' ) );
    }

    return array(
        'session'       => $session,
        'session_type'  => $session_type,
        'speakers'      => $speakers,
        'tracks'        => $session_tracks,
        'track_titles'  => $track_titles,
        'classes'       => $classes,
    );
}

/**
 * Calculate colspan value for multi-track session display.
 *
 * @param array  $entry      Session data for time slot
 * @param array  $columns    Column structure
 * @param mixed  $key        Current column key
 * @param int    $session_id Session post ID
 * @return int Colspan value for session cell
 * @since 2.0.0
 */
function confab_calculate_colspan( $entry, $columns, $key, $session_id ) {
	$colspan = 1;
	$remaining_columns = array_slice( $columns, array_search( $key, array_keys( $columns ) ) + 1, null, true );

	foreach ( $remaining_columns as $column_key => $term_id ) {
		if ( ! empty( $entry[ $term_id ] ) && absint( $entry[ $term_id ] ) === absint( $session_id ) ) {
			$colspan++;
		} else {
			break;
		}
	}

	return $colspan;
}

/**
 * Render session cell content with proper escaping and filtering support.
 *
 * @param array $session_data Session data from confab_get_session_data()
 * @param array $attr         Display attributes
 * @return string Sanitized HTML content for session cell
 * @since 2.0.0
 */
function confab_render_session_cell( $session_data, $attr ) {
	$session = $session_data['session'];
	$session_type = $session_data['session_type'];
	$speakers = $session_data['speakers'];
	
	$content = '<div class="confab-session-cell-content">';

	// Session content header filter
	$header_content = apply_filters( 'confab_session_content_header', $session->ID );
	if ( $header_content !== $session->ID ) {
		$content .= wp_kses_post( $header_content );
	}

	// Session title with appropriate linking based on session_link attribute only
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Intentionally applying WordPress core's 'the_title' filter for consistency with standard post title display and theme compatibility.
	$session_title = apply_filters( 'the_title', $session->post_title );

	if ( 'permalink' === $attr['session_link'] ) {
		$content .= sprintf(
			'<h3><a class="confab-session-title" href="%s">%s</a></h3>',
			esc_url( get_permalink( $session->ID ) ),
			esc_html( $session_title )
		);
	} elseif ( 'anchor' === $attr['session_link'] ) {
		$content .= sprintf(
			'<h3><a class="confab-session-title" href="%s">%s</a></h3>',
			esc_url( '#' . $session->post_name ),
			esc_html( $session_title )
		);
	} else {
		// 'none' - no link
		$content .= sprintf(
			'<h3><span class="confab-session-title">%s</span></h3>',
			esc_html( $session_title )
		);
	}

	// Session content display 
	if ( 'full' === $attr['content'] ) {
		$session_content = get_post_field( 'post_content', $session->ID );
		if ( $session_content ) {
			$content .= wp_kses_post( $session_content );
		}
	} elseif ( 'excerpt' === $attr['content'] ) {
		$session_excerpt = confab_get_session_excerpt( $session->ID );
		if ( $session_excerpt ) {
			$content .= '<p>' . esc_html( $session_excerpt ) . '</p>';
		}
	}

	// Speaker information
	if ( $speakers ) {
		$content .= sprintf(
			'<p class="confab-session-speakers"><strong>%s</strong> %s</p>',
			esc_html__( 'Speaker(s):', 'confab' ),
			esc_html( $speakers )
		);
	}

	// Location information
	$session_location = get_the_terms( $session_data['session']->ID, 'confab_location' );
	if ( $session_location && ! is_wp_error( $session_location ) ) {
		$location_name = $session_location[0]->name;
		$content .= sprintf(
			'<p class="confab-session-location"><strong>%s</strong> %s</p>',
			esc_html__( 'Location:', 'confab' ),
			esc_html( $location_name )
		);
	}

	// Session content footer filter
	$footer_content = apply_filters( 'confab_session_content_footer', $session->ID );
	if ( $footer_content !== $session->ID ) {
		$content .= wp_kses_post( $footer_content );
	}

	$content .= '</div>';

	return $content;
}

/**
 * Render individual session card for grid layout display.
 *
 * Creates a complete session card with time, title, content, and speakers
 * formatted specifically for grid layout display.
 *
 * @param array $session_data Session data from confab_get_session_data()
 * @param array $attr         Display attributes
 * @param int   $time         Session timestamp
 * @return string Sanitized HTML content for grid session card
 * @since 2.1.0
 */
function confab_render_grid_session( $session_data, $attr, $time ) {
	$session = $session_data['session'];
	$speakers = $session_data['speakers'];
	$time_format = get_option( 'time_format', 'g:i a' );
	
	$content = '<div class="confab-grid-session-card">';
	
	// Time display for grid cards
	$content .= sprintf(
		'<div class="confab-grid-session-time">%s</div>',
		esc_html( wp_date( $time_format, $time ) )
	);
	
	// Session content header filter
	$header_content = apply_filters( 'confab_session_content_header', $session->ID );
	if ( $header_content !== $session->ID ) {
		$content .= wp_kses_post( $header_content );
	}
	
	// Session title with appropriate linking
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- Intentionally applying WordPress core's 'the_title' filter for consistency with standard post title display and theme compatibility.
	$session_title = apply_filters( 'the_title', $session->post_title );

	if ( 'permalink' === $attr['session_link'] ) {
		$content .= sprintf(
			'<h3 class="confab-grid-session-title"><a href="%s">%s</a></h3>',
			esc_url( get_permalink( $session->ID ) ),
			esc_html( $session_title )
		);
	} elseif ( 'anchor' === $attr['session_link'] ) {
		$content .= sprintf(
			'<h3 class="confab-grid-session-title"><a href="%s">%s</a></h3>',
			esc_url( '#' . $session->post_name ),
			esc_html( $session_title )
		);
	} else {
		$content .= sprintf(
			'<h3 class="confab-grid-session-title">%s</h3>',
			esc_html( $session_title )
		);
	}

	// Session content display
	if ( 'full' === $attr['content'] ) {
		$session_content = get_post_field( 'post_content', $session->ID );
		if ( $session_content ) {
			$content .= '<div class="confab-grid-session-content">' . wp_kses_post( $session_content ) . '</div>';
		}
	} elseif ( 'excerpt' === $attr['content'] ) {
		$session_excerpt = confab_get_session_excerpt( $session->ID );
		if ( $session_excerpt ) {
			$content .= '<p class="confab-grid-session-excerpt">' . esc_html( $session_excerpt ) . '</p>';
		}
	}

	// Speaker information
	if ( $speakers ) {
		$content .= sprintf(
			'<p class="confab-grid-session-speakers"><strong>%s</strong> %s</p>',
			esc_html__( 'Speaker(s):', 'confab' ),
			esc_html( $speakers )
		);
	}

	// Location information
	$session_location = get_the_terms( $session->ID, 'confab_location' );
	if ( $session_location && ! is_wp_error( $session_location ) ) {
		$location_name = $session_location[0]->name;
		$content .= sprintf(
			'<p class="confab-grid-session-location"><strong>%s</strong> %s</p>',
			esc_html__( 'Location:', 'confab' ),
			esc_html( $location_name )
		);
	}

	// Session content footer filter
	$footer_content = apply_filters( 'confab_session_content_footer', $session->ID );
	if ( $footer_content !== $session->ID ) {
		$content .= wp_kses_post( $footer_content );
	}

	$content .= '</div>';

	return $content;
}

/**
 * Process schedule display requests from shortcode or block with multi-date tab support.
 *
 * Renders conference sessions in table or grid layouts with proper color scheme theming,
 * track filtering, and empty state handling. Coordinates tab navigation for multi-date
 * schedules and tracks usage statistics for review system integration.
 *
 * @param array $props Raw display properties from shortcode or block
 * @return string Sanitized HTML output for schedule display
 * @since 2.0.0
 */
function confab_scheduleOutput( $props ) {
	// Validate and sanitize input properties
	$attr = confab_preprocess_schedule_attributes( $props );
	
	// Track schedule display for review system
	$confab_instance = Confab_Plugin::get_instance();
	$confab_instance->increment_usage_stats( 'schedules_displayed' );
	
	if ( empty( $attr['date'] ) ) {
		return '<p>' . esc_html__( 'No schedule date specified.', 'confab' ) . '</p>';
	}

	// Process multiple dates for tabbed interface
	$dates = array_map( 'trim', explode( ',', $attr['date'] ) );
	$dates = array_filter( $dates ); // Remove empty values

	if ( empty( $dates ) ) {
		return '<p>' . esc_html__( 'Invalid schedule date format.', 'confab' ) . '</p>';
	}

	$output = '';

	// Render tab navigation for multi-date schedules
	if ( count( $dates ) > 1 ) {
		$output .= confab_render_schedule_tabs( $dates, $attr['color_scheme'] );
	}

	// Plugin Check compliant tab selection
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab selection for display only, no data processing
	$current_tab = isset( $_GET['confab-tab'] ) ? absint( wp_unslash( $_GET['confab-tab'] ) ) : 1;

	// Render schedule for each date
	$panel_count = 1;
	foreach ( $dates as $date ) {
		$date = sanitize_text_field( $date );
		
		// Multi-date panel wrapper
		if ( count( $dates ) > 1 ) {
			$is_hidden = ( $panel_count !== $current_tab ) ? 'hidden' : '';
			$output .= sprintf(
				'<div class="confab-tabs-panel" id="confab-panel-%d" role="tabpanel" tabindex="0" aria-labelledby="tab-%d" %s>',
				absint( $panel_count ),
				absint( $panel_count ),
				esc_attr( $is_hidden )
			);
		}

		// Generate schedule content
		$single_attr = $attr;
		$single_attr['date'] = $date;
		
		$tracks = confab_get_schedule_tracks( $single_attr['tracks'] );
		$tracks_explicitly_specified = ( 'all' !== $single_attr['tracks'] );
		$sessions = confab_get_schedule_sessions( $date, $tracks_explicitly_specified, $tracks );
		$columns = confab_get_schedule_columns( $tracks, $sessions, $tracks_explicitly_specified );

		// Check for empty sessions before rendering
			if ( empty( $sessions ) ) {
				$site_name = get_bloginfo( 'name' );
				if ( empty( $site_name ) ) {
					$site_name = __( 'This Site', 'confab' );
				}
				$output .= '<div class="confab-schedule-wrapper ' . esc_attr( $single_attr['align'] ) . '">';
				$output .= '<div class="confab-schedule confab-color-scheme-' . esc_attr( $single_attr['color_scheme'] ) . ' confab-layout-' . esc_attr( $single_attr['layout'] ) . '">';
				$output .= '<div class="confab-empty-state">';
				$output .= '<div class="confab-empty-state-header">';
				$output .= '<h2>' . 
						sprintf (
							/* translators: %s: site name */
							esc_html__( 'Conference Schedule For %s', 'confab' ),
							esc_html( $site_name )
						);
				$output .= '</h2>';
				$output .= '<span class="confab-empty-state-subtitle">' . esc_html__( 'ConFab • Professional Conference Scheduling', 'confab' ) . '</span>';
				$output .= '</div>';
				
				// Format and display the date
				$timestamp = strtotime( $date );
				$formatted_date = $timestamp ? wp_date( 'l, F j, Y', $timestamp ) : $date;
				
				$output .= '<div class="confab-empty-state-message">';
				$output .= sprintf(
					/* translators: %s: formatted date */
					esc_html__( 'There are no sessions scheduled for %s.', 'confab' ),
					esc_html( $formatted_date )
				);
				$output .= '</div>';
				
				$output .= '</div>';
				$output .= '</div>';
				$output .= '</div>';
			} else {
			// Render based on layout type
			if ( 'table' === $single_attr['layout'] ) {
				$output .= confab_render_table_layout( $single_attr, $tracks, $sessions, $columns );
			} elseif ( 'grid' === $single_attr['layout'] ) {
				$output .= confab_render_grid_layout( $single_attr, $tracks, $sessions, $columns );
			}
		}

		// Close panel wrapper
		if ( count( $dates ) > 1 ) {
			$output .= '</div>';
		}
		
		$panel_count++;
	}

	// Close tabs wrapper
	if ( count( $dates ) > 1 ) {
		$output .= '</div>';
	}

	return $output;
}

/**
 * Generate grid layout HTML for conference schedule display.
 *
 * Creates CSS Grid-based schedule layout with time slot grouping, track columns,
 * and proper handling of global sessions. Mirrors table layout logic for consistent
 * session organization and eliminates duplicate session rendering.
 *
 * @param array $attr     Sanitized display attributes
 * @param array $tracks   Track term objects from confab_get_schedule_tracks()
 * @param array $sessions Session data organized by time and track
 * @param array $columns  Column structure for grid display
 * @return string Sanitized HTML grid output
 * @since 2.1.0
 */
function confab_render_grid_layout( $attr, $tracks, $sessions, $columns ) {
	$time_format = get_option( 'time_format', 'g:i a' );
	
	// Build CSS classes with color scheme
	$grid_classes = array(
		'confab-schedule',
		'confab-color-scheme-' . esc_attr( $attr['color_scheme'] ),
		'confab-layout-' . esc_attr( $attr['layout'] ),
		'confab-table-style-' . esc_attr( $attr['table_style'] )
	);
	
	$html = '<div class="confab-schedule-wrapper ' . esc_attr( $attr['align'] ) . '">';
	$html .= '<div class="' . esc_attr( implode( ' ', $grid_classes ) ) . '">';
	
	// Loop through each time slot
	foreach ( $sessions as $time => $entry ) {
		$html .= confab_render_grid_time_slot( $time, $entry, $columns, $attr, $time_format );
	}
	
	$html .= '</div>';
	
	// Optional credit link
	if ( get_option( 'confab_field_byline' ) ) {
		$html .= '<div class="confab-promo"><small>' . 
				sprintf(
					/* translators: %s: plugin website URL */
					__( 'Powered by %s', 'confab' ),
					'<a href="' . esc_url( 'https://wordpress.org/plugins/confab/' ) . '" target="_blank">ConFab</a>'
				) . 
				'</small></div>';
	}
	
	$html .= '</div>';
	
	return $html;
}

/**
 * Render individual time slot for grid layout with track columns.
 *
 * Processes a single time slot, determining if sessions are global (spanning all tracks)
 * or concurrent (multiple sessions at same time). Handles track identification and
 * prevents duplicate session rendering.
 *
 * @param int    $time        Session timestamp
 * @param array  $entry       Session data for this time slot
 * @param array  $columns     Column structure
 * @param array  $attr        Display attributes
 * @param string $time_format WordPress time format
 * @return string Sanitized HTML for time slot
 * @since 2.1.0
 */
function confab_render_grid_time_slot( $time, $entry, $columns, $attr, $time_format ) {
	$html = '<div class="confab-grid-time-slot">';
	
	// Time label
	$html .= sprintf(
		'<div class="confab-grid-time-label">%s</div>',
		esc_html( wp_date( $time_format, $time ) )
	);
	
	// Check if this is a global session (single session spanning all tracks)
	$is_global = false;
	$first_session_id = null;
	
	foreach ( $entry as $term_id => $session_id ) {
		if ( is_numeric( $session_id ) ) {
			if ( $first_session_id === null ) {
				$first_session_id = $session_id;
			} elseif ( $first_session_id !== $session_id ) {
				// Different sessions found - not global
				break;
			}
		}
	}
	
	// If we only found one unique session ID across all tracks, it's global
	if ( $first_session_id !== null ) {
		$session_count = 0;
		foreach ( $entry as $session_id ) {
			if ( is_numeric( $session_id ) && absint( $session_id ) === absint( $first_session_id ) ) {
				$session_count++;
			}
		}
		
		if ( $session_count >= count( $columns ) ) {
			$is_global = true;
		}
	}
	
	if ( $is_global ) {
		// Global session - render once, full width
		$html .= '<div class="confab-grid-global">';
		$session_data = confab_get_session_data( $first_session_id );
		
		if ( $session_data ) {
			$html .= sprintf(
				'<div class="%s" data-track="All Tracks" data-session-id="%d">',
				esc_attr( implode( ' ', array_merge( array( 'confab-grid-track-cell' ), $session_data['classes'] ) ) ),
				absint( $first_session_id )
			);
			$html .= confab_render_grid_session( $session_data, $attr, $time );
			$html .= '</div>';
		}
		
		$html .= '</div>';
	} else {
		// Concurrent sessions - render track columns
		$column_count = count( $columns );
		$html .= sprintf(
			'<div class="confab-grid-tracks" style="grid-template-columns: repeat(%d, 1fr);">',
			absint( $column_count )
		);
		
		// Track which sessions we've already rendered to prevent duplicates
		$rendered_sessions = array();
		
		foreach ( $columns as $term_id ) {
			$track = get_term( $term_id, 'confab_track' );
			$track_name = ( $track && ! is_wp_error( $track ) ) ? $track->name : '';
			
			// Check if this track has a session
			if ( ! empty( $entry[ $term_id ] ) && is_numeric( $entry[ $term_id ] ) ) {
				$session_id = absint( $entry[ $term_id ] );
				
				// Skip if we've already rendered this session
				if ( in_array( $session_id, $rendered_sessions, true ) ) {
					continue;
				}
				
				$rendered_sessions[] = $session_id;
				$session_data = confab_get_session_data( $session_id );
				
				if ( $session_data ) {
					// Calculate colspan for multi-track sessions
					$colspan = confab_calculate_grid_colspan( $entry, $columns, $term_id, $session_id );
					
					$cell_classes = array_merge(
						array( 'confab-grid-track-cell' ),
						$session_data['classes']
					);
					
					$html .= sprintf(
						'<div class="%s" style="grid-column: span %d;" data-track="%s" data-session-id="%d">',
						esc_attr( implode( ' ', $cell_classes ) ),
						absint( $colspan ),
						esc_attr( $session_data['track_titles'] ),
						absint( $session_id )
					);
					$html .= confab_render_grid_session( $session_data, $attr, $time );
					$html .= '</div>';
				}
			} else {
				// Empty cell
				$html .= '<div class="confab-grid-empty"></div>';
			}
		}
		
		$html .= '</div>';
	}
	
	$html .= '</div>';
	
	return $html;
}

/**
 * Calculate grid column span for multi-track sessions.
 *
 * Determines how many track columns a session should span based on its
 * track assignments. Prevents duplicate rendering by identifying sessions
 * that appear in consecutive track columns.
 *
 * @param array $entry      Session data for time slot
 * @param array $columns    Column structure
 * @param int   $term_id    Current track term ID
 * @param int   $session_id Session post ID
 * @return int Column span value for CSS Grid
 * @since 2.1.0
 */
function confab_calculate_grid_colspan( $entry, $columns, $term_id, $session_id ) {
	$colspan = 1;
	$found_current = false;
	
	foreach ( $columns as $column_term_id ) {
		// Skip until we find the current term_id
		if ( ! $found_current ) {
			if ( $column_term_id === $term_id ) {
				$found_current = true;
			}
			continue;
		}
		
		// Check subsequent columns for same session
		if ( ! empty( $entry[ $column_term_id ] ) && absint( $entry[ $column_term_id ] ) === absint( $session_id ) ) {
			$colspan++;
		} else {
			break;
		}
	}
	
	return $colspan;
}

/**
 * Get session meta with backward compatibility for old wpcs_ prefixes.
 *
 * @param int    $post_id Post ID
 * @param string $meta_key New confab_ meta key
 * @return mixed Meta value or empty string
 * @since 2.0.0
 */
function confab_get_session_meta_compat( $post_id, $meta_key ) {
	// Try new confab_ prefix first
	$value = get_post_meta( $post_id, $meta_key, true );
	
	if ( $value ) {
		return $value;
	}
	
	// Fallback to old wpcs_ prefix
	$old_key = str_replace( '_confab_', '_wpcs_', $meta_key );
	return get_post_meta( $post_id, $old_key, true );
}

/**
 * Get session excerpt with custom 150-character length for schedule display.
 *
 * Generates excerpts specifically for schedule display with consistent length
 * and proper word boundary handling.
 *
 * @param int $session_id Session post ID
 * @return string Generated excerpt or empty string
 * @since 2.0.0
 */
function confab_get_session_excerpt( $session_id ) {
	$session = get_post( $session_id );
	
	if ( ! $session || 'confab_session' !== $session->post_type ) {
		return '';
	}
	
	$excerpt_length = CONFAB_EXCERPT_LENGTH;
	
	// If there's a manual excerpt, use it (but still limit length)
	if ( ! empty( $session->post_excerpt ) ) {
		$excerpt = $session->post_excerpt;
		if ( strlen( $excerpt ) <= $excerpt_length ) {
			return $excerpt;
		}
		// Trim manual excerpt if it's too long
		$excerpt = substr( $excerpt, 0, $excerpt_length );
		$last_space = strrpos( $excerpt, ' ' );
		// Don't cut too aggressively - keep at least 1/3 of target length
		$min_length = (int) ( $excerpt_length / 3 );
		if ( $last_space !== false && $last_space > $min_length ) {
			$excerpt = substr( $excerpt, 0, $last_space );
		}
		return $excerpt . '...';
	}
	
	// Generate excerpt from content
	$content = strip_shortcodes( $session->post_content );
	$content = wp_strip_all_tags( $content );
	$content = trim( $content );
	
	if ( empty( $content ) ) {
		return '';
	}
	
	// Split content into paragraphs
	$paragraphs = preg_split( '/\n\s*\n/', $content );
	$paragraphs = array_filter( array_map( 'trim', $paragraphs ) );
	
	if ( empty( $paragraphs ) ) {
		return '';
	}
	
	// Get first paragraph
	$first_paragraph = $paragraphs[0];
	
	// If first paragraph is at or under limit, use it as-is
	if ( strlen( $first_paragraph ) <= $excerpt_length ) {
		return $first_paragraph;
	}
	
	// First paragraph is too long - truncate it at character limit
	$excerpt = substr( $first_paragraph, 0, $excerpt_length );
	$last_space = strrpos( $excerpt, ' ' );
	
	// Don't cut too aggressively - keep at least 1/3 of target length
	$min_length = (int) ( $excerpt_length / 3 );
	if ( $last_space !== false && $last_space > $min_length ) {
		$excerpt = substr( $excerpt, 0, $last_space );
	}
	
	return $excerpt . '...';
}