<?php

/**
 * Custom Post Type Registration
 * FILE: inc/post-types.php
 *
 * Registers confab_session custom post type for conference schedule management.
 * Includes admin interface customizations and dashboard integration.
 *
 * @package confab
 * @subpackage Core\PostTypes
 * @since 2.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Initialize custom post type registration and related functionality.
 *
 * @since 2.0.0
 */
function confab_init_post_types() {
	add_action( 'init', 'confab_register_post_types' );
	add_action( 'gettext', 'confab_change_title_text' );
	add_action( 'dashboard_glance_items', 'confab_cpt_at_glance' );
	add_filter( 'single_template', 'confab_set_single_session_template' );
}

// Initialize post type functionality
confab_init_post_types();

/**
 * Register confab_session custom post type for conference session management.
 *
 * Creates a custom post type with full WordPress feature support including
 * block editor, revisions, thumbnails, and REST API integration.
 *
 * @since 2.0.0
 */
function confab_register_post_types() {

	// Session post type labels for admin interface
	$session_labels = array(
		'name'                  => _x( 'Sessions', 'post type general name', 'confab' ),
		'singular_name'         => _x( 'Session', 'post type singular name', 'confab' ),
		'menu_name'             => _x( 'Sessions', 'admin menu', 'confab' ),
		'name_admin_bar'        => _x( 'Session', 'add new on admin bar', 'confab' ),
		'add_new'               => _x( 'Add New', 'session', 'confab' ),
		'add_new_item'          => __( 'Add New Session', 'confab' ),
		'new_item'              => __( 'New Session', 'confab' ),
		'edit_item'             => __( 'Edit Session', 'confab' ),
		'view_item'             => __( 'View Session', 'confab' ),
		'view_items'            => __( 'View Sessions', 'confab' ),
		'all_items'             => __( 'All Sessions', 'confab' ),
		'search_items'          => __( 'Search Sessions', 'confab' ),
		'parent_item_colon'     => __( 'Parent Sessions:', 'confab' ),
		'not_found'             => __( 'No sessions found.', 'confab' ),
		'not_found_in_trash'    => __( 'No sessions found in Trash.', 'confab' ),
		'featured_image'        => _x( 'Session Featured Image', 'Overrides the "Featured Image" phrase for this post type. Added in 4.3', 'confab' ),
		'set_featured_image'    => _x( 'Set session image', 'Overrides the "Set featured image" phrase for this post type. Added in 4.3', 'confab' ),
		'remove_featured_image' => _x( 'Remove session image', 'Overrides the "Remove featured image" phrase for this post type. Added in 4.3', 'confab' ),
		'use_featured_image'    => _x( 'Use as session image', 'Overrides the "Use as featured image" phrase for this post type. Added in 4.3', 'confab' ),
		'archives'              => _x( 'Session archives', 'The post type archive label used in nav menus. Default "Post Archives". Added in 4.4', 'confab' ),
		'insert_into_item'      => _x( 'Insert into session', 'Overrides the "Insert into post"/"Insert into page" phrase (used when inserting media into a post). Added in 4.4', 'confab' ),
		'uploaded_to_this_item' => _x( 'Uploaded to this session', 'Overrides the "Uploaded to this post"/"Uploaded to this page" phrase (used when viewing media attached to a post). Added in 4.4', 'confab' ),
		'filter_items_list'     => _x( 'Filter sessions list', 'Screen reader text for the filter links heading on the post type listing screen. Default "Filter posts list"/"Filter pages list". Added in 4.4', 'confab' ),
		'items_list_navigation' => _x( 'Sessions list navigation', 'Screen reader text for the pagination heading on the post type listing screen. Default "Posts list navigation"/"Pages list navigation". Added in 4.4', 'confab' ),
		'items_list'            => _x( 'Sessions list', 'Screen reader text for the items list heading on the post type listing screen. Default "Posts list"/"Pages list". Added in 4.4', 'confab' ),
	);

	// Register session post type with full WordPress integration
	register_post_type(
		'confab_session',
		array(
			'labels'              => $session_labels,
			'description'         => __( 'Conference sessions with time scheduling and speaker information.', 'confab' ),
			'public'              => true,
			'publicly_queryable'  => true,
			'show_ui'             => true,
			'show_in_menu'        => true,
			'show_in_admin_bar'   => true,
			'show_in_nav_menus'   => true,
			'can_export'          => true,
			'has_archive'         => true,
			'exclude_from_search' => false,
			'capability_type'     => 'post',
			'hierarchical'        => false,
			'menu_position'       => 21,
			'menu_icon'           => 'dashicons-schedule',
			'supports'            => array(
				'title',
				'editor',
				'author',
				'thumbnail',
				'excerpt',
				'revisions',
				'custom-fields',
			),
			'rewrite'             => array(
				'slug'       => 'sessions',
				'with_front' => false,
			),
			'query_var'           => true,
			'show_in_rest'        => true,
			'rest_base'           => 'sessions',
		)
	);
}

/**
 * Customize post title placeholder text for session post type.
 *
 * Changes the default "Add title" placeholder to more contextual text
 * specifically for conference sessions.
 *
 * @param string $translation Translated text from WordPress
 * @return string Modified translation for session post type
 * @since 2.0.0
 */
function confab_change_title_text( $translation ) {
	global $post;
	
	if ( ! isset( $post ) || ! is_object( $post ) ) {
		return $translation;
	}

	if ( 'confab_session' === $post->post_type && 'Add title' === $translation ) {
		return __( 'Enter Session Title Here', 'confab' );
	}

	return $translation;
}

/**
 * Add custom post type counts to WordPress dashboard At a Glance widget.
 *
 * Displays published session counts alongside default WordPress post counts
 * in the dashboard overview widget.
 *
 * @since 2.0.0
 */
function confab_cpt_at_glance() {
	// Only show to users who can edit posts
	if ( ! current_user_can( 'edit_posts' ) ) {
		return;
	}

	$post_types = get_post_types(
		array(
			'public'   => true,
			'_builtin' => false,
		),
		'objects'
	);

	foreach ( $post_types as $post_type ) {
		// Skip if this isn't our session post type
		if ( 'confab_session' !== $post_type->name ) {
			continue;
		}

		$num_posts = wp_count_posts( $post_type->name );
		$num = number_format_i18n( $num_posts->publish );
		$text = _n( 
			'Session',    // Literal string for singular
			'Sessions',   // Literal string for plural
			intval( $num_posts->publish ),
			'confab'
		);

		$output = sprintf(
			'<a href="%s">%s %s</a>',
			esc_url( admin_url( 'edit.php?post_type=' . $post_type->name ) ),
			esc_html( $num ),
			esc_html( $text )
		);

		printf(
			'<li class="post-count %s-count">%s</li>',
			esc_attr( $post_type->name ),
			wp_kses_post( $output )
		);
	}
}

/**
 * Set custom template for single session post type display.
 *
 * Loads plugin template file for individual session pages if it exists,
 * otherwise falls back to theme's single.php template.
 *
 * @param string $single_template Path to single post template
 * @return string Modified template path for session post type
 * @since 2.0.0
 */
function confab_set_single_session_template( $single_template ) {
	global $post;

	if ( ! isset( $post ) || ! is_object( $post ) ) {
		return $single_template;
	}

	if ( 'confab_session' === $post->post_type ) {
		$custom_template = CONFAB_DIR . 'templates/session-template.php';
		
		if ( file_exists( $custom_template ) ) {
			return $custom_template;
		}
	}

	return $single_template;
}

/**
 * Flush rewrite rules when plugin is activated to register new post type URLs.
 *
 * Called during plugin activation to ensure custom post type permalinks
 * work immediately after plugin activation.
 *
 * @since 2.0.0
 */
function confab_flush_rewrites() {
	confab_register_post_types();
	flush_rewrite_rules();
}

// Register activation hook for rewrite rules
register_activation_hook( CONFAB_FILE, 'confab_flush_rewrites' );
register_deactivation_hook( CONFAB_FILE, 'flush_rewrite_rules' );