<?php

/**
 * ConFab Plugin Bootstrap
 * FILE: confab.php
 *
 * Creates sessions post types for conference websites. Includes shortcode and custom block 
 * for fully mobile-responsive conference schedule in table or grid format.
 *
 * @package confab
 * @since 2.1.0
 *
 * @wordpress-plugin
 * Plugin Name:       ConFab
 * Plugin URI:        https://low.li
 * Description:       Creates sessions post types for conference websites. Includes shortcode and custom block for fully mobile-responsive conference schedule in table or grid format.
 * Version:           2.2.0
 * Author:            LaughterOnWater
 * Author URI:        https://profiles.wordpress.org/laughteronwater/#content-plugins
 * Donate link:       https://shop.low.li/downloads/help-the-legend-sail-on/
 * License:           GPL-3.0+
 * License URI:       http://www.gnu.org/licenses/gpl-3.0.txt
 * Text Domain:       confab
 * Domain Path:       /languages
 * Requires at least: 5.0
 * Tested up to:      6.9
 * Requires PHP:      7.4
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

// Plugin version
define( 'CONFAB_VERSION', '2.2.0' );

// Plugin directory
define( 'CONFAB_DIR', plugin_dir_path( __FILE__ ) );

// Plugin URL
define( 'CONFAB_URL', plugin_dir_url( __FILE__ ) );

// Plugin File
define( 'CONFAB_FILE', __FILE__ );

// Includes
require_once CONFAB_DIR . 'inc/post-types.php';
require_once CONFAB_DIR . 'inc/taxonomies.php';
require_once CONFAB_DIR . 'inc/schedule-output-functions.php';
require_once CONFAB_DIR . 'inc/settings.php';
require_once CONFAB_DIR . 'inc/time-migration.php';

/**
 * ConFab Main Plugin Class
 *
 * Handles plugin initialization, admin interface, block registration,
 * and all core functionality for conference session management.
 *
 * @package confab
 * @since 2.0.0
 */
class Confab_Plugin {

	/**
	 * Plugin instance.
	 *
	 * @var Confab_Plugin
	 * @since 2.0.0
	 */
	private static $instance = null;

	/**
	 * Get singleton instance of confab plugin coordinator.
	 *
	 * @return Confab_Plugin Single plugin instance
	 * @since 2.0.0
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Initialize WordPress conference schedule plugin with hooks and filters.
	 *
	 * Sets up all WordPress hooks for admin interface, frontend display,
	 * block editor integration, and custom post type management.
	 *
	 * @since 2.0.0
	 */
	private function __construct() {
		add_action( 'admin_init', array( $this, 'admin_init' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		
		// Post type specific hooks
		add_action( 'save_post_confab_session', array( $this, 'save_session_post' ), 10, 2 );
		add_action( 'manage_posts_custom_column', array( $this, 'manage_post_columns_output' ), 10, 2 );
		add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
		
		// Block editor and REST API
		add_action( 'init', array( $this, 'register_blocks' ) );
		add_action( 'rest_api_init', array( $this, 'register_rest_endpoints' ) );

		// Hook into admin_menu to reposition after WordPress loads
		add_action( 'add_meta_boxes_confab_session', array( $this, 'reposition_excerpt_metabox' ) );

		// Full Data Deletion Warning on Plugin Page.
		add_action( 'after_plugin_row_confab/confab.php', array( $this, 'plugin_row_deletion_warning' ), 10, 3 );

		
		// Filters
		add_filter( 'manage_confab_session_posts_columns', array( $this, 'manage_post_columns' ) );
		add_filter( 'manage_edit-confab_session_sortable_columns', array( $this, 'manage_sortable_columns' ) );
		add_filter( 'display_post_states', array( $this, 'display_post_states' ) );
		add_filter( 'excerpt_length', array( $this, 'custom_session_excerpt_length' ), 10 );
		add_filter( 'excerpt_more', array( $this, 'custom_session_excerpt_more' ) );
		// Plugin action links in the plugin list page
		add_filter( 'plugin_action_links_confab/confab.php', array( $this, 'add_plugin_action_links' ) );
		add_filter( 'plugin_row_meta', array( $this, 'add_plugin_row_meta' ), 10, 2 );

		// Shortcode
		add_shortcode( 'confab_schedule', array( $this, 'shortcode_schedule' ) );
		
		// Activation/Deactivation hooks
		register_activation_hook( CONFAB_FILE, array( $this, 'activate' ) );
		register_deactivation_hook( CONFAB_FILE, array( $this, 'deactivate' ) );
	}

	/**
	 * Add settings link to plugin action links on plugins page.
	 *
	 * @param array $links Existing plugin action links
	 * @return array Modified plugin action links with settings added
	 * @since 2.0.0
	 */
	public function add_plugin_action_links( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url( admin_url( 'themes.php?page=confab-settings' ) ),
			esc_html__( 'Settings', 'confab' )
		);
		
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Add Donate and Rate links to plugin meta row.
	 *
	 * Adds custom links below the plugin description in the plugins list page,
	 * similar to standard WordPress plugin directory entries.
	 *
	 * @param array  $links Existing meta links
	 * @param string $file  Plugin file path
	 * @return array Modified meta links
	 * @since 2.2.0
	 */
	public function add_plugin_row_meta( $links, $file ) {
		if ( 'confab/confab.php' === $file ) {
			$donate_link = sprintf(
				'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
				esc_url( 'https://shop.low.li/downloads/help-the-legend-sail-on/' ),
				esc_html__( 'Donate', 'confab' )
			);
			
			$rate_link = sprintf(
				'<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
				esc_url( 'https://wordpress.org/plugins/confab/#reviews' ),
				esc_html__( 'Rate Plugin', 'confab' )
			);
			
			$links[] = $donate_link;
			$links[] = $rate_link;
		}
		
		return $links;
	}

	/**
	 * Configure admin area functionality and query modifications.
	 *
	 * @since 2.0.0
	 */
	public function admin_init() {
		add_action( 'pre_get_posts', array( $this, 'admin_pre_get_posts' ) );
	}

	/**
	 * Modify admin post queries for session time-based sorting.
	 *
	 * @param WP_Query $query The WordPress query object to modify
	 * @since 2.0.0
	 */
	public function admin_pre_get_posts( $query ) {
		if ( ! is_admin() || ! $query->is_main_query() ) {
			return;
		}

		$screen = get_current_screen();
		if ( ! $screen ) {
			return;
		}

		// Order by session time
		if ( 'edit-confab_session' === $screen->id && '_confab_session_time' === $query->get( 'orderby' ) ) {
			$query->set( 'meta_key', '_confab_session_time' );
			$query->set( 'orderby', 'meta_value_num' );
		}
	}

	/**
	 * Enqueue admin scripts and styles for session management and settings interface.
	 *
	 * @param string $hook_suffix Current admin page hook suffix
	 * @since 2.0.0
	 */
	public function admin_enqueue_scripts( $hook_suffix ) {
		// Debug versioning - use timestamp when debugging, version when in production
		$version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : CONFAB_VERSION;
		
		// Session post type admin interface
		$screen = get_current_screen();
		if ( $screen && 'confab_session' === $screen->post_type ) {
			// Session management styles
			wp_enqueue_style( 
				'confab-admin', 
				CONFAB_URL . 'assets/css/admin.css', 
				array(), 
				$version 
			);

			// Session admin JavaScript for form enhancements (no longer needs datepicker)
			wp_enqueue_script(
				'confab-admin',
				CONFAB_URL . 'assets/js/confab-admin.js',
				array( 'jquery' ),
				$version,
				true
			);
		}
		
		// Settings page modern interface
		if ( 'appearance_page_confab-settings' === $hook_suffix ) {
			wp_enqueue_style(
				'confab-admin-modern',
				CONFAB_URL . 'assets/css/confab-admin-modern.css',
				array(),
				$version
			);
			
			wp_enqueue_script(
				'confab-admin-settings',
				CONFAB_URL . 'assets/js/confab-admin-settings.js',
				array( 'jquery' ),
				$version,
				true
			);
		}

		// Minimal admin notice CSS - All other admin pages when review should show (except our own settings page)
		$screen = get_current_screen();
		if ( is_admin() && confab_should_show_review_request() && $screen && 'appearance_page_confab-settings' !== $screen->id ) {
			wp_enqueue_style(
				'confab-admin-notices',
				CONFAB_URL . 'assets/css/confab-admin-notices.css',
				array(),
				$version
			);
		}
	}

	/**
	 * Enqueue frontend styles and scripts for schedule display.
	 *
	 * @since 2.0.0
	 */
	public function enqueue_scripts() {
		// Debug versioning - use timestamp when debugging, version when in production
		$version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : CONFAB_VERSION;

		wp_enqueue_style( 
			'confab-styles', 
			CONFAB_URL . 'assets/css/style.css', 
			array(), 
			$version 
		);

		wp_enqueue_script(
			'confab-tabs',
			CONFAB_URL . 'assets/js/confab-tabs.js',
			array(), // No dependencies
			$version,
			true // Load in footer
		);
	}


	/**
	 * Register Gutenberg block types for schedule display.
	 *
	 * @since 2.0.0
	 */
	public function register_blocks() {
		if ( ! function_exists( 'register_block_type' ) ) {
			return;
		}

		// Prevent double registration
		if ( WP_Block_Type_Registry::get_instance()->is_registered( 'confab/schedule-block' ) ) {
			return;
		}

		// Register the schedule block with EXACT same config as JavaScript
		register_block_type( 'confab/schedule-block', array(
			'title'       => __( 'Conference Schedule (Confab)', 'confab' ),
			'description' => __( 'Display conference sessions in table or grid format', 'confab' ),
			'icon'        => 'schedule',
			'category'    => 'common',
			'keywords'    => array( 'conference', 'schedule', 'session', 'event' ),
			'supports'    => array(
				'align' => array( 'wide', 'full' ),
			),
			'attributes' => array(
				'date' => array(
					'type'    => 'string',
					'default' => gmdate( 'Y-m-d' ), // Match JS default
				),
				'color_scheme' => array(
					'type'    => 'string',
					'default' => 'light',
				),
				'layout' => array(
					'type'    => 'string',
					'default' => 'table',
				),
				'row_height' => array(
					'type'    => 'string',
					'default' => 'match', // Match JS default
				),
				'session_link' => array(
					'type'    => 'string',
					'default' => 'permalink', // Match JS default
				),
				'tracks' => array(
					'type'    => 'string',
					'default' => '', // Match JS (empty string not null)
				),
				'align' => array(
					'type'    => 'string',
					'default' => '',
				),
				'content' => array(
					'type'    => 'string',
					'default' => 'none',
				),
			),
			'render_callback' => array( $this, 'render_schedule_block' ),
			'editor_script'   => 'confab-schedule-block', // Link to our JS
		) );

		// Enqueue block editor assets
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
	}

	/**
	 * Enqueue block editor JavaScript and CSS assets.
	 *
	 * @since 2.0.0
	 */
	public function enqueue_block_editor_assets() {
		// Debug versioning - use timestamp when debugging, version when in production
		$version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : CONFAB_VERSION;

		// Try minified version first, fallback to development version
		$script_file = 'confab-block-hybrid.min.js';
		if ( ! file_exists( CONFAB_DIR . 'assets/js/' . $script_file ) ) {
			$script_file = 'confab-block-hybrid.js';
		}
		
		wp_enqueue_script(
			'confab-schedule-block',
			CONFAB_URL . 'assets/js/' . $script_file,
			array( 'wp-blocks', 'wp-i18n', 'wp-editor', 'wp-components', 'wp-element', 'wp-api-fetch' ),
			$version,
			true
		);

		// Enqueue frontend CSS in editor for accurate preview styling
		wp_enqueue_style( 
			'confab-editor-styles', 
			CONFAB_URL . 'assets/css/style.css', 
			array(), 
			$version 
		);

		wp_set_script_translations( 'confab-schedule-block', 'confab' );
	}

	/**
	 * Register REST API endpoints for block editor preview.
	 *
	 * @since 2.0.0
	 */
	public function register_rest_endpoints() {
		register_rest_route( 'wp/v2', '/confab-preview', array(
			'methods'             => 'POST',
			'callback'            => array( $this, 'rest_preview_callback' ),
			'permission_callback' => array( $this, 'rest_preview_permissions' ),
			'args'                => array(
				'date' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => '',
				),
				'tracks' => array(
					'type'              => 'string', 
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => '',
				),
				'color_scheme' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => 'light',
					'enum' => array( 
						'light', 'dark', 
						'earthtones', 'earthtones-dark', 
						'jeweltones', 'jeweltones-dark', 
						'pastels', 'pastels-dark', 
						'ocean', 'ocean-dark', 
						'forest', 'forest-dark',
						'autumn', 'autumn-dark',
						'terracotta', 'terracotta-dark',
						'burgundy', 'burgundy-dark',
						'honey', 'honey-dark',
						'sunrise', 'sunrise-dark',
						'lagoon', 'lagoon-dark',
						'sanguine', 'sanguine-dark'
					),
				),
				'table_style' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => 'bordered',
					'enum'              => array( 'bordered', 'borderless', 'rounded' ),
				),
				'layout' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field', 
					'default'           => 'table',
					'enum'              => array( 'table', 'grid' ),
				),
				'row_height' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => 'match',
					'enum'              => array( 'match', 'auto' ),
				),
				'session_link' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => 'permalink', 
					'enum'              => array( 'permalink', 'anchor', 'none' ),
				),
				'content' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => 'none',
					'enum'              => array( 'none', 'excerpt', 'full' ),
				),
				'align' => array(
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'default'           => '',
				),
			),
		) );
	}

	/**
	 * Check permissions for preview REST endpoint.
	 *
	 * @return bool Whether user can access preview endpoint
	 * @since 2.0.0
	 */
	public function rest_preview_permissions() {
		return current_user_can( 'edit_posts' );
	}

	/**
	 * Handle REST API preview requests using proven confab_scheduleOutput().
	 *
	 * @param WP_REST_Request $request REST request object
	 * @return WP_REST_Response|WP_Error Response object or error
	 * @since 2.0.0
	 */
	public function rest_preview_callback( $request ) {
		// Get sanitized parameters (already sanitized by REST API)
		$attributes = array(
			'date'         => $request->get_param( 'date' ),
			'tracks'       => $request->get_param( 'tracks' ), 
			'color_scheme' => $request->get_param( 'color_scheme' ),
			'table_style'  => $request->get_param( 'table_style' ),  // ADD THIS LINE
			'layout'       => $request->get_param( 'layout' ),
			'row_height'   => $request->get_param( 'row_height' ),
			'session_link' => $request->get_param( 'session_link' ),
			'content'      => $request->get_param( 'content' ),
			'align'        => $request->get_param( 'align' ),
		);

		// Use the proven confab_scheduleOutput function
		if ( function_exists( 'confab_scheduleOutput' ) ) {
			$html = confab_scheduleOutput( $attributes );

			return rest_ensure_response( array(
				'success' => true,
				'html'    => $html,
			) );
		}

		return new WP_Error( 
			'function_not_found',
			__( 'Schedule output function not available.', 'confab' ),
			array( 'status' => 500 )
		);
	}

	/**
	 * Process confab_schedule shortcode with sanitized attributes.
	 *
	 * @param array $atts Shortcode attributes from WordPress
	 * @return string Rendered schedule HTML output
	 * @since 2.0.0
	 */
	public function shortcode_schedule( $atts ) {
		$atts = shortcode_atts( array(
			'date' => '',
			'color_scheme' => 'light',
			'table_style' => 'bordered',  // ADD THIS LINE
			'layout' => 'table',
			'row_height' => 'normal',
			'session_link' => 'page',
			'tracks' => '',
			'content' => 'none',
		), $atts, 'confab_schedule' );

		return $this->render_schedule_block( $atts );
	}

	/**
	 * Render schedule display HTML for both blocks and shortcodes.
	 *
	 * @param array $attributes Block or shortcode attributes
	 * @return string Sanitized HTML output for schedule display
	 * @since 2.0.0
	 */
	public function render_schedule_block( $attributes ) {
		// Sanitize attributes
		$attributes = array_map( 'sanitize_text_field', $attributes );
		
		if ( function_exists( 'confab_scheduleOutput' ) ) {
			return confab_scheduleOutput( $attributes );
		}
		
		return '<p>' . esc_html__( 'Schedule output function not found.', 'confab' ) . '</p>';
	}

	/**
	 * Register WordPress meta boxes for session post type.
	 *
	 * @since 2.0.0
	 */
	public function add_meta_boxes() {
		add_meta_box(
			'confab-session-info',
			esc_html__( 'Session Info', 'confab' ),
			array( $this, 'render_session_info_metabox' ),
			'confab_session',
			'normal'
		);
	}

	/**
	 * Render session information metabox with native HTML5 date controls and speaker field.
	 *
	 * @param WP_Post $post Current WordPress post object being edited
	 * @since 2.0.0
	 */
	public function render_session_info_metabox( $post ) {
		if ( ! $post ) {
			return;
		}

		wp_nonce_field( 'confab_save_session_info', 'confab_session_info_nonce' );

	// Get session timestamps (stored as UTC)
	// wp_date() converts UTC to WordPress timezone for display
	$session_time     = absint( confab_get_session_meta_compat( $post->ID, '_confab_session_time' ) );
	$session_date     = $session_time ? wp_date( 'Y-m-d', $session_time ) : wp_date( 'Y-m-d' );
	$session_hours    = $session_time ? wp_date( 'g', $session_time ) : wp_date( 'g' );
	$session_minutes  = $session_time ? wp_date( 'i', $session_time ) : '00';
	$session_meridiem = $session_time ? wp_date( 'a', $session_time ) : 'am';
	$session_type     = confab_get_session_meta_compat( $post->ID, '_confab_session_type' );

	$session_end_time     = absint( confab_get_session_meta_compat( $post->ID, '_confab_session_end_time' ) );
	$session_end_hours    = $session_end_time ? wp_date( 'g', $session_end_time ) : wp_date( 'g' );
	$session_end_minutes  = $session_end_time ? wp_date( 'i', $session_end_time ) : '00';
	$session_end_meridiem = $session_end_time ? wp_date( 'a', $session_end_time ) : 'am';

		// Get speaker names
		$session_speakers = confab_get_session_meta_compat( $post->ID, '_confab_session_speakers' );

		?>
		<table class="form-table">
			<tr>
				<th scope="row">
					<label for="confab-session-speakers"><?php esc_html_e( 'Speaker Name(s):', 'confab' ); ?></label>
				</th>
				<td>
					<input type="text" 
						id="confab-session-speakers" 
						name="confab_session_speakers" 
						value="<?php echo esc_attr( $session_speakers ); ?>" 
						class="regular-text" 
						placeholder="<?php esc_attr_e( 'John Doe, Jane Smith', 'confab' ); ?>"
						aria-describedby="confab-speakers-description" />
					<p id="confab-speakers-description" class="description">
						<?php esc_html_e( 'Enter speaker names separated by commas for multiple speakers.', 'confab' ); ?>
					</p>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="confab-session-date"><?php esc_html_e( 'Date:', 'confab' ); ?></label>
				</th>
				<td>
					<input type="date" 
						id="confab-session-date" 
						name="confab_session_date" 
						value="<?php echo esc_attr( $session_date ); ?>" 
						class="regular-text" 
						aria-describedby="confab-date-description" />
					<p id="confab-date-description" class="description">
						<?php 
						esc_html_e( 'Select the session date using the date picker.', 'confab' ); 
						echo ' ';?><br><?php
						printf(
							/* translators: %s: site timezone string */
							esc_html__( 'Times display in your site timezone (%s).', 'confab' ),
							esc_html( wp_timezone_string() )
						);
						?>
					</p>
				</td>
			</tr>
			<tr>
				<th scope="row"><?php esc_html_e( 'Start Time:', 'confab' ); ?></th>
				<td>
					<select name="confab_session_hour" aria-label="<?php esc_attr_e( 'Session Start Hour', 'confab' ); ?>">
						<?php for ( $i = 1; $i <= 12; $i++ ) : ?>
							<option value="<?php echo esc_attr( $i ); ?>" <?php selected( $i, $session_hours ); ?>>
								<?php echo esc_html( $i ); ?>
							</option>
						<?php endfor; ?>
					</select>
					:
					<select name="confab_session_minutes" aria-label="<?php esc_attr_e( 'Session Start Minutes', 'confab' ); ?>">
						<?php for ( $i = 0; $i <= 55; $i += 5 ) : ?>
							<?php $minute_val = sprintf( '%02d', $i ); ?>
							<option value="<?php echo esc_attr( $minute_val ); ?>" <?php selected( $minute_val, $session_minutes ); ?>>
								<?php echo esc_html( $minute_val ); ?>
							</option>
						<?php endfor; ?>
					</select>
					<select name="confab_session_meridiem" aria-label="<?php esc_attr_e( 'Session Start Meridiem', 'confab' ); ?>">
						<option value="am" <?php selected( 'am', $session_meridiem ); ?>>AM</option>
						<option value="pm" <?php selected( 'pm', $session_meridiem ); ?>>PM</option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row"><?php esc_html_e( 'End Time:', 'confab' ); ?></th>
				<td>
					<select name="confab_session_end_hour" aria-label="<?php esc_attr_e( 'Session End Hour', 'confab' ); ?>">
						<?php for ( $i = 1; $i <= 12; $i++ ) : ?>
							<option value="<?php echo esc_attr( $i ); ?>" <?php selected( $i, $session_end_hours ); ?>>
								<?php echo esc_html( $i ); ?>
							</option>
						<?php endfor; ?>
					</select>
					:
					<select name="confab_session_end_minutes" aria-label="<?php esc_attr_e( 'Session End Minutes', 'confab' ); ?>">
						<?php for ( $i = 0; $i <= 55; $i += 5 ) : ?>
							<?php $minute_val = sprintf( '%02d', $i ); ?>
							<option value="<?php echo esc_attr( $minute_val ); ?>" <?php selected( $minute_val, $session_end_minutes ); ?>>
								<?php echo esc_html( $minute_val ); ?>
							</option>
						<?php endfor; ?>
					</select>
					<select name="confab_session_end_meridiem" aria-label="<?php esc_attr_e( 'Session End Meridiem', 'confab' ); ?>">
						<option value="am" <?php selected( 'am', $session_end_meridiem ); ?>>AM</option>
						<option value="pm" <?php selected( 'pm', $session_end_meridiem ); ?>>PM</option>
					</select>
				</td>
			</tr>
			<tr>
				<th scope="row">
					<label for="confab-session-type"><?php esc_html_e( 'Display Style:', 'confab' ); ?></label>
				</th>
				<td>
					<select id="confab-session-type" name="confab_session_type">
						<option value="normal" <?php selected( $session_type, 'normal' ); ?>>
							<?php esc_html_e( 'Normal', 'confab' ); ?>
						</option>
						<option value="featured" <?php selected( $session_type, 'featured' ); ?>>
							<?php esc_html_e( 'Featured', 'confab' ); ?>
						</option>
						<option value="alternate" <?php selected( $session_type, 'alternate' ); ?>>
							<?php esc_html_e( 'Alternate', 'confab' ); ?>
						</option>
					</select>
					<p class="description">
						<?php esc_html_e( 'Featured and Alternate sessions display with distinctive background colors.', 'confab' ); ?>
					</p>
				</td>
			</tr>
		</table>
		<?php
	}

	/**
	 * Reposition excerpt metabox for better visibility in session editing.
	 *
	 * Moves the WordPress excerpt metabox to 'normal' context with high priority
	 * so it appears prominently below the content editor instead of hidden in sidebar.
	 *
	 * @since 2.0.0
	 */
	public function reposition_excerpt_metabox() {
		// Only on our session post type
		global $post;
		if ( ! $post || 'confab_session' !== $post->post_type ) {
			return;
		}
		
		// Remove excerpt metabox from default sidebar position
		remove_meta_box( 'postexcerpt', 'confab_session', 'normal' );
		
		// Add it back in prominent position with custom title
		add_meta_box(
			'confab-session-excerpt',
			__( 'Session Description (Brief Summary)', 'confab' ),
			array( $this, 'excerpt_metabox_callback' ),  // FIXED: Use array callback
			'confab_session',
			'normal',          // Main content area
			'high'             // High priority (appears early)
		);
	}

	/**
	 * Render prominent excerpt metabox with better labeling and instructions.
	 *
	 * @param WP_Post $post Current post object
	 * @since 2.0.0
	 */
	public function excerpt_metabox_callback( $post ) {
		?>
		<div class="excerpt-help-text" style="margin-bottom: 10px;">
			<p class="description">
				<?php esc_html_e( 'Write a brief description that appears in schedule listings. This should be less than 150 characters.', 'confab' ); ?>
			</p>
		</div>
		
		<textarea 
			id="excerpt" 
			name="excerpt" 
			rows="4" 
			style="width: 100%;" 
			placeholder="<?php esc_attr_e( 'Brief session description for schedule displays...', 'confab' ); ?>"
		><?php echo esc_textarea( $post->post_excerpt ); ?></textarea>
		
		<div class="excerpt-additional-help" style="margin-top: 8px;">
			<p class="description">
				<?php esc_html_e( 'Leave empty to auto-generate from your full content. Keep it under 150 characters for best display.', 'confab' ); ?>
			</p>
		</div>
		<?php
	}

	/**
	 * Customize excerpt length for session post type to approximately 150 characters.
	 *
	 * WordPress excerpt_length filter expects word count, so we estimate
	 * based on average word length to achieve ~150 character limit.
	 *
	 * @param int $length Current excerpt word length
	 * @return int Modified word length for session post type
	 * @since 2.0.0
	 */
	public function custom_session_excerpt_length( $length ) {
		// Only apply to our session post type in admin
		if ( is_admin() && function_exists( 'get_current_screen' ) ) {
			$screen = get_current_screen();
			if ( $screen && 'confab_session' === $screen->post_type ) {
				// Average word length ~5 characters + 1 space = 6 chars per word
				// 150 characters ÷ 6 = ~17 words
				return 17;
			}
		}
		
		// On frontend, check if we're displaying a session
		global $post;
		if ( $post && 'confab_session' === $post->post_type ) {
			return 17;
		}
		
		return $length;
	}

	/**
	 * Customize excerpt "more" text for session post type.
	 *
	 * @param string $more Current "more" text
	 * @return string Modified "more" text
	 * @since 2.0.0
	 */
	public function custom_session_excerpt_more( $more ) {
		global $post;
		if ( $post && 'confab_session' === $post->post_type ) {
			return '...';
		}
		return $more;
	}

	/**
	 * Save session post metadata with Plugin Check compliant input validation.
	 *
	 * @param int     $post_id WordPress post ID being saved
	 * @param WP_Post $post    WordPress post object being saved
	 * @since 2.0.0
	 */
	public function save_session_post( $post_id, $post ) {
		// Verify nonce with Plugin Check compliant pattern
		if ( ! isset( $_POST['confab_session_info_nonce'] ) || 
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['confab_session_info_nonce'] ) ), 'confab_save_session_info' ) ) {
			return;
		}

		// Check user capabilities
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		// Avoid infinite loops
		if ( wp_is_post_revision( $post_id ) || wp_is_post_autosave( $post_id ) ) {
			return;
		}

		// Save speaker names with Plugin Check compliant validation
		if ( isset( $_POST['confab_session_speakers'] ) ) {
			$speakers = sanitize_text_field( wp_unslash( $_POST['confab_session_speakers'] ) );
			update_post_meta( $post_id, '_confab_session_speakers', $speakers );
		}

		// Save session date and time with Plugin Check compliant validation
		$session_date = isset( $_POST['confab_session_date'] ) ? sanitize_text_field( wp_unslash( $_POST['confab_session_date'] ) ) : '';
		$session_hour = isset( $_POST['confab_session_hour'] ) ? absint( wp_unslash( $_POST['confab_session_hour'] ) ) : 0;
		$session_minutes = isset( $_POST['confab_session_minutes'] ) ? absint( wp_unslash( $_POST['confab_session_minutes'] ) ) : 0;
		$session_meridiem = isset( $_POST['confab_session_meridiem'] ) ? sanitize_text_field( wp_unslash( $_POST['confab_session_meridiem'] ) ) : 'am';

		if ( $session_date && $session_hour ) {
			// Build time string and parse in WordPress timezone, store as UTC
			$time_string = sprintf(
				'%s %d:%02d %s',
				$session_date,
				$session_hour,
				$session_minutes,
				'pm' === strtolower( $session_meridiem ) ? 'pm' : 'am'
			);
			
			// Create DateTime in WordPress timezone
			$datetime = DateTime::createFromFormat( 'Y-m-d g:i a', $time_string, wp_timezone() );
			
			if ( $datetime ) {
				// Store as UTC timestamp
				$session_time = $datetime->getTimestamp();
				update_post_meta( $post_id, '_confab_session_time', $session_time );
				// Track usage for review system
				$this->increment_usage_stats( 'sessions_created' );
			}
		}

		// Save session end time with Plugin Check compliant validation
		$session_end_hour = isset( $_POST['confab_session_end_hour'] ) ? absint( wp_unslash( $_POST['confab_session_end_hour'] ) ) : 0;
		$session_end_minutes = isset( $_POST['confab_session_end_minutes'] ) ? absint( wp_unslash( $_POST['confab_session_end_minutes'] ) ) : 0;
		$session_end_meridiem = isset( $_POST['confab_session_end_meridiem'] ) ? sanitize_text_field( wp_unslash( $_POST['confab_session_end_meridiem'] ) ) : 'am';

		if ( $session_date && $session_end_hour ) {
			// Build time string and parse in WordPress timezone, store as UTC
			$end_time_string = sprintf(
				'%s %d:%02d %s',
				$session_date,
				$session_end_hour,
				$session_end_minutes,
				'pm' === strtolower( $session_end_meridiem ) ? 'pm' : 'am'
			);
			
			// Create DateTime in WordPress timezone
			$datetime_end = DateTime::createFromFormat( 'Y-m-d g:i a', $end_time_string, wp_timezone() );
			
			if ( $datetime_end ) {
				// Store as UTC timestamp
				$session_end_time = $datetime_end->getTimestamp();
				update_post_meta( $post_id, '_confab_session_end_time', $session_end_time );
			}
		}

		// Save session type with Plugin Check compliant validation
		if ( isset( $_POST['confab_session_type'] ) ) {
			$session_type = sanitize_text_field( wp_unslash( $_POST['confab_session_type'] ) );
			$allowed_types = array( 'normal', 'featured', 'alternate' );
			
			if ( in_array( $session_type, $allowed_types, true ) ) {
				update_post_meta( $post_id, '_confab_session_type', $session_type );
			}
		}
	}

	/**
	 * Add time column to session posts admin listing.
	 *
	 * @param array $columns Existing WordPress admin columns
	 * @return array Modified columns array with time column added
	 * @since 2.0.0
	 */
	public function manage_post_columns( $columns ) {
		$new_columns = array();
		
		foreach ( $columns as $key => $value ) {
			$new_columns[ $key ] = $value;
			
			// Add time column after title
			if ( 'title' === $key ) {
				$new_columns['confab_session_time'] = esc_html__( 'Time', 'confab' );
			}
		}
		
		return $new_columns;
	}

	/**
	 * Display session time data in admin column output.
	 *
	 * @param string $column  WordPress admin column identifier
	 * @param int    $post_id WordPress post ID for current row
	 * @since 2.0.0
	 */
	public function manage_post_columns_output( $column, $post_id ) {
		if ( 'confab_session_time' === $column ) {
		// Use compatibility function for backward compatibility
		$session_time = absint( confab_get_session_meta_compat( $post_id, '_confab_session_time' ) );
		
		if ( $session_time ) {
			echo esc_html( wp_date( get_option( 'time_format' ), $session_time ) );
		} else {
			echo '&mdash;';
		}
		}
	}

	/**
	 * Configure session time column as sortable in admin interface.
	 *
	 * @param array $sortable Existing sortable columns array
	 * @return array Modified sortable columns with session time added
	 * @since 2.0.0
	 */
	public function manage_sortable_columns( $sortable ) {
		$sortable['confab_session_time'] = '_confab_session_time';
		return $sortable;
	}

	/**
	 * Add session type indicators to post states in admin listing.
	 *
	 * @param array $states Existing WordPress post states
	 * @return array Modified states array with session type labels
	 * @since 2.1.0
	 */
	public function display_post_states( $states ) {
		$post = get_post();

		if ( ! $post || 'confab_session' !== $post->post_type ) {
			return $states;
		}

		// Use compatibility function for backward compatibility
		$session_type = confab_get_session_meta_compat( $post->ID, '_confab_session_type' );
		
		switch ( $session_type ) {
			case 'featured':
				$states['confab-session-type'] = esc_html__( 'Featured', 'confab' );
				break;
			case 'alternate':
				$states['confab-session-type'] = esc_html__( 'Alternate', 'confab' );
				break;
			default:
				$states['confab-session-type'] = esc_html__( 'Normal', 'confab' );
				break;
		}

		return $states;
	}

	/**
	 * Migrate legacy WP Conference Schedule data to ConFab format.
	 *
	 * @since 2.0.0
	 */
	private function migrate_legacy_data() {
		global $wpdb;
		
		// Check if old data still exists (regardless of migration flag)
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Migration detection query
		$old_taxonomies_exist = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->term_taxonomy} 
			WHERE taxonomy IN ('wpcs_track', 'wpcs_location')"
		);
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Migration detection query
		$old_posts_exist = $wpdb->get_var(
			"SELECT COUNT(*) FROM {$wpdb->posts} 
			WHERE post_type = 'wpcs_session'"
		);
		
		// If no old data exists and migration was already marked complete, skip
		if ( ! $old_taxonomies_exist && ! $old_posts_exist && get_option( 'confab_migration_complete' ) ) {
			return;
		}

		// Migrate taxonomies first (tracks, locations)
		if ( $old_taxonomies_exist ) {
			$this->migrate_legacy_taxonomies();
		}
		
		// Migrate sessions from old post type
		if ( $old_posts_exist ) {
			$this->migrate_legacy_sessions();
		}
		
		// Migrate settings (always safe to run)
		$this->migrate_legacy_settings();
		
		// Mark migration as complete
		update_option( 'confab_migration_complete', time() );
	}

	/**
	 * Migrate legacy taxonomies by renaming them in the database.
	 *
	 * Updates taxonomy names directly in wp_term_taxonomy table to preserve
	 * all existing terms, relationships, and hierarchy without duplication.
	 *
	 * @since 2.0.0
	 */
	private function migrate_legacy_taxonomies() {
		global $wpdb;
		
		// Rename wpcs_track to confab_track
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration operation
		$wpdb->update(
			$wpdb->term_taxonomy,
			array( 'taxonomy' => 'confab_track' ),
			array( 'taxonomy' => 'wpcs_track' ),
			array( '%s' ),
			array( '%s' )
		);
		
		// Rename wpcs_location to confab_location
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration operation
		$wpdb->update(
			$wpdb->term_taxonomy,
			array( 'taxonomy' => 'confab_location' ),
			array( 'taxonomy' => 'wpcs_location' ),
			array( '%s' ),
			array( '%s' )
		);
	}

	/**
	 * Migrate legacy session posts and meta data.
	 *
	 * @since 2.0.0
	 */
	private function migrate_legacy_sessions() {
		// Find sessions with old post type or old meta keys
		$sessions_query = new WP_Query( array(
			'post_type' => array( 'confab_session', 'wpcs_session' ),
			'posts_per_page' => -1,
			'post_status' => 'any',
			'meta_query' => array(
				'relation' => 'OR',
				array(
					'key' => '_wpcs_session_time',
					'compare' => 'EXISTS',
				),
				array(
					'key' => '_confab_session_time',
					'compare' => 'EXISTS',
				),
			),
		) );

		if ( ! $sessions_query->have_posts() ) {
			return;
		}

		foreach ( $sessions_query->posts as $session ) {
			// Update post type if needed
			if ( 'wpcs_session' === $session->post_type ) {
				wp_update_post( array(
					'ID' => $session->ID,
					'post_type' => 'confab_session',
				) );
			}

			// Migrate meta fields
			$this->migrate_session_meta( $session->ID );
			
			// Migrate taxonomy terms
			$this->migrate_session_taxonomies( $session->ID );
		}
	}

	/**
	 * Migrate session meta fields from wpcs_ to confab_ prefixes with timezone conversion.
	 *
	 * Converts WPCS timestamps (stored in server timezone) to UTC timestamps for ConFab.
	 *
	 * @param int $session_id Session post ID
	 * @since 2.2.0
	 */
	private function migrate_session_meta( $session_id ) {
		// Non-timestamp fields: simple copy
		$simple_fields = array(
			'_wpcs_session_type' => '_confab_session_type',
			'_wpcs_session_speakers' => '_confab_session_speakers',
		);

		foreach ( $simple_fields as $old_key => $new_key ) {
			$old_value = get_post_meta( $session_id, $old_key, true );
			
			if ( $old_value && ! get_post_meta( $session_id, $new_key, true ) ) {
				update_post_meta( $session_id, $new_key, $old_value );
			}
		}
		
		// Timestamp fields: convert from server timezone to UTC
		$timestamp_fields = array(
			'_wpcs_session_time' => '_confab_session_time',
			'_wpcs_session_end_time' => '_confab_session_end_time',
		);

		foreach ( $timestamp_fields as $old_key => $new_key ) {
			$old_timestamp = get_post_meta( $session_id, $old_key, true );
			
			if ( $old_timestamp && ! get_post_meta( $session_id, $new_key, true ) ) {
				// Convert WPCS timestamp (server timezone) to UTC
				$utc_timestamp = $this->convert_wpcs_timestamp_to_utc( $old_timestamp );
				update_post_meta( $session_id, $new_key, $utc_timestamp );
			}
		}
		
		// Keep old meta for safety during transition
	}

	/**
	 * Convert WPCS timestamp from server timezone to UTC.
	 *
	 * WPCS stored timestamps using strtotime() which interprets times in server timezone.
	 * We need to interpret the timestamp correctly and convert to UTC for ConFab.
	 *
	 * @param int $wpcs_timestamp The timestamp as stored by WPCS
	 * @return int UTC timestamp
	 * @since 2.2.0
	 */
	private function convert_wpcs_timestamp_to_utc( $wpcs_timestamp ) {
		// WPCS stored timestamp using strtotime() which interprets the time
		// in the PHP server timezone context (often UTC on production servers).
		// 
		// Example scenario (your case):
		// - Server timezone: UTC
		// - WordPress timezone: America/Los_Angeles (UTC-8)
		// - User entered "7:45 AM" intending Pacific time
		// - WPCS used strtotime('2025-10-22 7:45 am') = 7:45 AM UTC
		// - But user MEANT 7:45 AM Pacific = 3:45 PM UTC (8 hours later)
		// 
		// Solution: Treat the stored timestamp as if it's in WordPress timezone,
		// then convert to proper UTC.
		
		// Get the date/time string from the WPCS timestamp
		$datetime_string = gmdate( 'Y-m-d H:i:s', $wpcs_timestamp );
		
		// Parse this datetime as if it's in WordPress timezone
		$dt = DateTime::createFromFormat( 'Y-m-d H:i:s', $datetime_string, wp_timezone() );
		
		if ( $dt ) {
			// Return the correct UTC timestamp
			return $dt->getTimestamp();
		}
		
		// Fallback: return original if conversion fails
		return $wpcs_timestamp;
	}

	/**
	 * Migrate session taxonomy terms from wpcs_ to confab_ taxonomies.
	 *
	 * This method is intentionally empty because taxonomy migration happens
	 * at the database level in migrate_legacy_taxonomies() during activation.
	 * Individual session taxonomy assignments are automatically preserved.
	 *
	 * @param int $session_id Session post ID
	 * @since 2.0.0
	 */
	private function migrate_session_taxonomies( $session_id ) {
		// Taxonomies are migrated globally in migrate_legacy_taxonomies()
		// No per-session action needed
	}

	/**
	 * Migrate legacy plugin settings to new option names.
	 *
	 * @since 2.0.0
	 */
	private function migrate_legacy_settings() {
		// Migrate byline setting
		$old_byline = get_option( 'wpcs_field_byline' );
		if ( $old_byline && ! get_option( 'confab_field_byline' ) ) {
			update_option( 'confab_field_byline', $old_byline );
		}

		// Migrate schedule page URL
		$old_schedule_url = get_option( 'wpcs_field_schedule_page_url' );
		if ( $old_schedule_url && ! get_option( 'confab_field_schedule_page_url' ) ) {
			update_option( 'confab_field_schedule_page_url', $old_schedule_url );
		}
	}

		/**
	 * Migrate legacy session type values to new semantic naming.
	 *
	 * Converts old values (session/mainstage/custom) to new values (normal/featured/alternate)
	 * for better semantic clarity and user understanding.
	 *
	 * @since 2.0.0
	 */
	private function migrate_session_types() {
		// Check if migration already completed
		if ( get_option( 'confab_session_types_migrated_v2' ) ) {
			return;
		}
		
		global $wpdb;
		
		// Direct database update for efficiency
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration operation
		$wpdb->query( 
			$wpdb->prepare(
				"UPDATE {$wpdb->postmeta} 
				SET meta_value = %s 
				WHERE meta_key = %s 
				AND meta_value = %s",
				'normal',
				'_confab_session_type',
				'session'
			)
		);
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration operation
		$wpdb->query(
			$wpdb->prepare(
				"UPDATE {$wpdb->postmeta} 
				SET meta_value = %s 
				WHERE meta_key = %s 
				AND meta_value = %s",
				'featured',
				'_confab_session_type',
				'mainstage'
			)
		);
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- One-time migration operation
		$wpdb->query(
			$wpdb->prepare(
				"UPDATE {$wpdb->postmeta} 
				SET meta_value = %s 
				WHERE meta_key = %s 
				AND meta_value = %s",
				'alternate',
				'_confab_session_type',
				'custom'
			)
		);
		
		// Mark migration as complete
		update_option( 'confab_session_types_migrated_v2', time() );
	}

	/**
	 * Initialize plugin default settings and migrate legacy data on activation.
	 *
	 * @since 2.0.0
	 */
	public function activate() {
		// Flush rewrite rules
		flush_rewrite_rules();

		// Track installation date
		if ( ! get_option( 'confab_install_date' ) ) {
			add_option( 'confab_install_date', time() );
		}
		
		// Initialize individual settings (needed for Settings API to work)
		// Check legacy array first for migration
		$legacy_settings = get_option( 'confab_settings', array() );
		
		if ( false === get_option( 'confab_field_byline' ) ) {
			$default = isset( $legacy_settings['show_credit_link'] ) ? $legacy_settings['show_credit_link'] : true;
			add_option( 'confab_field_byline', $default );
		}
		
		if ( false === get_option( 'confab_field_schedule_page_url' ) ) {
			$default = isset( $legacy_settings['schedule_page_url'] ) ? $legacy_settings['schedule_page_url'] : '';
			add_option( 'confab_field_schedule_page_url', $default );
		}
		
		if ( false === get_option( 'confab_excerpt_length' ) ) {
			add_option( 'confab_excerpt_length', 150 );
		}
		
		if ( false === get_option( 'confab_delete_initial_confirmed' ) ) {
			add_option( 'confab_delete_initial_confirmed', false );
		}
		
		if ( false === get_option( 'confab_delete_final_confirmed' ) ) {
			add_option( 'confab_delete_final_confirmed', false );
		}
		
		// Leave legacy array intact for reference, but clear temp submission
		delete_option( 'confab_temp_submission' );
		
		// Register taxonomies before migration (they're normally registered on 'init')
		if ( function_exists( 'confab_register_taxonomies' ) ) {
			confab_register_taxonomies();
		}
		
		// Run migrations
		$this->migrate_legacy_data();
		$this->migrate_session_types();
	}

	/**
	 * Clean up rewrite rules on plugin deactivation.
	 *
	 * @since 2.0.0
	 */
	public function deactivate() {
		// Flush rewrite rules
		flush_rewrite_rules();
	}

	/**
	 * Increment ConFab usage statistics for review system threshold tracking.
	 *
	 * Tracks plugin usage events to determine appropriate timing for review requests
	 * based on demonstrated value to users through actual plugin utilization.
	 *
	 * @param string $stat_type Type of usage event to increment
	 * @since 2.0.0
	 */
	public function increment_usage_stats( $stat_type ) { // <-- line 961
		$stats = get_option( 'confab_usage_stats', array(
			'sessions_created' => 0,
			'schedules_displayed' => 0,
			'blocks_used' => 0,
			'tracks_filtered' => 0,
		) );
		
		// Validate stat type
		if ( ! array_key_exists( $stat_type, $stats ) ) {
			return;
		}
		
		$stats[ $stat_type ] = absint( $stats[ $stat_type ] ) + 1;
		update_option( 'confab_usage_stats', $stats );
	}

	/**
	 * Display critical warning in plugin row when data deletion is enabled.
	 *
	 * Shows prominent red warning directly below plugin description when both 
	 * confirmation toggles are checked, visible right next to deactivate/delete actions.
	 *
	 * @param string $plugin_file Path to the plugin file relative to plugins directory
	 * @param array  $plugin_data Plugin metadata array
	 * @param string $status      Plugin status (active, inactive, etc)
	 * @since 2.1.0
	 */
	public function plugin_row_deletion_warning( $plugin_file, $plugin_data, $status ) {
		$initial_confirmed = get_option( 'confab_delete_initial_confirmed', false );
		$final_confirmed = get_option( 'confab_delete_final_confirmed', false );
		
		if ( $initial_confirmed && $final_confirmed ) {
			?>
			<tr class="plugin-update-tr active">
				<td colspan="4" class="plugin-update colspanchange">
					<div class="notice inline notice-error notice-alt">
						<p>
							<strong><?php esc_html_e( '⚠ Data Deletion Warning:', 'confab' ); ?></strong>
							<?php esc_html_e( 'Uninstalling ConFab will permanently delete ALL conference sessions, tracks, locations, and settings. This cannot be undone.', 'confab' ); ?>
							<a href="<?php echo esc_url( admin_url( 'themes.php?page=confab-settings' ) ); ?>">
								<?php esc_html_e( 'Disable data deletion', 'confab' ); ?>
							</a>
						</p>
					</div>
				</td>
			</tr>
			<?php
		}
	}

}

// Initialize the plugin
Confab_Plugin::get_instance();