/**
 * ConFab Hybrid Gutenberg Block
 * FILE: assets/js/confab-block-hybrid.js
 *
 * Hybrid React block using wp.apiFetch instead of ServerSideRender to eliminate
 * query cache race conditions while preserving all working React components.
 *
 * @package confab
 * @subpackage Assets\JavaScript
 * @since 2.0.0
 */

( function( wp ) {
	'use strict';

	var registerBlockType = wp.blocks.registerBlockType;
	var InspectorControls = wp.blockEditor.InspectorControls;
	var SelectControl = wp.components.SelectControl;
	var CheckboxControl = wp.components.CheckboxControl;
	var TextControl = wp.components.TextControl;
	var Spinner = wp.components.Spinner;
	var el = wp.element.createElement;
	var useState = wp.element.useState;
	var useEffect = wp.element.useEffect;
	var apiFetch = wp.apiFetch;
	var __ = wp.i18n.__;

	/**
	 * Format date to YYYY-MM-DD format
	 */
	function confab_dateFormatted( date ) {
		if ( date == null ) {
			date = new Date();
		} else {
			date = new Date( date );
		}
		
		var dd = String( date.getDate() ).padStart( 2, '0' );
		var mm = String( date.getMonth() + 1 ).padStart( 2, '0' );
		var yyyy = date.getFullYear();

		return yyyy + '-' + mm + '-' + dd;
	}

	/**
	 * Track terms array
	 */
	var trackTermsArray = [];
	
	/**
	 * Fetch track terms from WordPress REST API
	 */
	if ( wp.apiFetch ) {
		wp.apiFetch({ path: "/wp/v2/confab_track" })
			.then( function( posts ) {
				if ( posts && Array.isArray( posts ) ) {
					posts.forEach( function( val ) {
						trackTermsArray.push({
							id: val.id,
							name: val.name,
							slug: val.slug
						});
					});
				}
			})
			.catch( function( error ) {
				console.warn( 'ConFab: Could not load track terms', error );
			});
	}

	/**
	 * Custom Preview Component using wp.apiFetch
	 */
	function ConfabPreview( props ) {
		var attributes = props.attributes;
		var previewHtml = useState( '' );
		var setPreviewHtml = previewHtml[1];
		var previewHtmlValue = previewHtml[0];
		var isLoading = useState( true );
		var setIsLoading = isLoading[1];
		var isLoadingValue = isLoading[0];
		var error = useState( null );
		var setError = error[1];
		var errorValue = error[0];

		// Fetch preview when attributes change
		useEffect( function() {
			setIsLoading( true );
			setError( null );

			// Call our custom REST endpoint
			apiFetch({
				path: '/wp/v2/confab-preview',
				method: 'POST',
				data: {
					date: attributes.date || '',
					tracks: attributes.tracks || '',
					color_scheme: attributes.color_scheme || 'light',
					table_style: attributes.table_style || 'bordered',
					layout: attributes.layout || 'table',
					row_height: attributes.row_height || 'match',
					session_link: attributes.session_link || 'permalink',
					content: attributes.content || 'none',
					align: attributes.align || ''
				}
			})
			.then( function( response ) {
				if ( response && response.success && response.html ) {
					setPreviewHtml( response.html );
				} else {
					setError( 'Invalid response from server' );
				}
				setIsLoading( false );
			})
			.catch( function( err ) {
				console.error( 'ConFab preview error:', err );
				setError( err.message || 'Failed to load preview' );
				setIsLoading( false );
			});
		}, [
			attributes.date,
			attributes.tracks, 
			attributes.color_scheme,
			attributes.table_style,
			attributes.layout,
			attributes.row_height,
			attributes.session_link,
			attributes.content,
			attributes.align
		] );


		// Render loading state
		if ( isLoadingValue ) {
			return el( 'div', {
				style: { 
					textAlign: 'center', 
					padding: '20px' 
				}
			}, 
				el( Spinner ),
				el( 'p', { style: { marginTop: '10px' } }, 'Loading schedule preview...' )
			);
		}

		// Render error state
		if ( errorValue ) {
			return el( 'div', {
				style: { 
					background: '#f8d7da', 
					color: '#721c24', 
					padding: '10px', 
					borderRadius: '4px' 
				}
			}, 
				'Error loading preview: ' + errorValue
			);
		}

		// Render HTML preview
		return el( 'div', {
			dangerouslySetInnerHTML: { __html: previewHtmlValue }
		} );
	}

	/**
	 * Register ConFab Schedule Block
	 */
	registerBlockType( 'confab/schedule-block', {
		title: 'Conference Schedule (Confab)',
		description: 'Display conference sessions in table or grid format',
		icon: 'schedule',
		category: 'common',
		keywords: [ 'conference', 'schedule', 'session', 'event' ],
		supports: {
			align: [ 'wide', 'full' ]
		},

		attributes: {
			date: {
				type: 'string',
				default: confab_dateFormatted( null )
			},
			color_scheme: {
				type: 'string',
				default: 'light'
			},
			table_style: {
				type: 'string',
				default: 'bordered'
			},
			layout: {
				type: 'string',
				default: 'table'
			},
			row_height: {
				type: 'string',
				default: 'match'
			},
			session_link: {
				type: 'string',
				default: 'permalink'
			},
			tracks: {
				type: 'string',
				default: ''
			},
			align: {
				type: 'string',
				default: ''
			},
			content: {
				type: 'string',
				default: 'none'
			}
		},

		edit: function( props ) {
			var attributes = props.attributes;
			var setAttributes = props.setAttributes;

			// Process tracks string into array
			var tracksArray = attributes.tracks ? attributes.tracks.split( ',' ) : [];

			// Build track controls as proper React elements array
			var trackControlElements = [];
			
			if ( trackTermsArray.length > 0 ) {
				// Add header
				trackControlElements.push(
					el( 'p', { key: 'track-controls-header' }, 'Select Tracks:' )
				);
				
				// Add each checkbox with proper React keys
				trackTermsArray.forEach( function( trackTerm, index ) {
					var isChecked = tracksArray.indexOf( trackTerm.slug ) !== -1;
					
					trackControlElements.push(
						el( CheckboxControl, {
							key: 'track-checkbox-' + trackTerm.id + '-' + index,
							label: trackTerm.name,
							checked: isChecked,
							onChange: function( slug ) {
								return function( checked ) {
									var newTracks = tracksArray.slice();
									var trackIndex = newTracks.indexOf( slug );
									
									if ( checked && trackIndex === -1 ) {
										newTracks.push( slug );
									} else if ( ! checked && trackIndex !== -1 ) {
										newTracks.splice( trackIndex, 1 );
									}
									
									setAttributes({ tracks: newTracks.join( ',' ) });
								};
							}( trackTerm.slug )
						})
					);
				});
			}

			// Build inspector controls with all elements properly keyed
			var inspectorControls = el( InspectorControls, { key: 'inspector-controls' },
				el( 'div', { key: 'controls-container' },
					el( TextControl, {
						key: 'date-control',
						label: 'Date (YYYY-MM-DD)',
						value: attributes.date,
						onChange: function( value ) {
							setAttributes({ date: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					el( SelectControl, {
						key: 'color-scheme-control',
						label: 'Color Scheme',
						value: attributes.color_scheme,
						options: [
							{ value: 'light', label: 'Light' },
							{ value: 'dark', label: 'Dark' },
							{ value: 'lagoon', label: 'Lagoon' },
							{ value: 'lagoon-dark', label: 'Lagoon Dark' },
							{ value: 'ocean', label: 'Ocean' },
							{ value: 'ocean-dark', label: 'Ocean Dark' },
							{ value: 'pastels', label: 'Pastels' },
							{ value: 'pastels-dark', label: 'Pastels Dark' },
							{ value: 'jeweltones', label: 'Jeweltones' },
							{ value: 'jeweltones-dark', label: 'Jeweltones Dark' },
							{ value: 'forest', label: 'Forest' },
							{ value: 'forest-dark', label: 'Forest Dark' },
							{ value: 'earthtones', label: 'Earthtones' },
							{ value: 'earthtones-dark', label: 'Earthtones Dark' },
							{ value: 'honey', label: 'Honey' },
							{ value: 'honey-dark', label: 'Honey Dark' },
							{ value: 'sunrise', label: 'Sunrise' },
							{ value: 'sunrise-dark', label: 'Sunrise Dark' },
							{ value: 'autumn', label: 'Autumn' },
							{ value: 'autumn-dark', label: 'Autumn Dark' },
							{ value: 'terracotta', label: 'Terracotta' },
							{ value: 'terracotta-dark', label: 'Terracotta Dark' },
							{ value: 'burgundy', label: 'Burgundy' },
							{ value: 'burgundy-dark', label: 'Burgundy Dark' },
							{ value: 'sanguine', label: 'Sanguine' },
							{ value: 'sanguine-dark', label: 'Sanguine Dark' }
						],
						onChange: function( value ) {
							setAttributes({ color_scheme: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					el( SelectControl, {
						key: 'table-style-control',
						label: 'Table Style',
						value: attributes.table_style,
						options: [
							{ value: 'bordered', label: 'Bordered (Default)' },
							{ value: 'borderless', label: 'Borderless' },
							{ value: 'rounded', label: 'Rounded Cards' }
						],
						onChange: function( value ) {
							setAttributes({ table_style: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					el( SelectControl, {
						key: 'layout-control',
						label: 'Layout',
						value: attributes.layout,
						options: [
							{ value: 'table', label: 'Table' },
							{ value: 'grid', label: 'Grid' }
						],
						onChange: function( value ) {
							setAttributes({ layout: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					el( SelectControl, {
						key: 'row-height-control',
						label: 'Row Height',
						value: attributes.row_height,
						options: [
							{ value: 'match', label: 'Match' },
							{ value: 'auto', label: 'Auto' }
						],
						onChange: function( value ) {
							setAttributes({ row_height: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					el( SelectControl, {
						key: 'session-link-control',
						label: 'Session Link',
						value: attributes.session_link,
						options: [
							{ value: 'permalink', label: 'Permalink' },
							{ value: 'anchor', label: 'Anchor' },
							{ value: 'none', label: 'None' }
						],
						onChange: function( value ) {
							setAttributes({ session_link: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					el( SelectControl, {
						key: 'content-control',
						label: 'Content Display',
						value: attributes.content,
						options: [
							{ value: 'none', label: 'Title Only' },
							{ value: 'excerpt', label: 'Title + Excerpt' },
							{ value: 'full', label: 'Title + Full Content' }
						],
						onChange: function( value ) {
							setAttributes({ content: value });
						},
						__next40pxDefaultSize: true,
						__nextHasNoMarginBottom: true
					}),

					// Track controls section with proper container
					el( 'div', { key: 'track-controls-section' }, trackControlElements )
				)
			);

			// Build preview using our custom component
			var preview = el( ConfabPreview, {
				key: 'confab-preview',
				attributes: attributes
			});

			// Return block elements with proper keys
			return [
				el( 'div', { key: 'preview-container' }, preview ),
				inspectorControls
			];
		},

		save: function() {
			return null;
		}
	});

})( window.wp );