/**
 * ConFab Admin Settings JavaScript
 * FILE: assets/js/confab-admin-settings.js
 *
 * Handles accordion functionality and interactive elements for the ConFab
 * settings page with accessibility and keyboard navigation support.
 *
 * @package confab
 * @subpackage Admin/Assets
 * @since 2.0.0
 */

(function($) {
    'use strict';

    /**
     * Initialize ConFab admin settings functionality when document is ready.
     */
    $(document).ready(function() {
        initAccordions();
        initDataManagement();
    });

    /**
     * Initialize accordion functionality with accessibility support.
     *
     * Sets up click handlers, keyboard navigation, and proper ARIA attributes
     * for the documentation accordion sections in the ConFab settings interface.
     *
     * @since 2.0.0
     */
    function initAccordions() {
        $('.confab-accordion-header').on('click', function(e) {
            e.preventDefault();
            toggleAccordion($(this));
        });

        // Keyboard navigation support
        $('.confab-accordion-header').on('keydown', function(e) {
            // Enter or Space key
            if (e.which === 13 || e.which === 32) {
                e.preventDefault();
                toggleAccordion($(this));
            }
        });

        // Set initial focus management
        $('.confab-accordion-header').attr('tabindex', '0');
    }

    /**
     * Toggle accordion section open/closed state with accessibility management.
     *
     * Manages accordion expansion behavior by closing other sections and
     * opening the target section with proper ARIA state updates.
     *
     * @param {jQuery} header The accordion header element that was activated
     * @since 2.0.0
     */
    function toggleAccordion(header) {
        var isExpanded = header.attr('aria-expanded') === 'true';
        var targetId = header.data('target');
        var content = $('#' + targetId);

        // Close all other accordions first
        closeAllAccordions();

        // Toggle this accordion if it wasn't already open
        if (!isExpanded) {
            openAccordion(header, content);
        }
    }

    /**
     * Close all accordion sections and reset ARIA expanded attributes.
     *
     * Ensures only one accordion section is open at a time by resetting
     * all sections to their collapsed state.
     *
     * @since 2.0.0
     */
    function closeAllAccordions() {
        $('.confab-accordion-header').attr('aria-expanded', 'false');
        $('.confab-accordion-content').removeClass('confab-accordion-open');
    }

    /**
     * Open specific accordion section with smooth scrolling behavior.
     *
     * Expands the target accordion section and implements intelligent
     * scrolling to ensure the opened content is visible in the viewport.
     *
     * @param {jQuery} header The accordion header element to expand
     * @param {jQuery} content The accordion content element to show
     * @since 2.0.0
     */
    function openAccordion(header, content) {
        header.attr('aria-expanded', 'true');
        content.addClass('confab-accordion-open');

        // Scroll the opened section into view if needed
        setTimeout(function() {
            var headerOffset = header.offset().top;
            var scrollTop = $(window).scrollTop();
            var windowHeight = $(window).height();

            // If header is not visible, scroll to it
            if (headerOffset < scrollTop || headerOffset > scrollTop + windowHeight) {
                $('html, body').animate({
                    scrollTop: headerOffset - 100
                }, 300);
            }
        }, 100);
    }

    /**
     * Initialize data management dual-confirmation toggle system.
     *
     * Implements three-state interface for ConFab data deletion configuration:
     * State 1 (both false) - normal toggle with basic description,
     * State 2 (configuring) - warning section visible with full deletion details,
     * State 3 (both confirmed) - danger state with red toggle and alert indicator.
     *
     * @since 2.0.0
     */
    function initDataManagement() {
        const warningDetails = document.getElementById('confab-warning-details');
        const initialToggle = document.getElementById('confab_delete_initial_confirmed');
        const finalToggle = document.getElementById('confab_delete_final_confirmed');
        
        if (!warningDetails || !initialToggle) {
            return;
        }
        
        // Set initial visibility based on saved state
        if (initialToggle.checked && finalToggle && !finalToggle.checked) {
            // State 2: First checked, second unchecked - show warning details
            warningDetails.classList.add('show-details');
        } else {
            // State 1 (both false) or State 3 (both true) - hide warning details
            warningDetails.classList.remove('show-details');
        }
        
        // Handle initial toggle changes to show/hide warning section
        initialToggle.addEventListener('change', function() {
            if (this.checked) {
                warningDetails.classList.add('show-details');
            } else {
                warningDetails.classList.remove('show-details');
                
                // Clear final confirmation when initial is unchecked
                if (finalToggle) {
                    finalToggle.checked = false;
                }
            }
        });
    }

})(jQuery);