<?php

/**
 * Plugin Name: Conditional Block Visibility
 * Description: Manage Gutenberg block visibility based on user roles, login status, or device type.
 * Version:     1.0.3
 * Author:      Sanad Qazi
 * Author URI:  https://sanadqazi.com
 * Text Domain: conditional-block-visibility
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires PHP: 7.4
 */

if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

define('CONDITIONAL_BLOCK_VISIBILITY_VERSION', '1.0.0');
define('CONDITIONAL_BLOCK_VISIBILITY_PATH', plugin_dir_path(__FILE__));
define('CONDITIONAL_BLOCK_VISIBILITY_URL', plugin_dir_url(__FILE__));

/**
 * Enqueue editor assets for Gutenberg blocks.
 */
function conditional_block_visibility_enqueue_editor_assets()
{
    // Enqueue the JavaScript file.
    wp_enqueue_script(
        'conditional-block-visibility-script',
        plugins_url('assets/js/block-visibility.build.js', __FILE__), // Use transpiled file.
        ['wp-blocks', 'wp-editor', 'wp-components', 'wp-i18n', 'wp-element'],
        CONDITIONAL_BLOCK_VISIBILITY_VERSION,
        true
    );

    // Retrieve all roles.
    global $wp_roles;
    if (! isset($wp_roles)) {
        $wp_roles = new WP_Roles();
    }
    $roles = $wp_roles->roles;
    $role_options = [];
    foreach ($roles as $role_key => $role_details) {
        $role_options[] = [
            'label' => $role_details['name'],
            'value' => $role_key,
        ];
    }

    // Pass the roles to JavaScript.
    wp_localize_script(
        'conditional-block-visibility-script',
        'conditionalBlockVisibilityData',
        [
            'roles' => $role_options,
        ]
    );

    // Enqueue the CSS file.
    wp_enqueue_style(
        'conditional-block-visibility-style',
        plugins_url('assets/css/style.css', __FILE__),
        [],
        CONDITIONAL_BLOCK_VISIBILITY_VERSION
    );
}
add_action('enqueue_block_editor_assets', 'conditional_block_visibility_enqueue_editor_assets');

/**
 * Modify block rendering for conditional visibility.
 *
 * @param string $block_content Block content.
 * @param array  $block         Block data.
 * @return string Modified block content.
 */
function conditional_block_visibility_render_block($block_content, $block)
{
    if (isset($block['attrs']['visibility'])) {
        $visibility = $block['attrs']['visibility'];

        // User Role Restriction.
        if (isset($visibility['userRole']) && ! empty($visibility['userRole']) && ! current_user_can(sanitize_text_field($visibility['userRole']))) {
            return '';
        }

        // Login Status Restriction.
        if (isset($visibility['loginStatus'])) {
            $is_logged_in = is_user_logged_in();
            if (
                ('logged_in' === sanitize_text_field($visibility['loginStatus']) && ! $is_logged_in) ||
                ('logged_out' === sanitize_text_field($visibility['loginStatus']) && $is_logged_in)
            ) {
                return '';
            }
        }

        // Device Type Restriction.
        if (isset($visibility['deviceType'])) {
            $device_type  = esc_attr($visibility['deviceType']);
            $block_content = sprintf(
                '<div class="cbv-visibility-%s">%s</div>',
                $device_type,
                $block_content
            );
        }
    }

    return $block_content;
}
add_filter('render_block', 'conditional_block_visibility_render_block', 10, 2);
