<?php

/**
 * Main Plugin Class file.
 *
 * @package Color_Theme_Manager_For_Divi
 */
if ( !defined( 'ABSPATH' ) ) {
    exit;
    // Exit if accessed directly.
}
/**
 * The main plugin class.
 *
 * Responsible for loading dependencies, defining hooks, and initializing the plugin.
 *
 * @since 1.0.0
 */
class CTMD_Main {
    /**
     * Instance of the Cache Manager class.
     *
     * @var CTMD_Cache_Manager
     */
    public $cache_manager;

    /**
     * Instance of the Color Manager class.
     *
     * @var CTMD_Color_Manager
     */
    public $color_manager;

    /**
     * Instance of the Palette Manager class.
     *
     * @var CTMD_Palette_Manager
     */
    public $palette_manager;

    /**
     * Instance of the Import/Export Manager class.
     *
     * @var CTMD_Import_Export
     */
    public $import_export;

    /**
     * Instance of the Theme Generator class.
     *
     * @var CTMD_Theme_Generator
     */
    public $theme_generator;

    /**
     * Instance of the Layout Downloader class.
     *
     * @var CTMD_Layout_Downloader
     */
    public $layout_downloader;

    /**
     * Instance of the Admin class.
     *
     * @var CTMD_Admin
     */
    public $admin;

    /**
     * Constructor.
     *
     * Loads dependencies and defines the hooks for the plugin.
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->load_dependencies();
        $this->add_hooks();
    }

    /**
     * Loads the required dependencies for this plugin.
     *
     * Instantiates all the manager classes and injects cross-dependencies.
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function load_dependencies() {
        // Instantiate all the manager classes.
        $this->cache_manager = new CTMD_Cache_Manager();
        $this->color_manager = new CTMD_Color_Manager($this->cache_manager);
        $this->palette_manager = new CTMD_Palette_Manager($this->cache_manager);
        // Inject cross-dependencies.
        $this->color_manager->set_palette_manager( $this->palette_manager );
        $this->palette_manager->set_color_manager( $this->color_manager );
        // Admin depends on other managers, so it's last.
        $this->admin = new CTMD_Admin($this->color_manager, $this->palette_manager);
    }

    /**
     * Initializes internationalization properties for sub-classes.
     *
     * Calls the setup_properties methods on manager classes to ensure
     * translation strings are loaded correctly after 'init'.
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function initialize_i18n_properties() {
        $this->admin->setup_properties();
        $this->color_manager->setup_properties();
        $this->palette_manager->setup_properties();
    }

    /**
     * Registers all hooks related to the plugin functionality.
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function add_hooks() {
        add_action( 'init', array($this, 'initialize_i18n_properties') );
        add_action( 'admin_menu', array($this->admin, 'add_admin_menu') );
        add_action( 'admin_enqueue_scripts', array($this->admin, 'enqueue_admin_scripts') );
        add_filter( 'submenu_file', array($this->admin, 'highlight_correct_submenu') );
        add_action( 'admin_post_ctmd_save_colors', array($this->color_manager, 'save_colors') );
        add_action( 'admin_post_ctmd_add_new_color', array($this->color_manager, 'add_new_color') );
        add_action( 'admin_post_ctmd_create_palette', array($this->palette_manager, 'create_palette') );
        add_action( 'admin_post_ctmd_delete_palette', array($this->palette_manager, 'delete_palette') );
        add_action( 'admin_post_ctmd_restore_palette', array($this->palette_manager, 'restore_palette') );
        add_action( 'admin_post_ctmd_reset_colors', array($this->color_manager, 'reset_colors') );
        add_action( 'admin_post_ctmd_save_plugin_settings', array($this, 'save_plugin_settings') );
    }

    /**
     * Enqueues frontend scripts and dynamic CSS variables.
     *
     * Generates CSS variables based on the active theme palette (Light, Dark, Mixed)
     * and injects them as inline styles.
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function enqueue_frontend_scripts() {
        $settings = get_option( 'ctmd_settings', array() );
        if ( empty( $settings['active_theme_name'] ) ) {
            return;
        }
        return;
    }

    /**
     * Saves the general plugin settings.
     *
     * Handles the POST request from the settings tab, such as the "Delete on Uninstall" option.
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function save_plugin_settings() {
        if ( !current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Unauthorized.', 'color-theme-manager-for-divi' ) );
        }
        check_admin_referer( 'ctmd_plugin_settings_nonce' );
        $settings = get_option( 'ctmd_settings', array() );
        // Sanitize POST data early.
        $delete_on_uninstall = absint( filter_input( INPUT_POST, 'delete_on_uninstall', FILTER_SANITIZE_NUMBER_INT ) ?? 0 );
        $settings['delete_on_uninstall'] = 1 === $delete_on_uninstall;
        update_option( 'ctmd_settings', $settings );
        wp_safe_redirect( admin_url( 'admin.php?page=color-theme-manager-for-divi&tab=settings&message=settings_saved' ) );
        exit;
    }

    /**
     * Handles plugin uninstallation cleanup.
     *
     * Checks if the "Delete on Uninstall" setting is active and removes plugin data
     * and resets Divi colors if so.
     *
     * @since 1.0.0
     *
     * @return void
     */
    public static function uninstall() {
        $settings = get_option( 'ctmd_settings', array() );
        if ( !empty( $settings['delete_on_uninstall'] ) ) {
            $option_data = get_option( 'et_divi', array() );
            $option_data = maybe_unserialize( $option_data );
            if ( isset( $option_data['et_global_data'] ) ) {
                $et_global_data = maybe_unserialize( $option_data['et_global_data'] );
                $et_global_data['global_colors'] = array();
                $option_data['et_global_data'] = maybe_serialize( $et_global_data );
            }
            $default_special_colors = array(
                'accent_color'           => '#2ea3f2',
                'secondary_accent_color' => '#2ea3f2',
                'header_color'           => '#666666',
                'font_color'             => '#666666',
            );
            foreach ( $default_special_colors as $key => $color ) {
                $option_data[$key] = $color;
            }
            update_option( 'et_divi', $option_data );
            delete_option( 'ctmd_settings' );
        }
    }

}
