<?php
/**
 * Payment service for handling invoice payments.
 *
 * @package bitcoin-invoice-form
 */

declare(strict_types=1);

namespace CoinsnapBIF\Services;

use CoinsnapBIF\Database\Installer;
use CoinsnapBIF\Util\CoinsnapBIF_Logger;
use CoinsnapBIF\Util\CoinsnapBIF_Util_Provider_Factory;
use CoinsnapBIF\Admin\CoinsnapBIF_Admin_Settings as Settings;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Payment service for handling invoice payments.
 */
class CoinsnapBIF_Services_Payment_Service {
	/**
	 * Create a new invoice payment.
	 *
	 * @param int   $form_id Form ID.
	 * @param array $data    Form data.
	 * @return array Response data.
	 */
	public static function create_invoice( int $form_id, array $data ): array {
		global $wpdb;

		try {
			// Validate form exists
			$form = get_post( $form_id );
			$valid_post_types = array( 'coinsnapbif', 'coinsnap_invoice_form' );
                        
                        if ( ! $form || ! in_array( $form->post_type, $valid_post_types, true ) ) {
				return array(
					'success' => false,
					'message' => __( 'Invalid form ID.', 'coinsnap-bitcoin-invoice-form' ),
				);
			}

			// Get form configuration
			$fields = get_post_meta( $form_id, '_coinsnapbif_fields', true );
			$payment_config = get_post_meta( $form_id, '_coinsnapbif_payment', true );
			$email_config = get_post_meta( $form_id, '_coinsnapbif_email', true );

			// Set defaults
			$fields = wp_parse_args( $fields, array() );
			$payment_config = wp_parse_args( $payment_config, array(
				'amount'     => '',
				'currency'   => 'USD',
				'description' => '',
			) );

			// Calculate amount with locale-aware normalization (supports comma or dot decimals)
			$amount = 0.0;
			$raw_amount = '';
			if ( isset( $data['bif_amount'] ) && $data['bif_amount'] !== '' ) {
				$raw_amount = (string) $data['bif_amount'];
			} elseif ( ! empty( $payment_config['amount'] ) ) {
				$raw_amount = (string) $payment_config['amount'];
			}

			$raw_amount = trim( (string) $raw_amount );
			if ( $raw_amount !== '' ) {
				// Detect decimal separator and remove thousands separators safely
				$has_comma = strpos( $raw_amount, ',' ) !== false;
				$has_dot   = strpos( $raw_amount, '.' ) !== false;
				if ( $has_comma && $has_dot ) {
					// Determine which appears last; treat that as decimal
					$last_comma = strrpos( $raw_amount, ',' );
					$last_dot   = strrpos( $raw_amount, '.' );
					if ( $last_comma > $last_dot ) {
						// European style: 1.234,56 → remove dots, replace comma with dot
						$normalized = str_replace( '.', '', $raw_amount );
						$normalized = str_replace( ',', '.', $normalized );
					} else {
						// US style: 1,234.56 → remove commas
						$normalized = str_replace( ',', '', $raw_amount );
					}
				} elseif ( $has_comma && ! $has_dot ) {
					// Only comma present: treat as decimal separator
					$normalized = str_replace( ',', '.', $raw_amount );
				} else {
					// Only dot or neither: remove any stray spaces and commas
					$normalized = str_replace( ',', '', $raw_amount );
				}
				$amount = (float) $normalized;
			}

			if ( $amount <= 0 ) {
				return array(
					'success' => false,
					'message' => __( 'Invalid amount.', 'coinsnap-bitcoin-invoice-form' ),
				);
			}

			// Server-side required field validation for core and enabled+required fields
			$errors = array();
			// Determine recipient value with backward compatibility
			$recipient_val = isset( $data['bif_invoice_recipient'] ) ? trim( (string) $data['bif_invoice_recipient'] ) : '';
			if ( '' === $recipient_val && isset( $data['bif_name'] ) ) {
				$recipient_val = trim( (string) $data['bif_name'] );
			}
			if ( '' === $recipient_val ) {
				$errors[] = __( 'Invoice Recipient is required.', 'coinsnap-bitcoin-invoice-form' );
			}
			$inv_no_val = isset( $data['bif_invoice_number'] ) ? trim( (string) $data['bif_invoice_number'] ) : '';
			if ( '' === $inv_no_val ) {
				$errors[] = __( 'Invoice Number is required.', 'coinsnap-bitcoin-invoice-form' );
			}
			// Validate other fields if they are enabled and marked required in form config
			$maybe_required = array(
				'email' => __( 'Email', 'coinsnap-bitcoin-invoice-form' ),
				'company' => __( 'Company', 'coinsnap-bitcoin-invoice-form' ),
				'description' => __( 'Message', 'coinsnap-bitcoin-invoice-form' ),
			);
			foreach ( $maybe_required as $f => $label ) {
				$enabled_val = $fields[ $f . '_enabled' ] ?? '0';
				$required_val = $fields[ $f . '_required' ] ?? '0';
				$enabled = ( '1' === $enabled_val || 'on' === $enabled_val || true === $enabled_val );
				$required = ( '1' === $required_val || 'on' === $required_val || true === $required_val );
				if ( $enabled && $required ) {
                                    $key = 'bif_' . $f;
                                    $val = isset( $data[ $key ] ) ? trim( (string) $data[ $key ] ) : '';
                                    if ( '' === $val ) {
					$errors[] = sprintf(
                                            /* translators: 1: Required value */
                                            __( '%s is required', 'coinsnap-bitcoin-invoice-form' ), $label );
                                    } elseif ( 'email' === $f && ! is_email( $val ) ) {
					$errors[] = __( 'Please enter a valid email address.', 'coinsnap-bitcoin-invoice-form' );
                                    }
				}
			}
			if ( ! empty( $errors ) ) {
				return array(
					'success' => false,
					'message' => implode( ' ', $errors ),
				);
			}

			// Apply discount if configured on the form
			$discount_enabled = $fields['discount_enabled'] ?? '0';
			if ( '1' === $discount_enabled || 'on' === $discount_enabled ) {
				$discount_type  = $fields['discount_type'] ?? 'fixed';
				$discount_value = isset( $fields['discount_value'] ) ? floatval( $fields['discount_value'] ) : 0.0;
				if ( $discount_value > 0 ) {
					if ( 'percent' === $discount_type ) {
						$amount = $amount - ( $amount * ( $discount_value / 100 ) );
					} else { // fixed
						$amount = $amount - $discount_value;
					}
					// Ensure amount doesn't go below zero
					if ( $amount < 0 ) {
						$amount = 0;
					}
				}
			}

			// Convert amount to smallest currency unit (e.g., cents for USD)
			$amount_cents = intval( round( $amount * 100 ) );

			// Get currency strictly from form configuration (end user cannot change it)
			if ( ! empty( $payment_config['currency'] ) ) {
				$currency = $payment_config['currency'];
			} else {
				$currency = 'USD'; // Fallback if not configured on form
			}

			// Generate transaction ID
			$transaction_id = 'coinsnapbif_' . time() . '_' . wp_generate_password( 8, false );

			// Prepare invoice data
			$invoice_data = array(
				'name'            => sanitize_text_field( ( $data['bif_invoice_recipient'] ?? '' ) !== '' ? $data['bif_invoice_recipient'] : ( $data['bif_name'] ?? '' ) ),
				'email'           => sanitize_email( $data['bif_email'] ?? '' ),
				'company'         => sanitize_text_field( $data['bif_company'] ?? '' ),
				'invoice_number'  => sanitize_text_field( $data['bif_invoice_number'] ?? '' ),
				'description'     => sanitize_textarea_field( $data['bif_description'] ?? '' ),
			);

			// Create payment provider
			$payment_provider = CoinsnapBIF_Util_Provider_Factory::payment_for_form( $form_id );


			// Create invoice with payment provider
			$payment_result = $payment_provider->create_invoice( $form_id, $amount_cents, $currency, $invoice_data );
                        
                        if ( empty( $payment_result ) || empty( $payment_result['invoice_id'] ) ) {
				CoinsnapBIF_Logger::error( 'Failed to create payment invoice', array(
					'form_id' => $form_id,
					'amount'  => $amount,
					'currency' => $currency,
				) );
				return array(
					'success' => false,
					'message' => __( 'Failed to create payment invoice.', 'coinsnap-bitcoin-invoice-form' ),
				);
			}

			// Save transaction to database
			$table_name = Installer::table_name();

			// Get and sanitize user agent
			$user_agent = '';
			if ( isset( $_SERVER['HTTP_USER_AGENT'] ) ) {
				$user_agent = sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
			}

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Insert via $wpdb is appropriate here to persist transaction data.
			$insert_result = $wpdb->insert(
				$table_name,
				array(
					'form_id'            => $form_id,
					'transaction_id'     => $transaction_id,
					'invoice_number'     => $invoice_data['invoice_number'],
					'customer_name'      => $invoice_data['name'],
					'customer_email'     => $invoice_data['email'],
					'customer_company'   => $invoice_data['company'],
					'amount'             => $amount,
					'currency'           => $currency,
					'description'        => $invoice_data['description'],
					'payment_provider'   => (!empty($payment_config['provider_override']))? $payment_config['provider_override'] : Settings::get_settings()['payment_provider'],
					'payment_invoice_id' => $payment_result['invoice_id'],
					'payment_url'        => $payment_result['payment_url'] ?? '',
					'payment_status'     => 'unpaid',
					'ip'                 => self::get_client_ip(),
					'user_agent'         => $user_agent,
					'created_at'         => current_time( 'mysql' ),
					'updated_at'         => current_time( 'mysql' ),
				),
				array(
					'%d', '%s', '%s', '%s', '%s', '%s', '%f', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s'
				)
			);

			if ( false === $insert_result ) {
				CoinsnapBIF_Logger::error( 'Failed to save transaction to database', array(
					'form_id' => $form_id,
					'transaction_id' => $transaction_id,
					'wpdb_error' => $wpdb->last_error,
				) );
				return array(
					'success' => false,
					'message' => __( 'Failed to save transaction.', 'coinsnap-bitcoin-invoice-form' ),
				);
			}

			CoinsnapBIF_Logger::info( 'Invoice created successfully', array(
				'form_id' => $form_id,
				'transaction_id' => $transaction_id,
				'invoice_id' => $payment_result['invoice_id'],
				'amount' => $amount,
				'currency' => $currency,
			) );

			// Get redirect configuration
			$redirect_config = get_post_meta( $form_id, '_coinsnapbif_redirect', true );
			$redirect_config = wp_parse_args( $redirect_config, array(
				'success_page' => '',
				'error_page'   => '',
				'thank_you_message' => __( 'Thank you! Your payment has been processed successfully.', 'coinsnap-bitcoin-invoice-form' ),
			) );

			return array(
				'success' => true,
				'data'    => array(
					'transaction_id' => $transaction_id,
					'invoice_id'     => $payment_result['invoice_id'],
					'payment_url'    => $payment_result['payment_url'],
					'amount'         => $amount,
					'currency'       => $currency,
					'description'    => $invoice_data['description'],
					'success_page'   => $redirect_config['success_page'],
					'thank_you_message' => $redirect_config['thank_you_message'],
				),
			);

		} catch ( \Exception $e ) {
			CoinsnapBIF_Logger::error( 'Exception in create_invoice', array(
				'form_id' => $form_id,
				'error'   => $e->getMessage(),
				'trace'   => $e->getTraceAsString(),
			) );
			return array(
				'success' => false,
				'message' => __( 'An error occurred while creating the invoice.', 'coinsnap-bitcoin-invoice-form' ),
			);
		}
	}

	/**
	 * Handle payment webhook.
	 *
	 * @param string $provider Payment provider name.
	 * @param array  $data     Webhook data.
	 * @return array Response data.
	 */
	public static function handle_webhook( string $provider, array $data ): array {
		global $wpdb;

		try {
			// Create payment provider
			$payment_provider = CoinsnapBIF_Util_Provider_Factory::payment_for_form( 0 );
			if ( $provider === 'btcpay' ) {
				$payment_provider = new \CoinsnapBIF\Providers\Payment\BTCPayProvider();
			} else {
				$payment_provider = new \CoinsnapBIF\Providers\Payment\CoinsnapProvider();
			}

			// Handle webhook
			$webhook_result = $payment_provider->handle_webhook( $data );

			if ( empty( $webhook_result ) || empty( $webhook_result['invoice_id'] ) ) {
				return array(
					'success' => false,
					'message' => __( 'Invalid webhook data.', 'coinsnap-bitcoin-invoice-form' ),
				);
			}

			$invoice_id = $webhook_result['invoice_id'];
			$paid = $webhook_result['paid'] ?? false;

			// Update transaction status
			$table_name = Installer::table_name();
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Update via $wpdb is appropriate here to set status.
			$update_result = $wpdb->update(
				$table_name,
				array(
					'payment_status' => $paid ? 'paid' : 'failed',
					'updated_at'     => current_time( 'mysql' ),
				),
				array(
					'payment_invoice_id' => $invoice_id,
				),
				array( '%s', '%s' ),
				array( '%s' )
			);

			if ( false === $update_result ) {
				CoinsnapBIF_Logger::error( 'Failed to update transaction status', array(
					'invoice_id' => $invoice_id,
					'paid' => $paid,
					'wpdb_error' => $wpdb->last_error,
				) );
			}

			// If payment is successful, send email notification
			if ( $paid ) {
				self::send_payment_notification( $invoice_id );
			}

			CoinsnapBIF_Logger::info( 'Webhook processed successfully', array(
				'provider' => $provider,
				'invoice_id' => $invoice_id,
				'paid' => $paid,
			) );

			return array(
				'success' => true,
				'message' => __( 'Webhook processed successfully.', 'coinsnap-bitcoin-invoice-form' ),
			);

		} catch ( \Exception $e ) {
			CoinsnapBIF_Logger::error( 'Exception in handle_webhook', array(
				'provider' => $provider,
				'error'    => $e->getMessage(),
				'trace'    => $e->getTraceAsString(),
			) );
			return array(
				'success' => false,
				'message' => __( 'An error occurred while processing the webhook.', 'coinsnap-bitcoin-invoice-form' ),
			);
		}
	}

	/**
	 * Check payment status.
	 *
	 * @param string $invoice_id Invoice ID.
	 * @return array Response data.
	 */
	public static function check_payment_status( string $invoice_id ): array {
		global $wpdb;

		try {
			// Get transaction from database
			$table_name = Installer::table_name();

			// Properly prepare query inline
			$transaction = $wpdb->get_row(
				$wpdb->prepare(
                                        "SELECT * FROM %i WHERE payment_invoice_id = %s",
                                        $table_name,
                                        $invoice_id
				)
			);


			if ( ! $transaction ) {
				return array(
					'success' => false,
					'message' => __( 'Transaction not found.', 'coinsnap-bitcoin-invoice-form' ),
				);
			}

			// If already paid, return status
			if ( 'paid' === $transaction->payment_status ) {
				return array(
					'success' => true,
					'data'    => array(
						'invoice_id' => $invoice_id,
						'paid'       => true,
						'status'     => 'paid',
					),
				);
			}

			// Check with payment provider
			$payment_provider = CoinsnapBIF_Util_Provider_Factory::payment_for_form( $transaction->form_id );
			$status_result = $payment_provider->check_invoice_status( $invoice_id );

			if ( ! empty( $status_result ) && $status_result['paid'] ) {
				// Update status in database
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Update via $wpdb is appropriate here after status check.
				$wpdb->update(
					$table_name,
					array(
						'payment_status' => 'paid',
						'updated_at'     => current_time( 'mysql' ),
					),
					array(
						'payment_invoice_id' => $invoice_id,
					),
					array( '%s', '%s' ),
					array( '%s' )
				);

				// Send email notification
				self::send_payment_notification( $invoice_id );
			}

			return array(
				'success' => true,
				'data'    => array(
					'invoice_id' => $invoice_id,
					'paid'       => $status_result['paid'] ?? false,
					'status'     => $status_result['status'] ?? 'unknown',
				),
			);

		} catch ( \Exception $e ) {
			CoinsnapBIF_Logger::error( 'Exception in check_payment_status', array(
				'invoice_id' => $invoice_id,
				'error'      => $e->getMessage(),
				'trace'      => $e->getTraceAsString(),
			) );
			return array(
				'success' => false,
				'message' => __( 'An error occurred while checking payment status.', 'coinsnap-bitcoin-invoice-form' ),
			);
		}
	}

	/**
	 * Send payment notification email.
	 *
	 * @param string $invoice_id Invoice ID.
	 */
	private static function send_payment_notification( string $invoice_id ): void {
		global $wpdb;

		// Get transaction details
		$table_name = Installer::table_name();

		// Properly prepare query inline
		$transaction = $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM %i WHERE payment_invoice_id = %s",
                                $table_name,
				$invoice_id
			)
		);


		if ( ! $transaction ) {
			return;
		}

		// Get form email configuration (admin and customer in separate metaboxes; support legacy combined storage)
		$admin_defaults = array(
			'admin_email'     => get_option( 'admin_email' ),
			'email_subject'   => __( 'New Invoice Payment Received', 'coinsnap-bitcoin-invoice-form' ),
			'email_template'  => __( 'A new invoice payment has been received:

Invoice Number: {invoice_number}
Customer: {customer_name}
Email: {customer_email}
Amount: {amount} {currency}
Payment Status: {payment_status}

Payment Details:
Transaction ID: {transaction_id}
Payment Provider: {payment_provider}

Description: {description}', 'coinsnap-bitcoin-invoice-form' ),
		);
		$customer_defaults = array(
			'customer_email_enabled' => '1',
			'customer_email_subject' => __( 'Your payment receipt for invoice {invoice_number}', 'coinsnap-bitcoin-invoice-form' ),
			'customer_email_template' => __( "Hello {customer_name},\n\nThank you for your payment. Here are the details of your receipt:\n\nInvoice Number: {invoice_number}\nAmount Paid: {amount} {currency}\nPayment Status: {payment_status}\n\nDescription: {description}\n\nTransaction ID: {transaction_id}\nPayment Provider: {payment_provider}\n\nIf you have any questions, reply to this email.\n\nBest regards,\n{site_name}", 'coinsnap-bitcoin-invoice-form' ),
		);

		$admin_config    = get_post_meta( $transaction->form_id, '_coinsnapbif_email', true );
		$customer_config = get_post_meta( $transaction->form_id, '_coinsnapbif_email_customer', true );

		// Backward compatibility: if customer metabox not saved, fall back to any customer keys stored in admin metabox
		$legacy_customer_subset = array();
		if ( is_array( $admin_config ) ) {
			foreach ( array( 'customer_email_enabled', 'customer_email_subject', 'customer_email_template' ) as $k ) {
				if ( array_key_exists( $k, $admin_config ) ) {
					$legacy_customer_subset[ $k ] = $admin_config[ $k ];
				}
			}
		}

		$admin_config    = wp_parse_args( is_array( $admin_config ) ? $admin_config : array(), $admin_defaults );
		$customer_config = wp_parse_args( is_array( $customer_config ) ? $customer_config : array(), wp_parse_args( $legacy_customer_subset, $customer_defaults ) );

		// Replace placeholders in email template
		$placeholders = array(
			'{invoice_number}'  => $transaction->invoice_number,
			'{customer_name}'   => $transaction->customer_name,
			'{customer_email}'  => $transaction->customer_email,
			'{amount}'          => $transaction->amount,
			'{currency}'        => $transaction->currency,
			'{payment_status}'  => ucfirst( $transaction->payment_status ),
			'{transaction_id}'  => $transaction->transaction_id,
			'{payment_provider}' => ucfirst( $transaction->payment_provider ),
			'{description}'     => $transaction->description,
			'{site_name}'       => get_bloginfo( 'name' ),
		);

		$email_subject = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $admin_config['email_subject'] );
		$email_message = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $admin_config['email_template'] );

		// Send email to admin
		$headers = array( 'Content-Type: text/plain; charset=UTF-8' );
		wp_mail( $admin_config['admin_email'], $email_subject, $email_message, $headers );

		CoinsnapBIF_Logger::info( 'Payment notification email sent (admin)', array(
			'invoice_id' => $invoice_id,
			'admin_email' => $admin_config['admin_email'],
		) );

		// Optionally send email to customer
		$enabled_raw = isset( $customer_config['customer_email_enabled'] ) ? (string) $customer_config['customer_email_enabled'] : '';
		$send_customer = in_array( strtolower( $enabled_raw ), array( '1', 'on', 'yes', 'true' ), true ) || $enabled_raw === '';
		$customer_email = sanitize_email( (string) $transaction->customer_email );
		if ( $send_customer && ! empty( $customer_email ) && is_email( $customer_email ) ) {
			$customer_subject = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $customer_config['customer_email_subject'] );
			$customer_message = str_replace( array_keys( $placeholders ), array_values( $placeholders ), $customer_config['customer_email_template'] );
			wp_mail( $customer_email, $customer_subject, $customer_message, $headers );
			CoinsnapBIF_Logger::info( 'Payment notification email sent (customer)', array(
				'invoice_id' => $invoice_id,
				'customer_email' => $customer_email,
			) );
		} else {
			CoinsnapBIF_Logger::info( 'Customer email not sent', array(
				'invoice_id'      => $invoice_id,
				'enabled_flag'    => $enabled_raw,
				'has_customer_email' => ! empty( $customer_email ),
				'is_valid_email'  => is_email( $customer_email ),
			) );
		}
	}

	/**
	 * Get client IP address.
	 *
	 * @return string Client IP address.
	 */
	private static function get_client_ip(): string {
		$ip_keys = array( 'HTTP_CLIENT_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR' );

		foreach ( $ip_keys as $key ) {
			if ( isset( $_SERVER[ $key ] ) ) {
				// Unslash and sanitize the server variable
				$ip_string = sanitize_text_field( wp_unslash( $_SERVER[ $key ] ) );

				foreach ( explode( ',', $ip_string ) as $ip ) {
					$ip = trim( $ip );
					if ( filter_var( $ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE ) !== false ) {
						return $ip;
					}
				}
			}
		}

		// Return sanitized REMOTE_ADDR or fallback
		if ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
		}

		return '0.0.0.0';
	}
}
