<?php
/**
* Plugin Name: CodingBunny Mail SMTP
* Plugin URI: https://coding-bunny.com/mail-smtp/
* Description: Configure an SMTP server to send emails from your WordPress site. Simple, lightweight, and easy to set up.
* Version: 1.0.2
* Requires at least: 6.0
* Requires PHP: 8.0
* Author: CodingBunny
* Author URI: https://coding-bunny.com/
* License: GPLv2 or later
* License URI: https://www.gnu.org/licenses/gpl-2.0.html
* Text Domain: coding-bunny-mail-smtp
*/


if (!defined('ABSPATH')) {
	exit;
}

define('CODINGBUNNY_MAIL_SMTP_SLUG', 'coding-bunny-mail-smtp');
define('CODINGBUNNY_MAIL_SMTP_GROUP', 'codingbunny_mail_smtp_group');
define('CODINGBUNNY_MAIL_SMTP_VERSION', '1.0.2');

function codingbunny_mail_smtp_encrypt($plain) {
	$plain = (string) $plain;
	if ($plain === '') {
		return '';
	}
	if (!function_exists('openssl_encrypt') || !function_exists('random_bytes')) {
		return false;
	}
	try {
		$iv = random_bytes(16);
	} catch (Throwable $e) {
		return false;
	}
	$key = hash('sha256', wp_salt('auth'), true);
	$ct  = openssl_encrypt($plain, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
	if ($ct === false) {
		return false;
	}
	return base64_encode($iv . $ct);
}

function codingbunny_mail_smtp_decrypt($stored) {
	$stored = (string) $stored;
	if ($stored === '') return '';
	if (!function_exists('openssl_decrypt')) return '';
	$raw = base64_decode($stored, true);
	if ($raw === false || strlen($raw) < 17) return '';
	$iv  = substr($raw, 0, 16);
	$ct  = substr($raw, 16);
	$key = hash('sha256', wp_salt('auth'), true);
	$plain = openssl_decrypt($ct, 'AES-256-CBC', $key, OPENSSL_RAW_DATA, $iv);
	return ($plain !== false) ? $plain : '';
}

function codingbunny_mail_smtp_get_password_status() {
	$stored = get_option('codingbunny_mail_smtp_password', '');
	$stored = (string) $stored;
	if ($stored === '') {
		return 'none';
	}
	$decrypted = codingbunny_mail_smtp_decrypt($stored);
	if ($decrypted !== '') {
		return 'set';
	}
	return 'corrupt';
}

function codingbunny_mail_smtp_sanitize_password($new) {
	$new = trim((string) $new);
	if ($new === '') {
		return (string) get_option('codingbunny_mail_smtp_password', '');
	}
	$encrypted = codingbunny_mail_smtp_encrypt($new);
	if ($encrypted !== false && $encrypted !== '') {
		return $encrypted;
	}
	return (string) get_option('codingbunny_mail_smtp_password', '');
}

class CodingBunny_Mail_SMTP {

	public function __construct() {
		add_action('admin_menu', [$this, 'add_admin_menu']);
		add_action('admin_init', [$this, 'register_settings']);
		add_action('admin_init', [$this, 'handle_test_email']);
		add_action('phpmailer_init', [$this, 'configure_smtp']);
		add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
	}

	public function enqueue_admin_assets($hook_suffix) {
		$expected_hook = 'settings_page_' . CODINGBUNNY_MAIL_SMTP_SLUG;
		if ($hook_suffix !== $expected_hook) {
			return;
		}
		$css_rel_path = 'assets/css/cbsmtp-styles.css';
		$css_file = plugin_dir_path(__FILE__) . $css_rel_path;
		$version = file_exists($css_file) ? filemtime($css_file) : CODINGBUNNY_MAIL_SMTP_VERSION;
		wp_register_style('cbsmtp-styles', plugin_dir_url(__FILE__) . $css_rel_path, [], $version);
		wp_enqueue_style('cbsmtp-styles');
	}

	public function add_admin_menu() {
		add_options_page(
		__('CodingBunny Mail SMTP Settings', 'coding-bunny-mail-smtp'),
		__('Mail SMTP', 'coding-bunny-mail-smtp'),
		'manage_options',
		CODINGBUNNY_MAIL_SMTP_SLUG,
		[$this, 'settings_page']
	);
}

public function register_settings() {
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_enabled', 'intval');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_host', 'sanitize_text_field');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_port', 'absint');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_encryption', 'sanitize_text_field');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_auth', 'intval');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_username', 'sanitize_text_field');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_from_email', 'sanitize_email');
	register_setting(CODINGBUNNY_MAIL_SMTP_GROUP, 'codingbunny_mail_smtp_from_name', 'sanitize_text_field');
	register_setting(
	CODINGBUNNY_MAIL_SMTP_GROUP,
	'codingbunny_mail_smtp_password',
	array(
		'sanitize_callback' => 'codingbunny_mail_smtp_sanitize_password',
		'default'           => '',
	)
);
}

public function handle_test_email() {
if (!isset($_GET['page']) || $_GET['page'] !== CODINGBUNNY_MAIL_SMTP_SLUG) {
	return;
}

if (!current_user_can('manage_options')) {
	return;
}

if (!isset($_POST['codingbunny_send_test']) && !isset($_POST['codingbunny_save_test'])) {
	return;
}

check_admin_referer('codingbunny_test_action', 'codingbunny_test_nonce');

$to = sanitize_email(wp_unslash($_POST['test_email'] ?? ''));

if (!is_email($to)) {
	add_settings_error(CODINGBUNNY_MAIL_SMTP_GROUP, 'test_fail', __('Invalid email address.', 'coding-bunny-mail-smtp'), 'error');
	return;
}

if (isset($_POST['codingbunny_save_test'])) {
	update_option('codingbunny_mail_smtp_test', $to);
	add_settings_error(CODINGBUNNY_MAIL_SMTP_GROUP, 'test_saved', __('Test email address saved.', 'coding-bunny-mail-smtp'), 'success');
	return;
}

$result = wp_mail($to, __('SMTP Test Email', 'coding-bunny-mail-smtp'), __('This is a test email from CodingBunny Mail SMTP plugin!', 'coding-bunny-mail-smtp'));

if ($result) {
	update_option('codingbunny_mail_smtp_test', $to);
	add_settings_error(CODINGBUNNY_MAIL_SMTP_GROUP, 'test_success', __('Test email sent successfully!', 'coding-bunny-mail-smtp'), 'success');
} else {
	global $phpmailer;
	$error_msg = isset($phpmailer->ErrorInfo) ? ' ' . $phpmailer->ErrorInfo : '';
	add_settings_error(CODINGBUNNY_MAIL_SMTP_GROUP, 'test_fail', __('Failed to send test email.', 'coding-bunny-mail-smtp') . $error_msg, 'error');
}
}

public function settings_page() {
if (!current_user_can('manage_options')) return;
$host = get_option('codingbunny_mail_smtp_host', '');
$port = get_option('codingbunny_mail_smtp_port', 587);
$encryption = get_option('codingbunny_mail_smtp_encryption', 'tls');
$username = get_option('codingbunny_mail_smtp_username', '');
$from_email = get_option('codingbunny_mail_smtp_from_email', '');
$from_name = get_option('codingbunny_mail_smtp_from_name', '');
$enabled = (int) get_option('codingbunny_mail_smtp_enabled', 0);
$auth = (int) get_option('codingbunny_mail_smtp_auth', 1);
$logo_rel = 'assets/images/logo.svg';
$logo_path = plugin_dir_path(__FILE__) . $logo_rel;
$logo_url = file_exists($logo_path) ? plugin_dir_url(__FILE__) . $logo_rel : '';
$pwd_status = codingbunny_mail_smtp_get_password_status();
$test_email_value = get_option('codingbunny_mail_smtp_test', get_option('admin_email'));

?>
<div class="wrap cbsmtp-dashboard">
	<h1 class="screen-reader-text"><?php echo esc_html__('CodingBunny Mail SMTP', 'coding-bunny-mail-smtp'); ?></h1>
	<div class="cbsmtp-header">
		<div class="cbsmtp-header-left">
			<?php if ($logo_url) : ?>
				<img src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr__('CodingBunny logo', 'coding-bunny-mail-smtp'); ?>" class="cbsmtp-logo" role="img" aria-label="<?php echo esc_attr__('CodingBunny logo', 'coding-bunny-mail-smtp'); ?>" />
			<?php else : ?>
				<div class="cbsmtp-logo-fallback" aria-hidden="false"><?php echo esc_html__('CodingBunny', 'coding-bunny-mail-smtp'); ?></div>
			<?php endif; ?>
			<div class="cbsmtp-title">
				<p>
					<?php esc_html_e('CodingBunny Mail SMTP', 'coding-bunny-mail-smtp'); ?>
					<span class="cbsmtp-version">
						v<?php echo defined('CODINGBUNNY_MAIL_SMTP_VERSION') ? esc_html(CODINGBUNNY_MAIL_SMTP_VERSION) : ''; ?>
					</span>
				</p>
			</div>
		</div>
		<div class="cbsmtp-header-right">
			<p class="cbsmtp-doc-block">
				<span class="cbsmtp-doc-text">
					<?php esc_html_e('Need help or have questions?', 'coding-bunny-mail-smtp'); ?><br />
					<a class="cbsmtp-doc-text" href="https://coding-bunny.com/doc-category/mail-smtp/" target="_blank" rel="noopener"><?php esc_html_e('Check our documentation.', 'coding-bunny-mail-smtp'); ?></a>
				</span>
			</p>
		</div>
	</div>
	<div class="cbsmtp-section">				
		<h3><?php echo esc_html__('Settings', 'coding-bunny-mail-smtp'); ?></h3>
		<form method="post" action="options.php" aria-label="<?php echo esc_attr__('CodingBunny Mail SMTP settings form', 'coding-bunny-mail-smtp'); ?>">
			<?php settings_fields(CODINGBUNNY_MAIL_SMTP_GROUP); ?>
			<table class="form-table">
				<tr>
					<th><label for="codingbunny_mail_smtp_enabled"><?php echo esc_html__('Enable SMTP', 'coding-bunny-mail-smtp'); ?></label></th>
					<td>
						<input type="hidden" name="codingbunny_mail_smtp_enabled" value="0" />
						<label class="cbsmtp-toggle-label" for="codingbunny_mail_smtp_enabled">
							<input type="checkbox" class="cbsmtp-toggle" id="codingbunny_mail_smtp_enabled" name="codingbunny_mail_smtp_enabled" value="1" <?php checked($enabled, 1); ?> aria-checked="<?php echo $enabled ? 'true' : 'false'; ?>" aria-label="<?php echo esc_attr__('Enable SMTP', 'coding-bunny-mail-smtp'); ?>" />
							<span class="cbsmtp-slider" aria-hidden="true"></span>
							<span class="cbsmtp-toggle-text"><?php echo esc_html__('Enable to use SMTP for outgoing emails from WordPress.', 'coding-bunny-mail-smtp'); ?></span>
						</label>
					</td>
				</tr>
				<tr>
					<th><label for="codingbunny_mail_smtp_host"><?php echo esc_html__('SMTP Host', 'coding-bunny-mail-smtp'); ?></label></th>
					<td><input type="text" id="codingbunny_mail_smtp_host" name="codingbunny_mail_smtp_host" value="<?php echo esc_attr($host); ?>" class="regular-text" placeholder="smtp.example.com" aria-label="<?php echo esc_attr__('SMTP Host', 'coding-bunny-mail-smtp'); ?>"></td>
				</tr>
				<tr>
					<th><label for="codingbunny_mail_smtp_port"><?php echo esc_html__('SMTP Port', 'coding-bunny-mail-smtp'); ?></label></th>
					<td><input type="number" id="codingbunny_mail_smtp_port" name="codingbunny_mail_smtp_port" value="<?php echo esc_attr($port); ?>" class="small-text" placeholder="587" aria-label="<?php echo esc_attr__('SMTP Port', 'coding-bunny-mail-smtp'); ?>"></td>
				</tr>
				<tr>
					<th><label for="codingbunny_mail_smtp_encryption"><?php echo esc_html__('Encryption', 'coding-bunny-mail-smtp'); ?></label></th>
					<td>
						<select id="codingbunny_mail_smtp_encryption" name="codingbunny_mail_smtp_encryption" aria-label="<?php echo esc_attr__('SMTP Encryption', 'coding-bunny-mail-smtp'); ?>">
							<option value="none" <?php selected($encryption, 'none'); ?>><?php echo esc_html__('None', 'coding-bunny-mail-smtp'); ?></option>
							<option value="tls" <?php selected($encryption, 'tls'); ?>><?php echo esc_html__('TLS', 'coding-bunny-mail-smtp'); ?></option>
							<option value="ssl" <?php selected($encryption, 'ssl'); ?>><?php echo esc_html__('SSL', 'coding-bunny-mail-smtp'); ?></option>
						</select>
					</td>
				</tr>
				<tr>
					<td colspan="2" style="padding: 0"><p class="cbsmtp-info"><?php echo esc_html__('For most servers, the recommended option is TLS. If your SMTP provider offers both SSL and TLS options, we recommend using TLS.', 'coding-bunny-mail-smtp'); ?></p></td>
				</tr>
				<tr>
					<th><label for="codingbunny_mail_smtp_auth"><?php echo esc_html__('User Authentication', 'coding-bunny-mail-smtp'); ?></label></th>
					<td>
						<input type="hidden" name="codingbunny_mail_smtp_auth" value="0" />
						<label class="cbsmtp-toggle-label" for="codingbunny_mail_smtp_auth">
							<input type="checkbox" class="cbsmtp-toggle" id="codingbunny_mail_smtp_auth" name="codingbunny_mail_smtp_auth" value="1" <?php checked($auth, 1); ?> aria-checked="<?php echo $auth ? 'true' : 'false'; ?>" aria-label="<?php echo esc_attr__('SMTP requires authentication', 'coding-bunny-mail-smtp'); ?>" />
							<span class="cbsmtp-slider" aria-hidden="true"></span>
							<span class="cbsmtp-toggle-text"><?php echo esc_html__('Enable if your SMTP server requires a username and password.', 'coding-bunny-mail-smtp'); ?></span>
						</label>
					</td>
				</tr>
				<tr>
					<th><label for="codingbunny_mail_smtp_username"><?php echo esc_html__('SMTP Username', 'coding-bunny-mail-smtp'); ?></label></th>
					<td><input type="text" id="codingbunny_mail_smtp_username" name="codingbunny_mail_smtp_username" value="<?php echo esc_attr($username); ?>" class="regular-text" aria-label="<?php echo esc_attr__('SMTP Username', 'coding-bunny-mail-smtp'); ?>"></td>
				</tr>
				<tr>
					<th><label for="codingbunny_mail_smtp_password"><?php echo esc_html__('SMTP Password', 'coding-bunny-mail-smtp'); ?></label></th>
					<td>
						<input type="password" id="codingbunny_mail_smtp_password" name="codingbunny_mail_smtp_password" value="" class="regular-text" autocomplete="new-password" aria-describedby="cbsmtp-password-description cbsmtp-password-status" aria-label="<?php echo esc_attr__('SMTP Password (leave empty to keep current)', 'coding-bunny-mail-smtp'); ?>">
						<p id="cbsmtp-description" class="cbsmtp-description"><?php echo esc_html__('Leave blank to avoid overwriting other plugin settings.', 'coding-bunny-mail-smtp'); ?></p>
						<?php
						if ($pwd_status === 'none') :
							?>
							<p id="cbsmtp-password-status" class="cbsmtp-pwd-not-set" aria-live="polite"><?php echo esc_html__('No password set.', 'coding-bunny-mail-smtp'); ?></p>
							<?php
							elseif ($pwd_status === 'set') :
								?>
								<p id="cbsmtp-password-status" class="cbsmtp-pwd-set" aria-live="polite"><?php echo esc_html__('Password set correctly.', 'coding-bunny-mail-smtp'); ?></p>
								<?php
								else :
									?>
									<p id="cbsmtp-password-status" class="cbsmtp-pwd-corrupt" aria-live="polite"><?php echo esc_html__('Error — re-enter password to correct.', 'coding-bunny-mail-smtp'); ?></p>
								<?php endif; ?>
							</td>
						</tr>
						<tr>
							<td colspan="2" style="padding: 0"><p class="cbsmtp-info"><?php echo esc_html__('The password is stored in the database in encrypted form and is never displayed in plain text in the interface.', 'coding-bunny-mail-smtp'); ?></p></td>
						</tr>
						<tr>
							<th><label for="codingbunny_mail_smtp_from_email"><?php echo esc_html__('Sender Email', 'coding-bunny-mail-smtp'); ?></label></th>
							<td>
								<input type="email" id="codingbunny_mail_smtp_from_email" name="codingbunny_mail_smtp_from_email" value="<?php echo esc_attr($from_email); ?>" class="regular-text" aria-label="<?php echo esc_attr__('Sender email', 'coding-bunny-mail-smtp'); ?>">
								<p class="cbsmtp-description" id="cbsmtp-from-email-desc"><?php echo esc_html__('The email address from which emails are sent. Leave blank to avoid overwriting the settings of other plugins.', 'coding-bunny-mail-smtp'); ?></p>
							</td>
						</tr>
						<tr>
							<td colspan="2" style="padding: 0"><p class="cbsmtp-warning"><?php echo esc_html__('Some providers (e.g., Gmail, Yahoo) may block or modify messages if the sender address does not belong to the server domain or if SPF/DMARC are not configured correctly. For greater reliability, use an address from the site domain or a dedicated SMTP service.', 'coding-bunny-mail-smtp'); ?></p></td>
						</tr>
						<tr>
							<th><label for="codingbunny_mail_smtp_from_name"><?php echo esc_html__('Sender Name', 'coding-bunny-mail-smtp'); ?></label></th>
							<td>
								<input type="text" id="codingbunny_mail_smtp_from_name" name="codingbunny_mail_smtp_from_name" value="<?php echo esc_attr($from_name); ?>" class="regular-text" aria-label="<?php echo esc_attr__('Sender name', 'coding-bunny-mail-smtp'); ?>">
								<p class="cbsmtp-description" id="cbsmtp-from-name-desc"><?php echo esc_html__('The name from which emails are sent.', 'coding-bunny-mail-smtp'); ?></p>
							</td>
						</tr>
					</table>
					<?php submit_button(esc_html__('Save Settings', 'coding-bunny-mail-smtp'), 'primary', 'submit', true, array('aria-label' => esc_attr__('Save CodingBunny Mail SMTP settings', 'coding-bunny-mail-smtp'))); ?>
				</form>
			</div>
			<br>
			<div class="cbsmtp-section">
				<h3><?php echo esc_html__('Test Email', 'coding-bunny-mail-smtp'); ?></h3>
				<form method="post" action="<?php echo esc_url(admin_url('options-general.php?page=' . CODINGBUNNY_MAIL_SMTP_SLUG)); ?>" aria-label="<?php echo esc_attr__('Send test email form', 'coding-bunny-mail-smtp'); ?>">
					<?php wp_nonce_field('codingbunny_test_action', 'codingbunny_test_nonce'); ?>
					<table class="form-table">
						<tr>
							<td colspan="2" style="padding: 0"><p class="cbsmtp-info"><?php echo esc_html__('Send a test email to verify that the settings are correct.', 'coding-bunny-mail-smtp'); ?></p></td>
						</tr>
						<tr>
							<th><label for="test_email"><?php echo esc_html__('Send test email to', 'coding-bunny-mail-smtp'); ?></label></th>
							<td><input type="email" id="test_email" name="test_email" value="<?php echo esc_attr($test_email_value); ?>" class="regular-text" aria-label="<?php echo esc_attr__('Test email recipient', 'coding-bunny-mail-smtp'); ?>"><input type="submit" name="codingbunny_save_test" class="button button-primary" style="margin-left:10px;" value="<?php echo esc_attr__('Save Mail', 'coding-bunny-mail-smtp'); ?>" aria-label="<?php echo esc_attr__('Save test mail address', 'coding-bunny-mail-smtp'); ?>"></td>
						</tr>
					</table>
					<p class="submit">
						<input type="submit" name="codingbunny_send_test" class="button button-primary" value="<?php echo esc_attr__('Send Test Email', 'coding-bunny-mail-smtp'); ?>" aria-label="<?php echo esc_attr__('Send test email', 'coding-bunny-mail-smtp'); ?>">
					</p>
				</form>
			</div>
		</div>
		<?php
	}

	public function configure_smtp($phpmailer) {
		$enabled = (int) get_option('codingbunny_mail_smtp_enabled', 0);
		$host    = trim((string) get_option('codingbunny_mail_smtp_host', ''));
		if (!$enabled || $host === '') return;
		$phpmailer->isSMTP();
		$phpmailer->Host = $host;
		$phpmailer->Port = (int) get_option('codingbunny_mail_smtp_port', 587);
		$auth = (int) get_option('codingbunny_mail_smtp_auth', 1);
		$phpmailer->SMTPAuth = (bool) $auth;
		if ($auth) {
			$phpmailer->Username = (string) get_option('codingbunny_mail_smtp_username', '');
			$pw = codingbunny_mail_smtp_decrypt((string) get_option('codingbunny_mail_smtp_password', ''));
			$phpmailer->Password = ($pw !== '') ? $pw : '';
		} else {
			$phpmailer->Username = '';
			$phpmailer->Password = '';
		}
		$encryption = get_option('codingbunny_mail_smtp_encryption', 'tls');
		if (!in_array($encryption, ['none', 'tls', 'ssl'], true)) $encryption = 'tls';
		$phpmailer->SMTPSecure = ($encryption !== 'none') ? $encryption : '';
		$from_email = sanitize_email((string) get_option('codingbunny_mail_smtp_from_email', ''));
		$from_name  = sanitize_text_field((string) get_option('codingbunny_mail_smtp_from_name', ''));
		if ($from_email && is_email($from_email)) {
			$name = ($from_name !== '') ? $from_name : 'WordPress';
			$phpmailer->setFrom($from_email, $name);
			$phpmailer->Sender   = $from_email;
			$phpmailer->From     = $from_email;
			$phpmailer->FromName = $name;
		} elseif ($from_name !== '') {
    		$phpmailer->FromName = $from_name;
		}
	}
}

new CodingBunny_Mail_SMTP();

function codingbunny_mail_smtp_add_settings_link($links) {
	$settings_link = '<a href="' . esc_url(admin_url('options-general.php?page=' . CODINGBUNNY_MAIL_SMTP_SLUG)) . '">' . esc_html__('Settings', 'coding-bunny-mail-smtp') . '</a>';
	array_unshift($links, $settings_link);
	return $links;
}
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'codingbunny_mail_smtp_add_settings_link');

register_uninstall_hook(__FILE__, 'codingbunny_mail_smtp_uninstall');

function codingbunny_mail_smtp_uninstall() {
	$opts = [
		'codingbunny_mail_smtp_host', 'codingbunny_mail_smtp_port', 'codingbunny_mail_smtp_encryption',
		'codingbunny_mail_smtp_username', 'codingbunny_mail_smtp_password', 'codingbunny_mail_smtp_from_name',
		'codingbunny_mail_smtp_from_email', 'codingbunny_mail_smtp_enabled', 'codingbunny_mail_smtp_auth',
		'codingbunny_mail_smtp_test'
	];
	foreach ($opts as $opt) {
		delete_option($opt);
	}
}

register_activation_hook(__FILE__, 'codingbunny_mail_smtp_activate');

function codingbunny_mail_smtp_activate() {
	if (get_option('codingbunny_mail_smtp_password') === false) {
		add_option('codingbunny_mail_smtp_password', '', '', 'no');
	}
}
?>