<?php
/**
 * Plugin Name: COD Control
 * Description: Adds an admin list of customers (including guests) with fields Name, Email, City, and "Received COD" (Yes/No). If set to No for an email, the Cash on Delivery payment method is disabled for that customer on checkout/order-pay.
 * Author: changlee
 * Requires Plugins: woocommerce
 * Version: 1.0.4
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * Text Domain: cod-control
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

class COD_Control_Woo {
	const VERSION = '1.0.4';
	const TABLE   = 'woo_cod_control';

	public function __construct() {
		register_activation_hook( __FILE__, [ __CLASS__, 'activate' ] );

		add_action( 'woocommerce_checkout_update_order_review', function() {
			if ( isset($_POST['nonce']) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'codco_action' ) ) {
				WC()->session->set( 'billing_email', isset($_POST['billing_email']) ? sanitize_email( wp_unslash($_POST['billing_email']) ) : '' );
			}
		});

		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );
		
		add_action( 'admin_menu', [ $this, 'codco_register_admin_menu' ] );
		add_action( 'admin_post_wcc_save_entry', [ $this, 'codco_handle_save_entry' ] );
		add_action( 'admin_post_wcc_delete_entry', [ $this, 'codco_handle_delete_entry' ] );
		add_action( 'admin_post_wcc_scan_orders', [ $this, 'codco_handle_scan_orders' ] );

		// Disable COD when needed on checkout and order-pay pages
		add_filter( 'woocommerce_available_payment_gateways', [ $this, 'codco_filter_gateways' ], 10, 1 );
		
		add_action( 'wp_enqueue_scripts', [ $this, 'codco_enqueue_scripts' ] );

		add_action('wp_ajax_codco_handle_email_change',[ $this,'codco_handle_email_change_callback' ]);
		add_action('wp_ajax_nopriv_codco_handle_email_change', [ $this, 'codco_handle_email_change_callback']);
	}

	/**
	 * Create custom table on activation
	 */
	public static function activate() {
		global $wpdb;
		$table = esc_sql($wpdb->prefix . self::TABLE);

		$charset_collate = $wpdb->get_charset_collate();
		$sql = "CREATE TABLE IF NOT EXISTS `$table` (
			id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
			name VARCHAR(191) NULL,
			email VARCHAR(191) NOT NULL,
			city VARCHAR(191) NULL,
			received_cod TINYINT(1) NOT NULL DEFAULT 1, -- 1 = Yes (allowed), 0 = No (block COD)
			created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
			updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			UNIQUE KEY email (email)
		) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );
	}

	public function codco_enqueue_scripts() {
		wp_enqueue_style( 'codco-style', plugin_dir_url( __FILE__ ) . 'assets/css/codco-style.css',array(),'1.0.0' );
		wp_enqueue_script( 'codco-script', plugin_dir_url( __FILE__ ) . 'assets/js/codco-script.js', ['jquery'], '1.0.1', true );

		wp_localize_script( 'codco-script', 'codco_vars', [
			'ajaxurl' => esc_url( admin_url( 'admin-ajax.php' ) ),
			'nonce'   => esc_js( wp_create_nonce( 'codco_email_nonce' ) ),
			'loader' => esc_url( plugin_dir_url( __FILE__ ) . 'ajax-loader.gif' )
		]);
	}

	public function enqueue_admin_assets( $hook ) {
		if ( $hook === 'toplevel_page_cod-control' ) {
			wp_enqueue_style( 'codco-style-admin', plugin_dir_url( __FILE__ ) . 'assets/css/admin-codco-style.css', [], self::VERSION );
			wp_enqueue_script( 'codco-script-admin', plugin_dir_url( __FILE__ ) . 'assets/js/admin-codco-script.js', [ 'jquery' ], self::VERSION, true );
		}
	}


	public function codco_handle_email_change_callback() {
		// Verify nonce for security
		if ( ! isset( $_POST['security'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['security'] ) ), 'codco_email_nonce' ) ){
			wp_send_json_error( 'Nonce verification failed.' );
		}

		if (isset($_POST['new_email'])) {
			$new_email = sanitize_email($_POST['new_email']);
			$is_blocked = $this->is_cod_blocked_for_email( $new_email );
			
			if ( $is_blocked == true ) {
				$gateways = WC()->payment_gateways()->payment_gateways();	
				unset( $gateways['cod'] );
				wp_send_json_success('success');
			}
			else{
				wp_send_json_error('fail!');
			}

		} else {
			wp_send_json_error('fail!');
		}
	}

	/**
	 * Admin Menu
	 */
	public function codco_register_admin_menu() {
		add_menu_page(
			__( 'COD Control', 'cod-control' ),
			__( 'COD Control', 'cod-control' ),
			'manage_woocommerce',
			'cod-control',
			[ $this, 'render_admin_page' ],
			'dashicons-money-alt',
			58
		);
	}

	/**
	 * Fetch entries with pagination and search
	 */
	private function get_entries( $paged = 1, $per_page = 25, $search = '' ) {
		global $wpdb;
		$table = esc_sql($wpdb->prefix . self::TABLE);
		$offset = ( max( 1, (int) $paged ) - 1 ) * $per_page;

		$where = 'WHERE 1=1';
		$params = [];
		if ( $search ) {
			$where .= ' AND (name LIKE %s OR email LIKE %s OR city LIKE %s)';
			$like = '%' . $wpdb->esc_like( $search ) . '%';
			$params[] = $like; $params[] = $like; $params[] = $like;
		}

		$params[] = $per_page;
		$params[] = $offset;

		$prepared_query  = $wpdb->prepare("SELECT SQL_CALC_FOUND_ROWS * FROM $table $where ORDER BY updated_at DESC LIMIT %d OFFSET %d", $params);
		$rows = $wpdb->get_results( $prepared_query  );
		$total = (int) $wpdb->get_var( 'SELECT FOUND_ROWS()' );
		return [ $rows, $total ];
	}

	/**
	 * Render Admin Page
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			//wp_die( __( 'You do not have permission to access this page.', 'cod-control' ) );
			return false;
		}

		$paged   = isset( $_GET['paged'] ) ? max( 1, (int) $_GET['paged'] ) : 1;
		$perpage = 25;
		$search = '';
		if ( isset($_GET['codco_nonce']) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['codco_nonce'] ) ), 'codco_action' ) ) {
			$search = isset($_GET['s']) ? sanitize_text_field( wp_unslash($_GET['s']) ) : '';
		}
		list( $rows, $total ) = $this->get_entries( $paged, $perpage, $search );
		$total_pages = max( 1, (int) ceil( $total / $perpage ) );
		$base_url = admin_url( 'admin.php?page=cod-control' );
		?>
		
		<div class="wrap">
			<h1><?php esc_html_e( 'COD Control', 'cod-control' ); ?></h1>

			<form method="get" style="margin:12px 0;">
				<input type="hidden" name="page" value="cod-control" />
				<?php wp_nonce_field( 'codco_action', 'codco_nonce' ); ?>
				<input type="search" name="s" value="<?php echo esc_attr( $search ); ?>" placeholder="<?php esc_attr_e( 'Search name, email, city…', 'cod-control' ); ?>" />
				<button class="button"><?php esc_html_e( 'Search', 'cod-control' ); ?></button>
			</form>

			<h2 class="hide"><?php esc_html_e( 'Add / Update Entry', 'cod-control' ); ?></h2>
			<form class="hide" method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-bottom:24px;">
				<?php wp_nonce_field( 'wcc_save_entry', 'wcc_nonce' ); ?>
				<input type="hidden" name="action" value="wcc_save_entry" />
				<table class="form-table" role="presentation">
					<tr>
						<th><label for="wcc_name"><?php esc_html_e( 'Name', 'cod-control' ); ?></label></th>
						<td><input name="name" id="wcc_name" type="text" class="regular-text" /></td>
					</tr>
					<tr>
						<th><label for="wcc_email"><?php esc_html_e( 'Email *', 'cod-control' ); ?></label></th>
						<td><input name="email" id="wcc_email" type="email" class="regular-text" required /></td>
					</tr>
					<tr>
						<th><label for="wcc_city"><?php esc_html_e( 'City', 'cod-control' ); ?></label></th>
						<td><input name="city" id="wcc_city" type="text" class="regular-text" /></td>
					</tr>
					<tr>
						<th><?php esc_html_e( 'Received COD', 'cod-control' ); ?></th>
						<td>
							<label><input type="radio" name="received_cod" value="1" checked> <?php esc_html_e( 'Yes (COD allowed)', 'cod-control' ); ?></label>
							&nbsp;&nbsp;
							<label><input type="radio" name="received_cod" value="0"> <?php esc_html_e( 'No (Disable COD)', 'cod-control' ); ?></label>
						</td>
					</tr>
				</table>
				<p><button type="submit" class="button button-primary"><?php esc_html_e( 'Save', 'cod-control' ); ?></button></p>
			</form>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin: 16px 0 24px;">
				<?php wp_nonce_field( 'wcc_scan_orders', 'wcc_scan_nonce' ); ?>
				<input type="hidden" name="action" value="wcc_scan_orders" />
				<p>
					<button class="button"><?php esc_html_e( 'Scan recent orders', 'cod-control' ); ?></button>
					<br /><br />
					<span class="description"><?php esc_html_e( 'Imports unique emails with name/city from billing details. All imported entries default to Received COD = Yes (allowed). You can set No for those who should NOT get COD.', 'cod-control' ); ?></span>
				</p>
			</form>

			<h2><?php esc_html_e( 'Existing Users', 'cod-control' ); ?></h2>
			<table class="widefat fixed striped">
				<thead>
					<tr>
						<th><?php esc_html_e( 'Name', 'cod-control' ); ?></th>
						<th><?php esc_html_e( 'Email', 'cod-control' ); ?></th>
						<th><?php esc_html_e( 'City', 'cod-control' ); ?></th>
						<th><?php esc_html_e( 'Received COD', 'cod-control' ); ?></th>
						<th><?php esc_html_e( 'Updated', 'cod-control' ); ?></th>
						<th><?php esc_html_e( 'Actions', 'cod-control' ); ?></th>
					</tr>
				</thead>
				<tbody>
					<?php if ( empty( $rows ) ) : ?>
						<tr><td colspan="6"><?php esc_html_e( 'No entries found.', 'cod-control' ); ?></td></tr>
					<?php else: foreach ( $rows as $r ) : ?>
						<tr>
							<td><?php echo esc_html( $r->name ); ?></td>
							<td><?php echo esc_html( $r->email ); ?></td>
							<td><?php echo esc_html( $r->city ); ?></td>
							<td><?php echo $r->received_cod ? '<span class="dashicons dashicons-yes"></span> ' . esc_html__( 'Yes', 'cod-control' ) : '<span class="dashicons dashicons-no"></span> ' . esc_html__( 'No', 'cod-control' ); ?></td>
							<td><?php echo esc_html( $r->updated_at ); ?></td>
							<td>
								<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline-block;">
									<?php wp_nonce_field( 'wcc_save_entry', 'wcc_nonce' ); ?>
									<input type="hidden" name="action" value="wcc_save_entry" />
									<input type="hidden" name="email" value="<?php echo esc_attr( $r->email ); ?>" />
									<input type="hidden" name="name" value="<?php echo esc_attr( $r->name ); ?>" />
									<input type="hidden" name="city" value="<?php echo esc_attr( $r->city ); ?>" />
									<input type="hidden" name="received_cod" value="<?php echo $r->received_cod ? 0 : 1; ?>" />
									<button class="button" title="<?php echo $r->received_cod ? esc_attr__( 'Set to No (disable COD)', 'cod-control' ) : esc_attr__( 'Set to Yes (allow COD)', 'cod-control' ); ?>"><?php echo $r->received_cod ? esc_html__( 'Set No', 'cod-control' ) : esc_html__( 'Set Yes', 'cod-control' ); ?></button>
								</form>
								<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:inline-block;margin-left:6px;" onsubmit="return confirm('Delete this entry?');">
									<?php wp_nonce_field( 'wcc_delete_entry', 'wcc_del_nonce' ); ?>
									<input type="hidden" name="action" value="wcc_delete_entry" />
									<input type="hidden" name="email" value="<?php echo esc_attr( $r->email ); ?>" />
									<button class="button-link-delete">&times; <?php esc_html_e( 'Delete', 'cod-control' ); ?></button>
								</form>
							</td>
						</tr>
					<?php endforeach; endif; ?>
				</tbody>
			</table>

			<?php if ( $total_pages > 1 ) : ?>
				<div class="tablenav"><div class="tablenav-pages">
					<?php
					for ( $i = 1; $i <= $total_pages; $i++ ) {
						$url = add_query_arg( [ 'paged' => $i ], $base_url );
						if ( $search ) { $url = add_query_arg( [ 's' => $search ], $url ); }
						echo $i === $paged ? '<span class="tablenav-pages-navspan">' . esc_url($i) . '</span>' : '<a class="page-numbers" href="' . esc_url( $url ) . '">' .esc_url($i) . '</a>';
					}
					?>
				</div></div>
			<?php endif; ?>

		</div>
		<?php
	}

	/**
	 * Create / Update entry
	 */
	public function codco_handle_save_entry() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) { wp_die( 'Unauthorized' ); }
		check_admin_referer( 'wcc_save_entry', 'wcc_nonce' );

		$name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		$city = isset( $_POST['city'] ) ? sanitize_text_field( wp_unslash( $_POST['city'] ) ) : '';
		$received = isset( $_POST['received_cod'] ) ? (int) $_POST['received_cod'] : 1;

		if ( ! is_email( $email ) ) {
			wp_redirect( add_query_arg( [ 'page' => 'cod-control', 'wcc_msg' => 'invalid_email' ], admin_url( 'admin.php' ) ) );
			exit;
		}

		global $wpdb; $table = esc_sql($wpdb->prefix . self::TABLE);
		$existing_id = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table WHERE email = %s", $email ) );
		$data = [ 'name' => $name, 'email' => $email, 'city' => $city, 'received_cod' => $received ];
		$formats = [ '%s','%s','%s','%d' ];
		if ( $existing_id ) {
			$wpdb->update( $table, $data, [ 'id' => $existing_id ], $formats, [ '%d' ] );
		} else {
			$wpdb->insert( $table, $data, $formats );
		}

		wp_redirect( admin_url( 'admin.php?page=cod-control' ) );
		exit;
	}

	/**
	 * Delete entry
	 */
	public function codco_handle_delete_entry() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) { wp_die( 'Unauthorized' ); }
		check_admin_referer( 'wcc_delete_entry', 'wcc_del_nonce' );

		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		if ( is_email( $email ) ) {
			global $wpdb; $table = esc_sql($wpdb->prefix . self::TABLE);
			$wpdb->delete( $table, [ 'email' => $email ], [ '%s' ] );
		}
		wp_redirect( admin_url( 'admin.php?page=cod-control' ) );
		exit;
	}


	/**
	 * Scan recent orders and import unique customer emails
	 */
	public function codco_handle_scan_orders() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) { wp_die( 'Unauthorized' ); }
		check_admin_referer( 'wcc_scan_orders', 'wcc_scan_nonce' );

		if ( ! class_exists( 'WC_Order_Query' ) ) { wp_redirect( admin_url( 'admin.php?page=cod-control' ) ); exit; }

		$args = [
			'limit'   => 500,
			'orderby' => 'date',
			'order'   => 'DESC',
			'return'  => 'objects',
			'status'  => array_keys( wc_get_order_statuses() ),
		];
		$orders = wc_get_orders( $args );
		$imported = 0;

		global $wpdb; $table = esc_sql($wpdb->prefix . self::TABLE);
		foreach ( $orders as $order ) {
			$email = $order->get_billing_email();
			if ( ! is_email( $email ) ) { continue; }

			$name = trim( $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() );
			$city = $order->get_billing_city();

			$exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM $table WHERE email = %s", $email ) );
			if ( ! $exists ) {
				$wpdb->insert( $table, [
					'name' => $name,
					'email' => $email,
					'city' => $city,
					'received_cod' => 1, // default allow
				], [ '%s','%s','%s','%d' ] );
				$imported++;
			}
		}

		wp_redirect( add_query_arg( [ 'page' => 'cod-control', 'imported' => $imported ], admin_url( 'admin.php' ) ) );
		exit;
	}

	/**
	 * Determine the email currently in context (logged-in user, checkout form, or order-pay page)
	 */
	private function get_context_email() {

		// Verify nonce before processing input
		if ( isset( $_POST['codco_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['codco_nonce'] ) ), 'codco_action' ) ) {
			if ( isset( $_POST['billing_email'] ) ) {
				return strtolower( sanitize_email( wp_unslash( $_POST['billing_email'] ) ) );
			}
		}
		return '';

		// Logged-in user
		if ( is_user_logged_in() ) {
			$user = wp_get_current_user();
			if ( $user && $user->user_email ) {
				return strtolower( sanitize_email( $user->user_email ) );
			}
		}

		// Checkout billing email (works during AJAX refresh of payment gateways)
		if ( ! empty( $_POST['billing_email'] ) ) {
			return strtolower( sanitize_email( wp_unslash( $_POST['billing_email'] ) ) );
		}

		// Order-pay page
		if ( is_wc_endpoint_url( 'order-pay' ) ) {
			$order_id = absint( get_query_var( 'order-pay' ) );
			if ( $order_id ) {
				$order = wc_get_order( $order_id );
				if ( $order ) {
					return strtolower( sanitize_email( $order->get_billing_email() ) );
				}
			}
		}

		return '';
	}


	/**
	 * Check if COD should be disabled for a given email
	 */
	private function is_cod_blocked_for_email( $email ) {
		if ( ! $email || ! is_email( $email ) ) { return false; }
		global $wpdb; $table = esc_sql($wpdb->prefix . self::TABLE);
		$blocked = (int) $wpdb->get_var( $wpdb->prepare( "SELECT CASE WHEN received_cod = 0 THEN 1 ELSE 0 END FROM $table WHERE email = %s", $email ) );
		return (bool) $blocked;
	}

	/**
	 * Filter available payment gateways
	 */
	function codco_filter_gateways( $gateways ) {
    $email = $this->get_context_email();

    if ( ! empty( $email ) ) {
        // DEBUG: check what email is coming in
        error_log("Checkout Email: " . $email);

        if ( $this->is_cod_blocked_for_email( $email ) ) {
            if ( isset( $gateways['cod'] ) ) {
                unset( $gateways['cod'] );

                // Add notice for customer
                add_action( 'woocommerce_checkout_before_customer_details', function() use ( $email ) {
                    wc_print_notice(
                        sprintf(
							/* translators: %s: search term */
                            esc_html__( 'Cash on Delivery is not available for %s. Please choose another payment method.', 'cod-control' ),
                            esc_html( $email )
                        ),
                        'error'
                    );
                });
            }
        }
    }

    return $gateways;
}

}

new COD_Control_Woo();
