<?php

namespace CocktailRecipes\Shortcodes;

use CocktailRecipes\Plugin;
use CocktailRecipes\Admin\GeneralSettings;
use CocktailRecipes\Recipes\Recipe;
use CocktailRecipes\Core\Base\Shortcode;
use CocktailRecipes\Core\Helpers\Cache;
use CocktailRecipes\Core\Helpers\Context;
use CocktailRecipes\Core\Helpers\LangCache;
use CocktailRecipes\Core\Helpers\Locale;
use CocktailRecipes\Core\Helpers\Output;

class CocktailShortcode extends Shortcode
{
    // Requires raw content from WordPress
    public const RAW_CONTENT = true;

    // Cache keys
    public const RECIPE_CACHE_PREFIX = 'r-';
    public const UNIT_BTN_CACHE_NAME = 'unit_btns';

    // Namespace for unit classes
    private const UNITS_NS = 'CocktailRecipes\Recipes\Values\Units\\';

    // View parameter settings
    private const VIEW_PARAM   = 'view';
    private const VIEW_OPTIONS = ['short', 'full'];

    // Serving size settings
    private const SERVINGS_PARAM = 'servings';
    private const SERVINGS_BTNS  = [1, 2, 3];
    private const SERVINGS_MAX   = 10;

    // Ingredient units settings
    private const UNITS_PARAM   = 'units';
    private const COUNTRY_UNITS = [
        '_US' => 'oz',
        '_LR' => 'oz',
        '_MM' => 'oz'
    ];
    private const UNITS_MAP = [
        'oz' => 'Ounce',
        'ml' => 'Milliliter',
        'cl' => 'Centiliter',
        'cc' => 'CubicCentimeter'
    ];

    // Rendering configuration
    private static ?bool $enabled = null;
    private static array $config;

    // Number of recipes on page
    private static int $recipeCount = 0;

    // Recipe instance and metadata
    private string $recipeId;
    private Recipe $recipe;
    private float  $loadTime;
    private bool   $fromCache = false;

    // View parameters
    private string $paramSfx;
    private string $view;
    private int    $servings;
    private string $units;

    protected function init(): void
    {
        if (self::$enabled === null) $this->initConfig();
        $this->recipeId = md5($this->content);
    }

    private function initConfig(): void
    {
        $onListPage = Context::isPostList();
        if (self::$enabled = !$onListPage || Plugin::isAllowedOnListPage()) {
            $settings = GeneralSettings::all();
            $view = $settings->get($onListPage ? 'default_list_view' : 'default_post_view');
            if ($fixedView = substr($view, 0, 5) == 'only-') $view = substr($view, 5);
            $unitsList = ['oz', 'ml'];
            if ($cl = $settings->get('enable_cl')) $unitsList[] = 'cl';
            if ($cc = $settings->get('enable_cc')) $unitsList[] = 'cc';
            $units = $settings->get('default_units');
            if ($units == 'auto' || ($units == 'cl' && !$cl) || ($units == 'cc' && !$cc)) {
                $units = self::COUNTRY_UNITS[substr(Locale::code(), -3)] ?? 'ml';
            }
            self::$config = [
                'label_type' => $settings->get('control_labels'),   // always | desktop | never
                'default_view' => $view,                            // short | full
                'default_units' => $units,                          // oz | ml | cl | cc
                'units_avail' => array_flip($unitsList),
                'show_controls' => $controls = !$onListPage || !$fixedView,
                'show_view_btn' => !$fixedView,
                'show_servings' => $settings->get('enable_servings'),
                'show_admin_info' => !$onListPage,
                'show_errors' => $settings->get('show_errors'),
                'show_metadata' => $settings->get('show_metadata') || (defined('WP_DEBUG') && WP_DEBUG),
                'show_attribution' => $settings->get('show_attribution'),
                'card_link' => !$controls && ($linkType = $settings->get('list_view_link')) == 'card',
                'post_link' => !$controls && $linkType && $linkType != 'card' ? $linkType : '',
                'post_link_pos' => $settings->get('list_view_link_pos'),
                'post_link_text' => trim($settings->get('list_view_link_text')) ?: GeneralSettings::defaultLinkText(),
            ];
        }
    }

    protected function renderOnce()
    {
        return $this->tag . $this->recipeId;
    }

    protected function output(): string
    {
        if (!self::$enabled) return '';
        $this->loadRecipe()->setParams();
        return Output::render('cocktail-recipe', self::$config + [
            'param_sfx'    => $this->paramSfx,
            'ingredients'  => $this->recipe->ingredientsForDisplay([
                'servings' => $this->servings,
                'units'    => self::UNITS_MAP[$this->units],
            ]),
            'summary'        => $this->recipe->summary(),
            'instructions'   => $this->recipe->steps(),
            'notes'          => $this->recipe->notes(),
            'sources'        => $this->recipe->sources(),
            'view'           => $this->view,
            'view_full_btn'  => $show = __('Full Instructions', 'cocktail-recipes') . " \u{25BE}",
            'view_short_btn' => $hide = __('Summary', 'cocktail-recipes') . " \u{25B4}",
            'view_btn'       => $this->view == 'full' ? $hide : $show,
            'servings'       => $this->servings,
            'servings_btns'  => self::SERVINGS_BTNS,
            'units'          => $this->units,
            'unit_btns'      => $this->unitLabels(),
            'load_time'      => $this->loadTime,
            'from_cache'     => $this->fromCache,
            'cache_age'      => $this->recipe->parsedAt ? time() - $this->recipe->parsedAt : null,
            'errors'         => $this->recipe->errors,
        ]);
    }

    private function loadRecipe(): self
    {
        $startTime = microtime(true);
        $cacheKey  = self::RECIPE_CACHE_PREFIX . $this->recipeId;
        $cacheData = Cache::get($cacheKey);
        if ($cacheData && ($recipe = @unserialize($cacheData)) instanceof Recipe) {
            $this->recipe = $recipe;
            $this->fromCache = true;
        } else {
            $this->recipe = new Recipe($this->content);
            Cache::set($cacheKey, serialize($this->recipe));
        }
        $this->loadTime = microtime(true) - $startTime;
        return $this;
    }

    private function setParams(): self
    {
        $this->paramSfx = (++self::$recipeCount > 1) ? (string) self::$recipeCount : '';
        $this->view = self::$config['default_view'];
        $this->servings = 1;
        $this->units = self::$config['default_units'];
        if (!self::$config['show_controls']) return $this;

        if (self::$config['show_view_btn']) {
            $param = self::VIEW_PARAM . $this->paramSfx;
            // is_string() used to prevent array injection; e.g. view[]=x
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only display parameter
            $view = isset($_GET[$param]) && is_string($_GET[$param]) ? sanitize_text_field(wp_unslash($_GET[$param])) : '';
            if ($view && in_array($view, self::VIEW_OPTIONS, true)) $this->view = $view;
        }

        if (self::$config['show_servings']) {
            $param = self::SERVINGS_PARAM . $this->paramSfx;
            // is_string() used to prevent array injection; e.g. servings[]=x
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only display parameter
            $servings = isset($_GET[$param]) && is_string($_GET[$param]) ? absint(wp_unslash($_GET[$param])): 0;
            if ($servings >= 1 && $servings <= self::SERVINGS_MAX) $this->servings = $servings;
        }

        $param = self::UNITS_PARAM . $this->paramSfx;
        // is_string() used to prevent array injection; e.g. units[]=x
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- read-only display parameter
        $units = isset($_GET[$param]) && is_string($_GET[$param]) ? sanitize_key(wp_unslash($_GET[$param])) : '';
        if ($units && isset(self::$config['units_avail'][$units])) $this->units = $units;

        return $this;
    }

    private function unitLabels(): array
    {
        $labels = LangCache::get(self::UNIT_BTN_CACHE_NAME);
        if ($labels && is_array($labels)) return $labels;
        $labels = [];
        foreach (self::UNITS_MAP as $unit => $className) {
            $class = self::UNITS_NS . $className;
            $labels[$unit] = (new $class())->symbol;
        }
        LangCache::set(self::UNIT_BTN_CACHE_NAME, $labels);
        return $labels;
    }
}
