<?php

namespace CocktailRecipes\Recipes\Values\Units;

use CocktailRecipes\Recipes\RecipeEntity;
use CocktailRecipes\Core\Helpers\Number;

/**
 * @property string     $name       read-only localized name
 * @property ?string    $symbol     read-only localized symbol
 */
abstract class Unit extends RecipeEntity
{
    // Valid amounts allowed (extend in child as needed; set to null for none)
    protected const MIN_AMOUNT = 1;
    protected const MAX_AMOUNT = null;

    // Valid ranges (extend in child as needed; set null for none)
    protected const RANGE_WIDTH_RATIO = 2.0;    // ratio of lower value to range width to allow
    protected const RANGE_WIDTH_FLOOR = null;   // minimum range width to allow

    // 5% validation and conversion threshould tolerances
    protected const MIN_THRESHOLD = 0.95;
    protected const MAX_THRESHOLD = 1.05;

    // Singleton instances
    private static array $instances = [];

    /** Localized unit symbol (optional) */
    protected ?string $symbol = null;

    /** Get unit instance */
    public static function get(string $name): ?self
    {
        if (!($instance = self::$instances[$name] ?? null)) {
            $class = __NAMESPACE__ . '\\' . $name;
            if (!class_exists($class)) return null;
            $instance = self::$instances[$name] = new $class();
        }
        return $instance;
    }

    /**
     * True if value is valid
     *
     * @param float|array{float,float}
     */
    final public function isValid($value): bool
    {
        if (!is_array($value)) return $this->isValidAmount($value);
        [$min, $max] = $value;
        if (!$this->isValidAmount($min) || !$this->isValidAmount($max)) return false;
        $width = $max - $min;
        $minWidth = static::MIN_AMOUNT ?? 0;
        $maxWidth = static::RANGE_WIDTH_FLOOR ?? 0;
        if (static::RANGE_WIDTH_RATIO !== null) {
            $maxWidth = max($maxWidth, $min * (static::RANGE_WIDTH_RATIO - 1));
        }
        if ($maxWidth > 0) {
            if ($width > $maxWidth * static::MAX_THRESHOLD) return false;
            if ($minWidth > 0) $minWidth = min($minWidth, $maxWidth);
        }
        if ($minWidth > 0 && $width < $minWidth * static::MIN_THRESHOLD) return false;
        return true;
    }

    /** True if numeric amount for unit type is valid */
    protected function isValidAmount(float $value): bool
    {
        if (
            static::MIN_AMOUNT !== null
            && $value < static::MIN_AMOUNT * static::MIN_THRESHOLD
        ) return false;
        if (
            static::MAX_AMOUNT !== null
            && $value > static::MAX_AMOUNT * static::MAX_THRESHOLD
        ) return false;
        return true;
    }

    /**
     * Normalize quantity and unit
     *
     * @param  float|array{float,float}
     * @return array{float|array{float,float}, Unit}
     */
    public function normalize($value): array
    {
        return [$value, $this];
    }

    /** Format a number in HTML with fractions or as a decimal */
    public function htmlNumber(float $num): string
    {
        return Number::withFractions($num);
    }

    /**
     * Localized unit identifiers in recipes
     *
     * Can be a comma delimited string. Spaces after commas are okay
     * and will be trimmed. Entries for symbols, singular and plural
     * forms should be included. Abbreviations should be listed with
     * a trailing period, and will be matched with or without the
     * period. Include any Unicode characters in `\u{...}` form.
     *
     * Example: __('oz.,ounce,ounces', 'cocktail-recipes')
     *    This allows `oz.`, `oz`, 'ounce' or 'ounces` to be used
     *    after a quantity for an entity like an ingredient.
     *
     * @return  string  comma-delimited localized string
     *
     * @see upperSymbols() to support case sensitive variants
     */
    public static function identifiers(): string
    {
        return '';
    }

    /**
     * Uppercase-specific symbols
     *
     * Identifiers are normally case insensitive. This function allows
     * special use cases where the same identifier symbol must be used
     * in upper and lower forms. These entries will matched if entered
     * in the exact case listed here. Include any Unicode characters
     * in `\u{...}` form.
     *
     * NOTE: This should only be used for short, simple identifiers
     * where there is a conflict with the lowercased version.
     *
     * Example: _x("T.", 'uppercase tabespoon symbols', 'cocktail-recipes')
     *    Allows `T` or `T.` for Tablespoons, while `t` or `t.` can
     *    remain in the regular identifiers() list for teaspoons.
     */
    public static function upperSymbols(): string
    {
        return '';
    }

    public function __toString(): string
    {
        return $this->symbol ?? $this->name();
    }

    protected function signatureData(): array
    {
        return [$this->symbol ?? $this->name];
    }
}
