<?php

namespace CocktailRecipes\Recipes\Values\Units;

/**
 * Parts
 *
 * Using 1/4 oz as a "part" for normalizing convertible measurements
 */
class Part extends Unit
{
    protected const MIN_AMOUNT = 1;
    protected const MAX_AMOUNT = null;
    protected const RANGE_WIDTH_RATIO = 1.5;

    /** Precision used for rounding when displaying unit quantities */
    protected const PRECISION = 1;

    /**
     * Alternate units to use for small parts
     *
     * Each entry: [<threshold>, [precision], unit, [unit2]]
     * If under threshold, corresponding precision and/or unit(s) used
     * If precision specified (int|float), used as an alternate precision
     * If 2 units listed, uses closest match with a bias towards first
     */
    protected const SMALL_PARTS_AS = [];

    // Bias towards first unit when 2 in SMALL_PARTS_AS entry
    protected const UNIT_BIAS = 1e-6;

    public function name($count = 1): string
    {
        return _nx('part', 'parts', $count, 'measurement', 'cocktail-recipes');
    }

    final public function normalize($value): array
    {
        return [
            is_array($value)
                ? [$this->toParts($value[0]), $this->toParts($value[1])]
                : $this->toParts($value),
            Unit::get('Part')
        ];
    }

    /** Convert unit's quantity to equivalent in parts */
    protected function toParts(float $value): float
    {
        return $value;
    }

    /** Convert from parts to unit's quantity */
    protected function fromParts(float $value): float
    {
        return $value;
    }

    /**
     * Value and units adjusted for display purposes
     *
     * @param  float|array{float,float}
     * @return array{float|array{float,float}, Unit}
     */
    final public function displayAs($value): array
    {
        if (is_array($value)) {
            [$value, $max] = $value;
        } else {
            $max = null;
        }
        $matched = false;
        foreach (static::SMALL_PARTS_AS as $data) {
            if ($value < $data[0] * static::MIN_THRESHOLD) {
                $matched = true;
                break;
            }
        }
        if ($matched) {
            $precision = !is_string($data[1]) ? $data[1] : null;
            $unitPos = $precision ? 2 : 1;
            $unit  = isset($data[$unitPos]) ? Unit::get($data[$unitPos]) : $this;
            $unit2 = isset($data[++$unitPos]) ? Unit::get($data[$unitPos]) : null;
        } else {
            $unit = $this;
            $unit2 = null;
        }

        /** @var Part $unit */
        /** @var Part $unit2 */
        $precision ??= $unit::PRECISION;
        if ($unit2 === null) {
            $value = round($unit->fromParts($value) / $precision) * $precision;
        } else {
            // compare both unit options; prefer first unless second fits better
            $precision2 = $unit2::PRECISION;
            $exact2 = $unit2->fromParts($value);
            $value2 = round($exact2 / $precision2) * $precision2;
            $exact = $unit->fromParts($value);
            $value = round($exact / $precision) * $precision;
            if (abs($exact2 - $value2) + static::UNIT_BIAS < abs($exact - $value)) {
                $unit = $unit2;
                $precision = $precision2;
                $value = $value2;
            }
        }

        if ($max !== null) {
            $value = [$value, round($unit->fromParts($max) / $precision) * $precision];
        }
        return [$value, $unit];
    }
}
