<?php

namespace CocktailRecipes\Recipes\Values\Units;

use CocktailRecipes\Recipes\Features\ShowDecimalUnits;

/**
 * Milliliters
 *
 * Traditionally 30 mL used as 1 oz equivalent in cocktail recipes
 */
class Milliliter extends Part
{
    use ShowDecimalUnits;

    protected const MIN_AMOUNT = 1;     // 1 mL (~1/4 tsp)
    protected const MAX_AMOUNT = 240;   // 240 mL (~8 oz)
    protected const RANGE_WIDTH_FLOOR = 1;

    // Conversion constants
    protected const ML_PER_PART = 7.5;
    protected const PRECISION = 7.5;    // round to nearest 7.5 mL
    protected const SMALL_PARTS_AS = [
        [5 / 7.5, 0.25],        // under 5 mL: use 0.25 mL precision
        [10 / 7.5, 2.5 ],       // under 10 mL: use 2.5 mL precision
    ];

    protected function init(array $options): void
    {
        $this->symbol = _x('mL', 'measurement (symbol)', 'cocktail-recipes');
    }

    public function name($count = 1): string
    {
        return _nx('milliliter', 'milliliters', $count, 'measurement', 'cocktail-recipes');
    }

    protected function toParts(float $value): float
    {
        return $value / self::ML_PER_PART;
    }

    protected function fromParts(float $value): float
    {
        return $value * self::ML_PER_PART;
    }

    public static function identifiers(): string
    {
        // U+2113 = Script Small L
        // U+3396 = Square "ml"
        return __("ml.,m.l,m.l.,m\u{2113}.,\u{3396},milliltr,milliliter,milliliters,millilitre,millilitres", 'cocktail-recipes');
    }

    public static function upperSymbols(): string
    {
        return _x("-", 'uppercase milliliter symbols; "-" if none', 'cocktail-recipes');
    }
}
