<?php

namespace CocktailRecipes\Recipes\Values;

use CocktailRecipes\Recipes\Values\Units\Unit as Units;
use CocktailRecipes\Recipes\Helpers\TokenMap;
use CocktailRecipes\Core\Helpers\Number;
use CocktailRecipes\Core\Helpers\Text;

/**
 * @property int|float|array   $num
 */
class Quantity extends Value
{
    /** Narrow No-Break Space (NNBSP) */
    protected const NNBSP = "\u{202F}";

    /** En Dash */
    protected const NDASH = "\u{2013}";

    /** Range dash characters */
    protected const RANGE_DASH = self::NNBSP . self::NDASH . self::NNBSP;

    /** Characters which can delimit quantities from the item in the text */
    public const DELIMITERS = '-.,:;';

    /** @var int|float|array    number (int/float) or range (array) */
    protected $num;

    /** Unit connector word regular expression pattern */
    private static ?string $connectorRegex = null;

    /** @param int|float|array $num */
    protected function __construct($num)
    {
        $this->num = $num;
    }

    public function numToString(): string
    {
        return !is_array($this->num)
            ? Number::asDecimal($this->num)
            : Number::asDecimal($this->num[0]) . '-' . Number::asDecimal($this->num[1]);
    }

    public function __toString(): string
    {
        return $this->numToString();
    }

    /** Quantities with fractional amounts as HTML entitites */
    public function toHtml(): string
    {
        return !is_array($this->num)
            ? $this->htmlNumber($this->num)
            : $this->htmlNumber($this->num[0]) . self::RANGE_DASH . $this->htmlNumber($this->num[1]);
    }

    /** Format a number in HTML with fractions or as a decimal */
    protected function htmlNumber(float $num): string
    {
        return Number::withFractions($num);
    }

    public function pluralCount(?string $htmlNum = null)
    {
        // if range, use max value and at least 2
        if (is_array($this->num)) return max($this->num[1], 2);
        // fractions less than 1 are singular, else plural
        if ($this->num > 0 && $this->num < 1) {
            return ($htmlNum && Number::isFraction($htmlNum)) ? 1 : 2;
        }
        // treat values betwen 1-2 as plural; otherwise use number as-is
        return ($this->num > 1) ? max($this->num, 2) : $this->num;
    }

    public function adjustedCopy(?float $multiplier = null): self
    {
        $copy = clone $this;
        if ($multiplier) {
            $copy->num = !is_array($copy->num)
                ? $copy->num * $multiplier
                : [$copy->num[0] * $multiplier, $copy->num[1] * $multiplier];
        }
        return $copy;
    }

    /**
     * Extract quantity or measurement from text for recipe parsing
     *
     * Values to be extracted are expected at the start of the text.
     * Quantity can be listed in any of these formats:
     *    integer              e.g. '2', '3'
     *    integer range        e.g. '2-3', '1 - 2'
     *    fraction             e.g. '3/4'
     *    integer+fraction     e.g. '1-1/4', '2 1/2'
     *    decimal              e.g. '.25', '0.5', '1.5'
     * A unit after quantity is optional and must be one of the known units.
     * Space(s) before the unit indicator are suggested but not required.
     * The qty and/or unit  must be delimited from the rest of the text with
     *    a space or characters such as dash, comma, colon or semicolon.
     *
     * @return  Quantity|Measurement|null|false     null if none, false on error
     */
    public static function extract(string &$text)
    {
        $text = trim($text);
        if (!($bytes = strlen($text))) return null;

        // Detect qty...
        $qty = null;
        if (preg_match('/^(\d+)(?:\s|-|\s-\s)(\d+)\/(\d+)/', $text, $matches)) {
            // ...integer + fraction (float)
            $qty = intval($matches[1]) + intval($matches[2]) / intval($matches[3]);
        } elseif (preg_match('/^(\d+)\/(\d+)/', $text, $matches)) {
            // ...fraction (float)
            $qty = intval($matches[1]) / intval($matches[2]);
        } elseif (preg_match(
            '/^(\d+|\d*[.,]\d+)\s{0,2}--?\s{0,2}(\d*[.,]\d+|\d+)/',
            $text,
            $matches
        )) {
            // ...range (string)
            [$min, $max] = array_map(function ($num) {
                if (ctype_digit($num)) return (int) $num;
                $num = (float) str_replace(',', '.', $num);
                return (fmod($num, 1.0) === 0.0) ? (int) $num : $num;
            }, array_slice($matches, 1, 2));
            if ($min <= 0 || $max <= $min) return false;
            $qty = [$min, $max];
        } elseif (preg_match('/^(\d*[.,]\d+)/', $text, $matches)) {
            // ...decimal (float)
            // Note: supports `1.5` and `1,5` decimal formats
            $qty = (float) str_replace(',', '.', $matches[1]);
        } elseif (preg_match('/^(\d+)/', $text, $matches)) {
            // ...integer (int)
            $qty = (int) $matches[1];
        }
        if ($qty === null) return null;
        if (!is_array($qty) && $qty <= 0) return false;
        if (($pos = strlen($matches[0])) >= $bytes) return false;
        if (is_float($qty) && fmod($qty, 1.0) === 0.0) $qty = (int) $qty;

        // Detect unit indictactor after quantity...
        $unit = null;
        $map = TokenMap::get();
        if (
            ($regex = $map['unit_regex'] ?? null)
            && preg_match($regex, $text, $matches, 0, $pos)
            && ($match = Text::toToken(str_replace(['.', ' '], '', $matches[1]), ['lower' => false]))
            && ($name = $map['units'][$match] ?? $map['units'][mb_strtolower($match, 'UTF-8')] ?? null)
        ) {
            // when a unit is found, it is invalid if no other text on line
            if (($pos += strlen($matches[0])) >= $bytes) return false;
            // get unit object and normalize
            $unit = Units::get($name);
            if (!$unit->isValid($qty)) return false;
            [$qty, $unit] = $unit->normalize($qty);
        }

        // qty/units must be followed by space/delimiter and additional text
        if ($text[$pos] == ' ') {
            $text = ltrim(substr($text, $pos));
            if ($unit && (self::$connectorRegex ??= self::connectorRegex())) {
                if (preg_match(self::$connectorRegex, $text, $matches)) {
                    $text = ltrim(substr($text, strlen($matches[0])));
                }
            }
        } else {
            if (strpos(self::DELIMITERS, $text[$pos]) === false) return false;
            $text = ltrim(substr($text, ++$pos));
        }
        if ($text == '') return false;

        return $unit ? new Measurement($qty, $unit) : new self($qty);
    }

    private static function connectorRegex(): string
    {
        $words = array_filter(array_map(
            fn($word) => preg_quote(mb_strtolower(trim($word), 'UTF-8'), '/'),
            explode(',', _x(
                'of',
                'connector(s) after measurement unit (e.g. "2 oz of gin")',
                'cocktail-recipes'
            ))
        ));
        return $words
            ? '/^(' . implode('|', $words) . ')(?:\s|$)/ui'
            : '';
    }
}
