<?php

namespace CocktailRecipes\Recipes;

use CocktailRecipes\Core\Contracts\Renderable;
use CocktailRecipes\Core\Base\ReadOnlyProps;
use CocktailRecipes\Core\Helpers\HTML;
use LogicException;

/**
 * Base class for recipe related entities
 *
 * All child classes must set $this->name in init() or it must define
 * its own name($count) override.
 *
 * @property string $name   read-only localized name
 */
abstract class RecipeEntity extends ReadOnlyProps implements Renderable
{
    // Namespace for interfaces
    private const INTERFACE_NS = 'CocktailRecipes\Recipes\Contracts\\';

    // Delimiter for signature data entries
    protected const SIGNATURE_DELIMITER = ':';

    /**
     * Localized name of entity (non-dynamic)
     *
     * For entites which will not be used with a count, this value should
     * be set in the init() function and localized using __() or _x().
     */
    protected ?string $name = null;

    final public function __construct(array $options = [])
    {
        $this->init($options);
        foreach (get_class_methods($this) as $method) {
            if (substr($method, 0, 5) == 'init_') {
                $this->$method($options);
            }
        }
        if ($this->name === null && ($this->name = $this->name()) == '') {
            throw new LogicException('Must set $name or define name()');
        }
    }

    /** Set properties for entity using __() or _x() as needed */
    protected function init(array $options): void {}

    /**
     * Dynamic localized, count-specific name of entity
     *
     * For entities which will be used with a count, this function must
     * be extended and return the localized name using _n() or _nx().
     */
    public function name($count = 1): string
    {
        return $this->name ?? '';
    }

    /** HTML safe version of name */
    public function htmlName($count = 1): string
    {
        return HTML::esc($this->name($count));
    }

    public function __toString(): string
    {
        return $this->name();
    }

    public function toHtml(): string
    {
        return $this->htmlName();
    }

    final public function signature(): string
    {
        $data = $this->signatureData();
        $this->addSignatureSuffix($data);
        return implode(self::SIGNATURE_DELIMITER, $data);
    }

    protected function signatureData(): array
    {
        return array_slice(explode('\\', static::class), -2);
    }

    protected function addSignatureSuffix(&$data): self
    {
        return $this;
    }

    /** True if class supports a specific feature */
    public static function allows(string $feature): bool
    {
        $interface = self::INTERFACE_NS . 'Allows' . $feature;
        return interface_exists($interface) && is_subclass_of(static::class, $interface);
    }
}
