<?php

namespace CocktailRecipes\Recipes;

use CocktailRecipes\Recipes\Elements\Element;
use CocktailRecipes\Recipes\Elements\Garnishes\Garnish;
use CocktailRecipes\Recipes\Helpers\TokenMap;
use CocktailRecipes\Recipes\Helpers\Modifiers;
use CocktailRecipes\Recipes\Values\Measurement;
use CocktailRecipes\Recipes\Values\Quantity;
use CocktailRecipes\Recipes\Values\Units\Part;
use CocktailRecipes\Recipes\Values\Units\Unit;
use CocktailRecipes\Exceptions\UnknownComponentType;
use CocktailRecipes\Core\Base\ReadOnlyProps;
use CocktailRecipes\Core\Renderers\Text as RenderedText;
use CocktailRecipes\Core\Renderers\Markdown as RenderedMarkdown;
use CocktailRecipes\Core\Renderers\MarkdownFormat as RenderedMarkdownFormat;
use CocktailRecipes\Core\Renderers\Join as RenderedJoin;
use CocktailRecipes\Core\Renderers\JoinAnd as RenderedAnd;
use CocktailRecipes\Core\Renderers\JoinOr as RenderedOr;
use CocktailRecipes\Core\Renderers\JoinAndOr as RenderedAndOr;
use CocktailRecipes\Core\Helpers\Grammar;
use CocktailRecipes\Core\Helpers\Sanitizer;
use CocktailRecipes\Core\Helpers\Text;

/**
 * @property-read ?object   $method         preparation method
 * @property-read array     $ingredients    recipe ingredients
 * @property-read ?object   $glass          glass type
 * @property-read array     $altGlasses     alternate glass type(s)
 * @property-read ?object   $ice            ice type
 * @property-read array     $altIce         alternate ice type(s)
 * @property-read array     $garnishes      garnish(es)
 * @property-read array     $optGarnishes   optional garnish(es)
 * @property-read array     $altGarnishes   alternate garnish(es)
 * @property-read array     $notes          cocktail notes
 * @property-read array     $errors         parsing errors
 */
class Recipe extends ReadOnlyProps
{
    // Floating-point comparison tolerance for treating values as equal
    protected const FLOAT_EPSILON = 1e-9;

    // Namespace for receipe notes
    private const NOTES_NS = 'CocktailRecipes\Recipes\Notes\\';

    // Namespace for recipe elements
    private const ELEMENTS_NS = 'CocktailRecipes\Recipes\Elements\\';

    // Recipe properties
    protected ?object $method       = null;
    protected array   $ingredients  = [];
    protected ?object $glass        = null;
    protected array   $altGlasses   = [];
    protected ?object $ice          = null;
    protected array   $altIce       = [];
    protected array   $garnishes    = [];
    protected array   $optGarnishes = [];
    protected array   $altGarnishes = [];
    protected array   $notes        = [];
    protected array   $errors       = [];
    protected ?int    $parsedAt     = null;

    /** @internal Only used during recipe parsing in constructor */
    private ?array $dedupe = [];

    // constructor
    public function __construct(string $content)
    {
        $this->parseContent($content);
    }

    /** Get ingredients adjusted as needed for display */
    public function ingredientsForDisplay(array $options = []): array
    {
        $multiplier = null;
        $servings = $options['servings'] ?? null;
        if (is_int($servings) && $servings > 0) $multiplier = $servings;
        // @todo $options['yield'] - if set & differs from recipe size, adjust multipler
        if ($multiplier !== null && abs($multiplier - 1.0) < self::FLOAT_EPSILON) $multiplier = null;

        $toUnit = null;
        if (($units = $options['units'] ?? null) && ($unit = Unit::get($units))) {
            if ($unit instanceof Part && get_class($unit) !== Part::class) {
                $toUnit = $unit;
            }
        }

        $ingredients = [];
        foreach ($this->ingredients as $ingredient) {
            $adjust =
                (
                    $multiplier !== null
                    && $ingredient->quantity
                ) || (
                    $toUnit
                    && $ingredient->quantity instanceof Measurement
                    && $ingredient->quantity->unit instanceof Part
                );
            $ingredients[] = $adjust
                ? $ingredient->adjustedCopy($multiplier, $toUnit)
                : $ingredient;
        }
        return $ingredients;
    }

    /** Get summary labels and associated values */
    public function summary(): array
    {
        $summary = [];
        if ($val = $this->method->summary ?? null) {
            $summary[__('Method', 'cocktail-recipes')] = new RenderedText($val);
        }
        if ($val = $this->glass->summary ?? null) {
            $summary[__('Glass', 'cocktail-recipes')] = new RenderedText($val);
        }
        if ($val = $this->ice->summary ?? null) {
            $summary[__('Ice', 'cocktail-recipes')] = new RenderedText($val);
        }
        // include required, alternate and/or optional garnishes
        // NOTES:
        //  1. Required garnishes shown as a comma delimited list. Will also list up to 2
        //     optional garnishes if only 1 req garnish, or 1 optional if 2 req garnishes.
        //  2. If no req garnishes, show alternate garnish if they exist with "or". If more
        //     than 2
        //  3. If no req or alt garnishes, shows first optional one. Will show 2. optional
        //     ones only if there are exactly 2 optional garnishes.
        $garnishes = [];
        $join = RenderedJoin::class;
        if ($this->garnishes) {
            // show all required garnishes in summary
            $garnishes = $this->garnishes;
            if ($this->optGarnishes && ($numReq = count($garnishes)) < 3) {
                // if 1-2 required garnishes, can show the first optional garish
                $garnishes[] = $this->optGarnishes[0];
                if ($numReq == 1 && count($this->optGarnishes) == 2) {
                    // when 1 required garnish and 2 optional, show both optional
                    $garnishes[] = $this->optGarnishes[1];
                }
            }
        } elseif ($this->altGarnishes) {
            // when no required garnishes, show alternate garnishes in summary with "or"
            $join = RenderedOr::class;
            $garnishes[] = $this->altGarnishes[0];
            if (($numAlt = count($this->altGarnishes)) == 2) {
                $garnishes[] = $this->altGarnishes[1];
            } else if ($numAlt > 2) {
                $garnishes[] = new RenderedText(_x('others', 'other garnishes available', 'cocktail-recipes'));
            }
        } elseif ($this->optGarnishes) {
            // when no required or alternates, show 1 or 2 optional garnishes
            $garnishes[] = $this->optGarnishes[0];
            if (count($this->optGarnishes) == 2) {
                $garnishes[] = $this->optGarnishes[1];
            }
        }
        if ($garnishes) {
            $formatter = function ($name, $obj) {
                if ($obj instanceof Garnish) {
                    $name = mb_convert_case($name, MB_CASE_TITLE, 'UTF-8');
                    if ($obj->optional) $name .= ' (' . esc_html_x('opt', 'optional garnish suffix', 'cocktail-recipes') . ')';
                }
                return $name;
            };
            $summary[__('Garnish', 'cocktail-recipes')] = (new $join(...$garnishes))->with($formatter);
        }
        return $summary;
    }

    /** Get recipe instruction steps */
    public function steps(): array
    {
        $steps = [];
        $vars = [
            '{GLASS}' => $this->glass
                ? ($this->glass->a_name ?? Grammar::a($this->glass->name))
                : __('a glass', 'cocktail-recipes'),
            '{ICE}' => $this->ice
                ? ($this->ice->singular ? Grammar::a($this->ice->name) : $this->ice->name)
                : '',
            '{OVER_ICE}' => $this->ice->over_ice ?? '',
        ];
        $hasAltIce = (bool) $this->altIce;
        foreach ($this->method->steps ?? [] as $step) {
            $steps[] = new RenderedMarkdown(strtr($step, $vars));
            if ($hasAltIce && (strpos($step, '{OVER_ICE}') !== false || strpos($step, '{ICE}') !== false)) {
                $steps[] = [new RenderedMarkdownFormat(
                    /* translators: %s is the alternate ice type(s) */
                    _n(
                        'As an alternate, you can use %s',
                        'Alternate ice types include %s',
                        count($this->altIce),
                        'cocktail-recipes'
                    ),
                    (new RenderedOr(...$this->altIce))->with(
                        fn($name, $obj) => $obj->singular ? Grammar::a($name) : $name
                    )
                )];
                $hasAltIce = false;
            }
        }
        // add step(s) with required/optional garnishes (with "and") or all alternate garnishes (with "or")
        $formatter = fn($name, $obj) => $obj->quantity ? $name : Grammar::a($name);
        if ($this->garnishes || $this->altGarnishes) {
            $steps[] = new RenderedMarkdownFormat(
                /* translators: %s is the alternate garnish */
                __('Garnish with %s', 'cocktail-recipes'),
                $this->garnishes
                    ? (new RenderedAnd(...$this->garnishes))->with($formatter)
                    : (new RenderedOr(...$this->altGarnishes))->with($formatter)
            );
        }
        if ($this->altGarnishes && $this->garnishes) {
            $steps[] = [new RenderedMarkdownFormat(
                /* translators: %s is the alternate garnish */
                _n(
                    'As an alternative, you can garnish with %s',
                    'Alternate garnishes you can use are %s',
                    count($this->altGarnishes),
                    'cocktail-recipes'
                ),
               (new RenderedOr(...$this->altGarnishes))->with($formatter)
            )];
        }
        if ($this->optGarnishes) {
            $steps[] = new RenderedMarkdownFormat(
                /* translators: %s is the optional garnish */
                __('Optionally garnish with %s', 'cocktail-recipes'),
                (new RenderedAndOr(...$this->optGarnishes))->with($formatter)
            );
        }

        return $steps;
    }

    /** Get recipe notes */
    public function notes(): array
    {
        $notes = [];
        foreach ($this->notes['general'] ?? [] as $note) {
            $notes[] = new RenderedMarkdown($note);
        }
        if ($this->altGlasses) {
            $notes[] = new RenderedMarkdownFormat(
                /* translators: %s is the alternate glassware */
                __('Alternate glassware: %s', 'cocktail-recipes'),
                new RenderedOr(...$this->altGlasses)
            );
        }
        return $notes;
    }

    /** Get recipe sourdes */
    public function sources(): array
    {
        $sources = [];
        foreach ($this->notes['source'] ?? [] as $source) {
            $sources[] = new RenderedMarkdown($source);
        }
        return $sources;
    }

    // parse content and set recipe properties
    private function parseContent(string $content)
    {
        $map = TokenMap::get();
        if (!is_array($map)) {
            $this->error('Token map unavailable');
            return;
        }

        foreach (explode("\n", Sanitizer::cleanContent($content)) as $line) {
            if (($line = trim(Sanitizer::stripComments($line))) == '') continue;

            // split line into type/value or value if no keyword used
            // NOTES:
            //    The regex pattern below is intentionally very broad so it can match
            //    Unicode characters if used in the localization of any keywords. While
            //    it may capture more than just simple word(s), only if the matched
            //    value is a known type will the line be parsed as `<keyword>: <value>`
            if (
                preg_match('/^\s*([^=:]+)\s*[=:]\s*(.*?)\s*$/', $line, $matches)
                && ($keyword = Text::toToken($matches[1]))
                && ($type = $map['keywords'][$keyword] ?? null)
            ) {
                // instructions in `<keyword>: <value>` format
                // example: "garnish: lemon peel"
                $value = $matches[2];
                // handle notes in "#Note", "#Source", etc. format
                if ($type[0] == '#') {
                    $this->addNote($line, substr($type, 1), $value);
                    continue;
                }
            } else {
                // simple instructions without values
                // examples: "shaken" or "large ice cube"
                $type = null;
                $value = $line;
            }

            // extract any quantity from value
            // NOTES:
            //    If a qty or a qty and units are detected at the start of the value, then
            //    it will be removed from the value and Quantity::extract() will return a
            //    Quantity or Measurement object for this entry. We will determine if the
            //    current entry allows units later, after we determine what entry type it is.
            //    If no qty is detected, null is returned, and on error false is returned.
            $source = $value;
            $quantity = Quantity::extract($value);
            if ($quantity === false) {
                $this->error("invalid quantity: $source");
                continue;
            }

            // extract any modifier terms
            // NOTES:
            //    Modifier terms can be included for each entry as a comma or space delimited
            //    list of terms in square brackets at the end of the line.
            $modifiers = new Modifiers();
            if (preg_match('/(?<!\x1B)\[(.*?)\]$/u', $value, $matches)) {
                $value = rtrim(substr($value, 0, -strlen($matches[0])));
                foreach (preg_split('/[\s,]+/u', $matches[1]) as $term) {
                    if ($token = Text::toToken($term)) {
                        if ($modifier = $map['modifiers'][$token] ?? null) {
                            $modifiers->add($modifier);
                        } else {
                            $this->error("unknown modifier: [$term]");
                        }
                    }
                }
            }

            // attempt to match value to known entries...
            // NOTES:
            //    $type will already be set if the line is in `<keyword>: <value>` format
            //    and keyword matches a known type. Otherwise $type will be set in the elseif
            //    block (if the line matches a known token) or in the else block if not set.
            //    Thus, $type is guaranteed to be set after this if-elseif-else block.
            $token = Text::toToken($value);
            if ($type && ($name = $map[$type][$token] ?? null)) {
                $valid = $this->set($source, $type, $name, $quantity, $modifiers);
            } elseif ($entry = $map['tokens'][$token] ?? null) {
                [$entryType, $name] = explode('.', $entry);
                $valid = (!$type || $entryType == $type)
                    && $this->set($source, $type = $entryType, $name, $quantity, $modifiers);
            } else {
                $valid = $this->set($source, $type ??= 'ingredient', null, $quantity, $modifiers, $value);
            }
            if (!$valid) {
                $this->error("invalid $type: '$source'");
            }
        }

        // property cleanup and adjustments after parsing completes
        if (!$this->glass && $this->altGlasses) {
            // if no primary glass set, use first alternate glass as the primary
            $this->glass = array_shift($this->altGlasses);
            $this->error('no primary glassware; assuming ' . $this->glass . ' as primary');
        }
        if (!$this->altGlasses) {
            // use default alternate glasses for primary glass if not otherwise set
            foreach ($this->glass->alt ?? [] as $alt) {
                $class = self::ELEMENTS_NS . 'Glassware\\' . $alt;
                $this->altGlasses[] = new $class();
            }
        }
        if (!$this->garnishes && count($this->altGarnishes) == 1) {
            // if no required garnishes and 1 alternate, move alternate to required or optional
            $garnish = array_shift($this->altGarnishes);
            $garnish->optional
                ? array_unshift($this->optGarnishes, $garnish)
                : ($this->garnishes[] = $garnish);
        } else {
            // remove optional state from alternate garnishes
            foreach ($this->altGarnishes as $garnish) {
                $garnish->setOptional(false);
            }
        }
        if ($this->ice->alt ?? null) {
            // allow for alternate ice types based on primary ice type select
            foreach ($this->ice->alt as $alt) {
                $class = self::ELEMENTS_NS . 'Ice\\' . $alt;
                $this->altIce[] = new $class();
            }
        }

        $this->parsedAt = time();
        unset($this->dedupe);
    }

    private function addNote(
        string $source,         // original text from recipe
        string $className,      // Note, Source, etc.
        string $text            // text for note
    ) {
        $class = self::NOTES_NS . $className;
        $note  = new $class($text);
        if ($note->text == '') {
            $this->error("empty $note->type: '$source'");
        } elseif ($prevSource = $this->dedupe[$id = $note->signature()] ?? null) {
            $this->error("duplicate $note->type: '$source' same as '$prevSource'");
        } else {
            $this->dedupe[$id] = $source;
            $this->notes[$note->type][] = $note->text;
        }
    }

    private function set(
        string $source,         // original text value from recipe
        string $type,           // method, ice, ingredient, glass, garnish, etc.
        ?string $name,          // name of item type to set or null to use $text
        ?Quantity $quantity,    // optional qty or qty+units
        Modifiers $modifiers,   // modifier term helper
        ?string $text = null    // custom text for item such as ingredients, etc.
    ): bool {
        if (!($group = Element::group($type))) throw new UnknownComponentType(esc_html($type));

        // determine options and instantiate element
        $options = [];
        if ($name === null) {
            $class = self::ELEMENTS_NS . $group . '\\' . ucfirst($type);
            if ($text == '' || !$class::allows('CustomName')) return false;
            $options['name'] = $text;
        } else {
            $class = self::ELEMENTS_NS . $group . '\\' . $name;
        }
        if ($quantity) {
            if (!$class::allows('Quantity')) return false;
            if ($quantity instanceof Measurement && !$class::allows('Measurement')) return false;
            $options['qty'] = $quantity;
        }
        $options += $class::modifierOptions($modifiers);
        $item = new $class($options);

        // determine property to hold element
        switch ($type) {
            case 'ingredient':
                $property = 'ingredients';
                break;
            case 'garnish':
                $property = $modifiers->has('alt')
                    ? 'altGarnishes'
                    : ($item->optional ? 'optGarnishes' : 'garnishes');
                break;
            case 'glass':
                $property = $modifiers->has('alt') || $this->glass !== null
                    ? 'altGlasses'
                    : 'glass';
                break;
            default:
                $property = $type;
        }

        // make sure all modifiers were used
        foreach ($modifiers->unused() as $modifier) {
            $this->error("unexpected modifier $modifier: $source");
        }

        // add element to recipe property
        if ($prevSource = $this->dedupe[$id = $item->signature()] ?? null) {
            $this->error(
                "duplicate $type: '$source'"
                . ($source !== $prevSource ? " same as '$prevSource'" : '')
            );
        } else {
            $this->dedupe[$id] = $source;
            if (is_array($this->$property)) {
                $this->$property[] = $item;
            } elseif (!isset($this->$property)) {
                $this->$property = $item;
            } else {
                $this->error("$type already set: ignoring '$source'");
            }
        }
        return true;
    }

    private function error($message): void
    {
        // add error message, skipping duplicates
        $signature = 'error:' . $message;
        if (!isset($this->dedupe[$signature])) {
            $this->errors[] = $message;
            $this->dedupe[$signature] = '';
        }
    }
}
