<?php

namespace CocktailRecipes\Recipes\Notes;

use CocktailRecipes\Core\Base\ReadOnlyProps;
use CocktailRecipes\Core\Helpers\Text;

/**
 * Recipe note
 *
 * @property string     $type   read-only note type
 * @property string     $text   read-only note text
 */
class Note extends ReadOnlyProps
{
    /** Note type (override in each child class) */
    protected string $type = 'general';

    /** Note text (set by constructor) */
    protected string $text;

    final public function __construct(string $text)
    {
        $this->text = $this->normalize($text);
    }

    /**
     * Normalize comment text
     *
     * Text will already have been passed through Sanitizer::cleanContent()
     * Will also unquote text and then collapse/trim whitespace
     */
    protected function normalize(string $text): string
    {
        return trim(preg_replace('/\s+/', ' ', Text::unquote($text)));
    }

    public function __toString(): string
    {
        return $this->text;
    }

    final public function signature(): string
    {
        return 'Notes:' . $this->type . ':' . mb_strtolower($this->text, 'UTF-8');
    }

    /**
     * Localized keywords used in recipe text
     *
     * Can be a comma delimited string. Spaces after commas are
     * okay and will be trimmed. Include any Unicode characters
     * in `\u{...}` form.
     *
     * Example: __('note,comment', 'cocktail-recipes')
     *    Enables use of `note: <term>` or `comment: <term>`
     *    inside [cocktail] shortcodes.
     */
    public static function keywords(): string
    {
        return __('note,notes,comment,comments', 'cocktail-recipes');
    }
}
