<?php

namespace CocktailRecipes\Recipes\Helpers;

use CocktailRecipes\Recipes\Elements\Element;
use CocktailRecipes\Recipes\Values\Quantity;
use CocktailRecipes\Plugin;
use CocktailRecipes\Core\Helpers\Cache;
use CocktailRecipes\Core\Helpers\Logger;
use CocktailRecipes\Core\Helpers\Text;
use CocktailRecipes\Core\Iterators\PhpFileIterator;
use CocktailRecipes\Core\Helpers\Sanitizer;

class TokenMap
{
    // Recipe namespace and directory
    private const BASE_NS  = 'CocktailRecipes\Recipes\\';
    private const BASE_DIR = __DIR__ . '/../';

    /** Prefix for token map cache entries */
    private const CACHE_PREFIX = 'token_map-';

    /** Current token map */
    private static ?array $map = null;

    /** Get token map cache name */
    private static function name(): string
    {
        return self::CACHE_PREFIX . Plugin::version();
    }

    /** Check if token map exist in cache */
    public static function cached(): bool
    {
        return Cache::has(self::name());
    }

    /**
     * Get token map
     *
     * @param   bool    $rebuild    true to rebuild if missing/invalid
     */
    public static function get(bool $rebuild = true): ?array
    {
        if (self::$map === null) {
            $map = Cache::get(self::name());
            if (is_array($map)) {
                self::$map = $map;
            } elseif ($rebuild) {
                self::buildMap();
                self::saveMap();
            }
        }
        return self::$map;
    }

    /**
     * Rebuild and cache the token map
     *
     * @param   bool    $cleanup    true to also cleanup old cache entries
     * @return  bool    true if rebuilt map written to cache
     */
    public static function rebuild(bool $cleanup = true): bool
    {
        self::buildMap();
        if (!self::saveMap()) return false;
        if ($cleanup) self::cleanup();
        return true;
    }

    /**
     * Save the current token map to cache
     *
     * @return  bool    true if save is successful
     */
    private static function saveMap(): bool
    {
        return Cache::writable() && Cache::set(self::name(), self::$map);
    }

    /** Remove all token map cache files except the current version */
    public static function cleanup(): void
    {
        $currentName = self::name();
        Cache::each(self::CACHE_PREFIX, function ($name) use ($currentName) {
            if ($name !== $currentName) {
                Cache::remove($name);
                Logger::debug("Removed old token map cache: $name");
            }
        });
    }

    /** Scan all components and build a map of all keywords and tokens */
    private static function buildMap(): void
    {
        $map = [
            'built'      => gmdate('Y-m-d\TH:i:s\Z'),
            'keywords'   => [],
            'unit_regex' => '',
            'modifiers'  => [],
            'units'      => [],
            'tokens'     => [],
        ];

        // process units
        $words = $abbrs = [];
        foreach (new PhpFileIterator(self::BASE_DIR . 'Values/Units') as $name) {
            $class = self::BASE_NS . 'Values\Units\\' . $name;
            // store unit identifiers
            foreach (['identifiers' => false, 'upperSymbols' => true] as $func => $upper) {
                foreach (array_filter(explode(',', $class::$func())) as $identifier) {
                    if ($upper && $identifier == '-') continue;   // skip unused upperSymbols
                    $identifier = Sanitizer::normalizeText($identifier, ['collapse_space' => true]);
                    if ($token = Text::toToken(
                        str_replace(['.', ' '], '', $identifier),
                        ['lower' => !$upper]
                    )) {
                        $map['units'][$token] = $name;
                        $word = Sanitizer::stripEsc(mb_strtolower(trim($identifier), 'UTF-8'));
                        if (preg_match('/^[^.]+\.$/u', $word)) {
                            $abbrs[mb_substr($word, 0, -1, 'UTF-8')] = true;
                        } else {
                            $words[$word] = true;
                        }
                    }
                }
            }
        }

        // build unit regex
        $patterns = [];
        foreach ($abbrs as $abbr => $_) {
            $patterns[] = preg_quote($abbr, '/') . '\\.?';
            if (isset($words[$abbr])) unset($words[$abbr]);
        }
        foreach ($words as $word => $_) {
            $patterns[] = preg_quote($word, '/');
        }
        usort($patterns, fn($a, $b) =>
            (mb_strlen($b, 'UTF-8') <=> mb_strlen($a, 'UTF-8'))
            ?: strcmp($a, $b)
        );
        $delimiters = preg_quote(Quantity::DELIMITERS, '/');
        $map['unit_regex'] = $patterns
            ? '/\s*(' . implode('|', $patterns) . ')(?=\b|\s|[' . $delimiters . ']|$)/Aiu'
            : '';

        // process recipe elements
        $normalize = ['normalize' => true];
        foreach (Element::types() as $type => $group) {
            // store keywords for this component type
            $groupNamespace = self::BASE_NS . 'Elements\\' . $group . '\\';
            $class = $groupNamespace . ucfirst($type);
            $dir = self::BASE_DIR . 'Elements/' . $group;
            foreach (array_filter(explode(',', $class::keywords())) as $keyword) {
                if ($token = Text::toToken($keyword, $normalize)) {
                    $map['keywords'][$token] = $type;
                }
            }
            foreach (new PhpFileIterator($dir) as $name) {
                $class = $groupNamespace . $name;
                // store terms which can only be used with the keywords
                foreach (array_filter(explode(',', $class::shortTerms())) as $term) {
                    if ($token = Text::toToken($term, $normalize)) {
                        if (!isset($map[$type])) $map[$type] = [];
                        $map[$type][$token] = $name;
                    }
                }
                // store terms which can be used with or without keywords
                foreach (array_filter(explode(',', $class::terms())) as $term) {
                    if ($token = Text::toToken($term, $normalize)) {
                        $map['tokens'][$token] = "$type.$name";
                    }
                }
            }
        }

        // process modifier terms
        foreach (Modifiers::terms() as $modifier => $terms) {
            foreach (array_filter(explode(',', $terms)) as $term) {
                if ($token = Text::toToken($term, $normalize)) {
                     $map['modifiers'][$token] = $modifier;
                }
            }
        }

        // process recipe note types
        foreach (new PhpFileIterator(self::BASE_DIR . 'Notes') as $name) {
            $class = self::BASE_NS . 'Notes\\' . $name;
            foreach (array_filter(explode(',', $class::keywords())) as $keyword) {
                if ($token = Text::toToken($keyword, $normalize)) {
                    $map['keywords'][$token] = '#' . $name;
                }
            }
        }

        self::$map = $map;
    }
}
