<?php

namespace CocktailRecipes\Recipes\Features;

use CocktailRecipes\Recipes\Values\Quantity;
use CocktailRecipes\Core\Helpers\Grammar;
use InvalidArgumentException;

/**
 * Adds support for quantities
 *
 * This adds a $quantity property (externally read-only) and a corresponding
 * constructor 'qty' option for assigning this value.
 *
 * Used by Garnish class.
 */
trait HasQuantity
{
    protected ?Quantity $quantity = null;

    /** Set with quantity if passed as 'qty' in constructor options */
    protected function init_quantity(array $options): void
    {
        if (($qty = $options['qty'] ?? null) === null) return;
        if (!($qty instanceof Quantity)) {
            throw new InvalidArgumentException('Quantity instance required to set qty');
        }
        $this->quantity = $qty;
    }

    public function __toString(): string
    {
        return $this->quantity
            ? Grammar::numberWithWord($this->quantity, $this->name($this->qtyCount()))
            : $this->name();
    }

    public function toHtml(): string
    {
        return $this->quantity
            ? Grammar::numberWithWord(
                $htmlNum = $this->quantity->toHtml(),
                $this->htmlName($this->qtyCount($htmlNum)))
            : $this->htmlName();
    }

    protected function qtyCount(?string $htmlNum = null): string
    {
        return $this->quantity->pluralCount($htmlNum);
    }

    protected function addSignatureSuffix(&$data): self
    {
        if ($this->quantity) $data[] = $this->quantity->numToString();
        return $this;
    }
}
