<?php

namespace CocktailRecipes\Recipes\Features;

use CocktailRecipes\Core\Helpers\Grammar;
use CocktailRecipes\Core\Helpers\HTML;
use InvalidArgumentException;

/**
 * Adds support for custom name parsed from the recipe text
 *
 * Standard name behavior is maintained for child classes which set $name in
 * init(). This adds support for custom, user-supplied names (e.g. "candied
 * hibiscus flower") passed as a constructor 'name' option while preserving
 * standard entity behavior.
 *
 * Used by base Garnish and Ingredient classes.
 */
trait HasCustomName
{
    /** Custom name passed to constructor */
    protected ?string $_name = null;

    /** Set custom name if passed as 'name' in constructor options */
    protected function init_customName(array $options): void
    {
        if ($this->name !== null || ($this->name = $this->name()) != '') return;
        if (($name = $options['name'] ?? null) === null) return;
        if (!is_string($name) || $name == '') {
            throw new InvalidArgumentException('Non-empty string required for custom name');
        }
        $this->_name = $name;
        $this->name  = Grammar::pluralize($name, 1);
    }

    protected function customName($count = 1): string
    {
        return Grammar::pluralize($this->_name, $count);
    }

    public function name($count = 1): string
    {
        return $this->_name !== null
            ? $this->customName($count)
            : parent::name($count);
    }

    public function htmlName($count = 1): string
    {
        return $this->_name !== null
            ? HTML::esc($this->customName($count))
            : parent::htmlName($count);
    }

    protected function signatureData(): array
    {
        if ($this->_name === null) return parent::signatureData();
        return [
            substr(strrchr(static::class, '\\'), 1),
            $this->_name
        ];
    }
}
