<?php

namespace CocktailRecipes\Recipes\Elements;

use CocktailRecipes\Recipes\RecipeEntity;

/**
 * Base class for recipe elements (e.g. methods, ice, glassware, ingredients)
 *
 * Defines shared keyword and term definitions for recipe parsing,
 *
 * @property string $name   read-only localized name
 */
abstract class Element extends RecipeEntity
{
    /** @internal Map of element types to their directory/namespace group */
    private const TYPES = [
        'method'     => 'Methods',
        'ice'        => 'Ice',
        'glass'      => 'Glassware',
        'garnish'    => 'Garnishes',
        'ingredient' => 'Ingredients',
    ];

    /**
     * Recipe entry modifiers allowed and corresponding option/value
     *
     * Example:
     *    [ 'opt' => ['optional', true], ...]
     */
    protected const MODIFIERS = [];

    /** Get list of all component types and their dir/namespace group */
    final public static function types(): array
    {
        return self::TYPES;
    }

    /** Get directory/namespace group for a specific type */
    final public static function group(string $type): ?string
    {
        return self::TYPES[$type] ?? null;
    }

    /** Options to be set based on modifiers */
    final public static function modifierOptions($modifiers): array
    {
        $options = [];
        foreach (static::MODIFIERS as $modifier => [$option, $value]) {
            if ($modifiers->has($modifier)) {
                $options[$option] = $value;
            }
        }
        return $options;
    }

    /**
     * Localized keywords used in recipe text
     *
     * Can be a comma delimited string. Spaces after commas are
     * okay and will be trimmed. Include any Unicode characters
     * in `\u{...}` form.
     *
     * Example: __('method,technique', 'cocktail-recipes')
     *    Enables use of `method: <term>` or `technique: <term>`
     *    inside [cocktail] shortcodes.
     */
    public static function keywords(): string
    {
        return '';
    }

    /**
     * Localized short terms used with recipe keywords
     *
     * Can be a comma delimited string. Spaces after commas are
     * okay and will be trimmed. Include any Unicode characters
     * in `\u{...}` form.
     *
     * Example: __('cubed,rocks,yes', 'cocktail-recipes')
     *    Enables `ice: cubed`, `ice: rocks` or `ice: yes` in a recipe,
     *    but prevents `cubed`, `rocks` or `yes` on their own.
     */
    public static function shortTerms(): string
    {
        return '';
    }

    /**
     * Localized terms used in recipe text
     *
     * Can be a comma delimited string. Spaces after commas are
     * okay and will be trimmed. Include any Unicode characters
     * in `\u{...}` form.
     *
     * Example: __('large ice cube,large cube', 'cocktail-recipes')
     *    Enables `large ice cube` or `large cube` with or without
     *    the 'ice:' prefix.
     */
    public static function terms(): string
    {
        return '';
    }
}
