<?php

namespace CocktailRecipes;

require_once __DIR__ . '/Core/Base/Plugin.php';

use CocktailRecipes\Admin\BuildTokenMapAction;
use CocktailRecipes\Admin\ClearCacheAction;
use CocktailRecipes\Admin\ClearRecipesAction;
use CocktailRecipes\Admin\GeneralSettings;
use CocktailRecipes\Admin\SettingsPage;
use CocktailRecipes\Jobs\Daily;
use CocktailRecipes\Shortcodes\CocktailShortcode;
use CocktailRecipes\Recipes\Helpers\TokenMap;
use CocktailRecipes\Core\Admin\Page as AdminPage;
use CocktailRecipes\Core\Base\Plugin as BasePlugin;
use CocktailRecipes\Core\Helpers\Assets;
use CocktailRecipes\Core\Helpers\Cache;
use CocktailRecipes\Core\Helpers\Context;

/** Cocktail Recipes plugin */
class Plugin extends BasePlugin
{
    private const OPTION_ADMIN_CLEANUP = 'cocktail_recipes_admin_cleanup';

    protected const STYLESHEETS = [
        'cocktail' => 'cocktail-recipe',
        'admin'    => 'admin',
    ];

    protected const SCRIPTS = [
        'cocktail' => 'cocktail-controls',
        'admin'    => 'admin',
    ];

    protected const ADMIN_PAGES = [
        'cocktail-recipes' => SettingsPage::class,
    ];

    protected const ADMIN_MENUS = [
        '+options-general.php' => 'cocktail-recipes'
    ];

    protected const ADMIN_ACTIONS = [
        'clear-recipes'  => ClearRecipesAction::class,
        'clear-cache'    => ClearCacheAction::class,
        'build-tokenmap' => BuildTokenMapAction::class,
    ];

    public const SETTINGS_GROUPS = [
        'general' => GeneralSettings::class,
    ];

    public const SHORTCODES = [
        'cocktail' => ['new', CocktailShortcode::class],
    ];
    public const JOBS = [
        'daily' => ['daily', Daily::class],
    ];

    public static function activate(): void
    {
        if (Cache::enabled()) TokenMap::rebuild();
    }

    protected static function finishUpgrade(): void
    {
        if (Cache::enabled()) {
            Cache::clear(CocktailShortcode::RECIPE_CACHE_PREFIX);
            Cache::clear(CocktailShortcode::UNIT_BTN_CACHE_NAME . '-');
            TokenMap::rebuild();
        }
    }

    protected static function initializeAdmin(): void
    {
        // safety cleanup in case cron has not run, but only do this once per day
        if (Cache::enabled()) {
            if (time() - ((int) get_option(self::OPTION_ADMIN_CLEANUP, 0)) >= DAY_IN_SECONDS) {
                TokenMap::cleanup();
                update_option(self::OPTION_ADMIN_CLEANUP, time());
            }
        }
    }

    protected static function addFrontendAssets(): void
    {
        if (Context::isPostWithShortcode('cocktail') || self::isAllowedOnListPage()) {
            Assets::include('cocktail');
        }
    }

    public static function isAllowedOnListPage(): bool
    {
        static $allowed;
        return $allowed ??=
            (is_home() && GeneralSettings::field('show_on_home'))
            || (is_archive() && GeneralSettings::field('show_on_lists'))
            || (is_search() && GeneralSettings::field('show_on_search'));
    }

    protected static function addAdminAssets(): void
    {
        if (Context::isBlockEditor() && Context::isAdminPage('post', 'post-new')) {
            Assets::include();
        }
        if (Context::isAdminPage('settings_page_cocktail-recipes')) {
            Assets::include('admin');
        }
    }

    protected static function adminNotice(string $code): ?string
    {
        switch ($code) {
            case 'recipes-cleared': return __('Recipe cache entries cleared.', 'cocktail-recipes');
            case 'cache-cleared': return __('All cache entries cleared.', 'cocktail-recipes');
            case 'tokenmap-built': return __('Token map built successfully.', 'cocktail-recipes');
            case 'tokenmap-rebuilt':  return __('Token map rebuilt successfully.', 'cocktail-recipes');
        }
        return null;
    }

    protected static function pluginsActionLinks(): array
    {
        $settingsUrl = AdminPage::adminUrl('options-general.php', 'cocktail-recipes');
        return [
            'settings' => '<a href="' . esc_url($settingsUrl) . '">Settings</a>',
        ];
    }

    protected static function pluginsMetaData(): array
    {
        $referenceUrl = AdminPage::adminUrl('options-general.php', 'cocktail-recipes', 'reference');
        return [
            '<a href="' . esc_url($referenceUrl) . '">Recipe Guide</a>',
        ];
    }

}
