<?php

namespace CocktailRecipes\Core\i18n;

/**
 * Portuguese language locale handler
 */
class pt extends Handler
{
    protected array $fullDurations    = ['dia', 'hora', 'minuto', 'segundo'];
    protected array $shortDurations   = ['dia', 'hr', 'min', 'seg'];

    /**
     * Pluralize regular Portuguese nouns
     *
     * Covers roughly 90–95% of regular nouns:
     * - Adds 's' to most words ending in vowels (e.g. casa -> casas)
     * - Adds 'es' to words ending in r, s or z (e.g. flor -> flores)
     * - Changes final 'm' -> 'ns' (e.g. bom -> bons)
     * - Changes final 'ão' -> 'ões' (e.g. nação -> nações)
     * - Words ending in 'l' -> 'is' (e.g. animal -> animais)
     * Handles nearly all standard cases; irregulars (e.g. pão/pães, mal/males)
     * not included.
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;

        if (preg_match('/\x{00E3}o$/u', $text)) return mb_substr($text, 0, -2, 'UTF-8') . "\x{00F5}es";
        if (preg_match('/m$/iu', $text)) return mb_substr($text, 0, -1, 'UTF-8') . 'ns';
        if (preg_match('/l$/iu', $text)) return mb_substr($text, 0, -1, 'UTF-8') . 'is';
        if (preg_match('/[rsz]$/iu', $text)) return $text . 'es';
        return $text . 's';
    }

    /**
     * Add an indefinite article to a noun
     *
     * Uses 'um' as a neutral masculine default. Does not account for gender
     * or number. Avoids adding a second article if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(um|uma|uns|umas)(\s|$)/iu', $text)) return $text;
        return "um $text";
    }
}
