<?php

namespace CocktailRecipes\Core\i18n;

/**
 * Italian language locale handler
 */
class it extends Handler
{
    protected array $fullDurations    = ['giorno', 'ora', 'minuto', 'secondo'];
    protected array $shortDurations   = ['giorno', 'hr', 'min', 'sec'];

    /**
     * Pluralize regular Italian nouns
     *
     * Covers roughly 90–95% of nouns:
     * - Words ending in 'a' changed to 'e' (e.g. casa -> case)
     * - Words ending in 'o' or 'e' changed to 'i'  (e.g. libro -> libri, cane -> cani)
     * - Words ending in consonants (often foreign) remain unchanged
     * - Default rule adds 'i'
     * Handles standard cases; irregular plurals (e.g. uomo/uomini, dio/dei) not included.
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;

        if (preg_match('/a$/iu', $text)) return mb_substr($text, 0, -1, 'UTF-8') . 'e';
        if (preg_match('/[oe]$/iu', $text)) return mb_substr($text, 0, -1, 'UTF-8') . 'i';
        if (preg_match('/[bcdfghjklmnpqrstvwxyz]$/iu', $text)) return $text;
        return $text . 'i';
    }

    /**
     * Add indefinite article to a noun
     *
     * Uses "un" as a neutral default. Covers basic cases for readability; does not
     * account for grammatical gender or elision rules. Avoids adding a second article
     * if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(un|uno|una|un\')(\s|$)/iu', $text)) return $text;
        return "un $text";
    }
}
