<?php

namespace CocktailRecipes\Core\i18n;

use CocktailRecipes\Core\Helpers\Text;

/**
 * French language locale handler
 */
class fr extends Handler
{
    protected array $fullDurations    = ['jour', 'heure', 'minute', 'seconde'];
    protected array $shortDurations   = ['j', 'h', 'min', 's'];
    protected bool  $pluralizeShortDurations = false;

    /**
     * Pluralize regular French nouns
     *
     * Covers roughly 90%+ of nouns:
     * - Keeps words ending in 's', 'x', 'z' unchanged
     * - Words ending in 'al' changed to 'aux; (e.g. cheval -> chevaux)
     * - Words ending in 'eau' or 'eu' changed to 'x'
     * - Words ending in 'ou' changed to 's' with some known exceptions to 'x'
     * - Default rule adds 's'
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;

        // handle common exceptions
        $lower = mb_strtolower($text, 'UTF-8');
        static $exceptions = [
            'oeil'     => 'yeux',           // eye -> eyes
            'ciel'     => 'cieux',          // sky -> skies
            'monsieur' => 'messieurs',      // mr -> gentlemen/sirs
            'madame'   => 'mesdames',       // mrs -> ladies/madams
        ];
        if (isset($exceptions[$lower])) return Text::matchCase($text, $exceptions[$lower]);

        if (preg_match('/[sxz]$/iu', $text)) return $text;
        if (preg_match('/al$/iu', $text)) return mb_substr($text, 0, -2, 'UTF-8') . 'aux';
        if (preg_match('/(eau|eu)$/iu', $text)) return $text . 'x';
        if (preg_match('/ou$/iu', $text)) {
            return preg_match('/(bijou|caillou|chou|genou|hibou|joujou|pou)$/iu', $text)
                ? $text . 'x'
                : $text . 's';
        }
        return $text . 's';
    }

    /**
     * Add indefinite article to a noun
     *
     * Uses "un" as a neutral masculine default. Does not account for gender or
     * words beginning with a vowel or silent 'h'. Avoids adding a second article
     * if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(un|une|des)(\s|$)/iu', $text)) return $text;
        return "un $text";
    }
}
