<?php

namespace CocktailRecipes\Core\i18n;

/**
 * Spanish language locale handler
 */
class es extends Handler
{
    protected array $fullDurations    = ['día', 'hora', 'minuto', 'segundo'];
    protected array $shortDurations   = ['día', 'hr', 'min', 'seg'];

    /**
     * Pluralize regular Spanish nouns
     *
     * Covers roughly 95% of nouns:
     * - Adds 's' to words ending in vowels (e.g. casa -> casas)
     * - Adds 'es' to words ending in consonants (e.g. color -> colores)
     * - Words ending in 'z' changed to 'ces' (e.g. pez -> peces)
     * - Words ending in 's' or 'x' usually unchanged
     * Handles nearly all standard cases; irregulars (e.g. el/los, compound nouns) not included.
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;

        if (preg_match(
            '/[aeiou\x{00E1}\x{00E9}\x{00ED}\x{00F3}\x{00FA}]$/iu',
            $text
        )) return $text . 's';
        if (preg_match('/z$/iu', $text)) return mb_substr($text, 0, -1, 'UTF-8') . 'ces';
        if (preg_match('/[sx]$/iu', $text)) return $text;
        return $text . 'es';
    }

    /**
     * Add indefinite article to a noun
     *
     * Uses "un" as a neutral masculine default. Covers basic cases for readability;
     * does not account for gender, number or words beginning with a stressed "a".
     * Avoids adding a second article if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(un|una|unos|unas)(\s|$)/iu', $text)) return $text;
        return "un $text";
    }
}
