<?php

namespace CocktailRecipes\Core\i18n;

/**
 * English language locale handler
 */
class en extends Handler
{
    protected array $fullDurations    = ['day', 'hour', 'minute', 'second'];
    protected array $shortDurations   = ['day', 'hr', 'min', 'sec'];

    /**
     * Pluralize regular English nouns
     *
     * Covers roughly 90–95% of nouns:
     * - Adds 'es' to words ending with -ch, -s, -sh, -x, -z
     * - Words ending in 'is' changed to 'es'
     * - Words ending in 'y' changed to 'ies', unless preceded by vowel
     * - Default rule adds 's'
     * Handles nearly all standard cases; irregulars such as leaf/leaves, knife/knives,
     * roof/roofs, hero/heroes, photo/photos, are not included.
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;

        if (preg_match('/(ch|s|sh|x|z)$/iu', $text)) return $text . 'es';
        if (preg_match('/is$/iu', $text)) return mb_substr($text, 0, -2, 'UTF-8') . 'es';
        if (preg_match('/[^aeiou]y$/iu', $text)) return mb_substr($text, 0, -1, 'UTF-8') . 'ies';
        return $text . 's';
    }

    /**
     * Add indefinite article to a noun
     *
     * Uses 'an' if word starts with a vowel sound, otherwise 'a'. Does not account for silent
     * 'h' (e.g. hour), 'u/eu' with consonant sounds (e.g. university, European), or acronyms
     * with vowel sounds (e.g. an MRI). Avoids adding a second article if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(a|an)(\s|$)/iu', $text)) return $text;
        return preg_match('/^[aeiou]/iu', $text) ? "an $text" : "a $text";
    }
}
