<?php

namespace CocktailRecipes\Core\i18n;

use CocktailRecipes\Core\Helpers\Text;

/**
 * German language locale handler
 */
class de extends Handler
{
    protected array $fullDurations    = ['Tag', 'Stunde', 'Minute', 'Sekunde'];
    protected array $shortDurations   = ['Tag', 'Std', 'Min', 'Sek'];
    protected bool  $pluralizeShortDurations = false;

    /**
     * Pluralize regular German nouns
     *
     * Covers roughly 85–90% of common nouns:
     * - Adds 'er' (with umlaut) to several neuter or irregular nouns (e.g. Kind -> Kinder, Haus -> Häuser)
     * - Adds 'nen' to feminine agent nouns ending in -in (e.g. Lehrerin -> Lehrerinnen)
     * - Adds 'n' to most nouns ending in -e (e.g. Blume -> Blumen)
     * - Adds 'en' to many feminine or abstract nouns ending in -ung, -heit, -keit or -schaft (e.g. Zeitung -> Zeitungen)
     * - Adds 'en' to many nouns ending in -au (e.g. Frau -> Frauen)
     * - Leaves nouns ending in -el, -en or -er unchanged
     * - Adds 's' to foreign or abbreviated nouns ending in a vowel (e.g. Auto -> Autos)
     * - Adds 'e' as the general fallback (e.g. Tag -> Tage)
     * Includes frequent umlaut exceptions; still omits rare irregulars.
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;

        // handle very common exceptions that take -er (no umlaut handling here)
        $lower = mb_strtolower($text, 'UTF-8');
        static $exceptions = [
            'kind'    => 'kinder',           // child -> children
            'mann'    => "m\u{00E4}nner",    // man -> men
            'buch'    => "b\u{00FC}cher",    // book -> books
            'haus'    => "h\u{00E4}user",    // house -> houses
            'wald'    => "w\u{00E4}lder",    // forest -> forests
            'stadt'   => "st\u{00E4}dte",    // city -> cities
            'tochter' => "t\u{00F6}chter",   // daughter -> daughters
            'mutter'  => "m\u{00FC}tter",    // mother -> mothers
            'vater'   => "v\u{00E4}ter",     // father -> fathers
        ];
        if (isset($exceptions[$lower])) return Text::matchCase($text, $exceptions[$lower]);

        // feminine agent nouns: Lehrerin -> Lehrerinnen
        if (preg_match('/in$/iu', $text)) return $text . 'nen';
        // most nouns ending in -e: Blume -> Blumen
        if (preg_match('/e$/iu', $text)) return $text . 'n';
        // common feminine/abstract: -ung, -heit, -keit, -schaft -> +en
        if (preg_match('/(ung|heit|keit|schaft)$/iu', $text)) return $text . 'en';
        // many nouns ending in -au: Frau -> Frauen, Bau -> Bauten
        if (preg_match('/au$/iu', $text)) return $text . 'en';
        // many ending with -el, -en, -er unchanged in plural (simple heuristic)
        if (preg_match('/(el|en|er)$/iu', $text)) return $text;
        // loanwords/abbrev ending with vowel -> +s: Auto -> Autos
        if (preg_match('/[aiouy]$/iu', $text)) return $text . 's';
        // default fallback
        return $text . 'e';
    }

    /**
     * Add indefinite article to a noun
     *
     * Uses 'ein' as a gender-neutral default. Does not account for grammatical
     * gender or case. Avoids adding a second article if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(ein|eine)(\s|$)/iu', $text)) return $text;
        return "ein $text";
    }
}
