<?php

namespace CocktailRecipes\Core\i18n;

/**
 * Danish language locale handler
 */
class da extends Handler
{
    protected array $fullDurations    = ['dag', 'time', 'minut', 'sekund'];
    protected array $shortDurations   = ['dag', 't', 'min', 's'];
    protected bool  $pluralizeShortDurations = false;

    /**
     * Pluralize regular Danish nouns
     *
     * Covers roughly 90–95% of regular nouns:
     * - Adds 'er' to most nouns (e.g. bil -> biler)
     * - Adds 'r' to nouns ending in 'e' (e.g. pige -> piger)
     * - Adds 'e' to some short or one-syllable nouns (e.g. kat -> katte)
     * - Adds 's' to foreign words or abbreviations (e.g. foto -> fotos)
     * Does not handle irregular or stem-changing forms.
     */
    public function pluralize(string $text, int $count = 1): string
    {
        if (abs($count) == 1) return $text;
        if (preg_match('/e$/iu', $text)) return $text . 'r';
        if (preg_match('/[aiouy]$/iu', $text)) return $text . 's';
        if (preg_match('/([aeiouy])([^aeiouy])$/iu', $text, $m)) {
            // Single consonant after short vowel -> double and add 'e'
            return $text . $m[2] . 'e';
        }
        if (preg_match('/[^aeiouy]{2,4}$/iu', $text)) return $text . 'e';
        return $text . 'er';
    }

    /**
     * Add an indefinite article to a noun
     *
     * Uses 'en' as a default article. Does not account for gender ('et' neuter)
     * or plural forms. Avoids adding a second article if one already exists.
     */
    public function indefinite(string $text): string
    {
        if (preg_match('/^(en|et)(\s|$)/iu', $text)) return $text;
        return "en $text";
    }
}
