<?php

namespace CocktailRecipes\Core\i18n;

use CocktailRecipes\Plugin;
use CocktailRecipes\Core\Base\ReadOnlyProps;
use CocktailRecipes\Core\Helpers\Locale;

/**
 * Base class for locale and language handlers
 *
 * @property string[] $fullDurations        read-only [day, hour, minute, second]
 * @property string[] $shortDurations       read-only [day, hr, min, sec]
 * @property string[] $compactDurations     read-only [d, h, m, s]
 */
abstract class Handler extends ReadOnlyProps
{
    // ---------------------------------------------------------------------
    // Primary extension points
    // ---------------------------------------------------------------------

    // defaults
    protected array $fullDurations    = ['day', 'hour', 'minute', 'second'];
    protected array $shortDurations   = ['day', 'hr', 'min', 'sec'];
    protected array $compactDurations = ['d', 'h', 'm', 's'];
    protected bool  $pluralizeShortDurations = true;

    /** Covert to plural form */
    public function pluralize(string $text, int $count = 1): string
    {
        return $text;
    }

    /** Add indefinite article to text */
    public function indefinite(string $text): string
    {
        return $text;
    }

    /** Join a number with a word */
    public function numberWithWord($number, string $word, string $space = ' '): string
    {
        return $number . $space . $word;
    }

    /** Join 2 words with and/or conjunction */
    public function joinTwo(string $item1, string $item2, string $with): string
    {
        return $item1 . ' ' . $this->joinConjunction($with) . ' ' . $item2;
    }

    /** Join 3+ words with and/or conjunction */
    public function joinMany(array $mostItems, string $lastItem, string $with): string
    {
        return $this->joinTwo(implode(', ', $mostItems), $lastItem, $with);
    }

    /** Localized "and", "or" or "and/or" conjunction */
    protected function joinConjunction(string $type): string
    {
        switch ($type) {
            case 'and': return Plugin::__('and');
            case 'or':  return Plugin::__('or');
        }
        return Plugin::__('and_or');
    }


    // ---------------------------------------------------------------------
    // Internal framework logic
    // ---------------------------------------------------------------------

    /** Get current locale-specific handler instance */
    final public static function get(): Handler
    {
        [$lang, $region] = self::localeInfo();
        if (!$region || !class_exists($class = __NAMESPACE__ . '\\' . $lang . '_' . $region)) {
            if (!class_exists($class = __NAMESPACE__ . '\\' . $lang)) {
                $class = __NAMESPACE__ . '\en';
            }
        }
        return new $class();
    }

    /**
     * Get current locale's language and optional region
     *
     * @return array [lang, null] or [lang, region]
     */
    private static function localeInfo(): array
    {
        $default = ['en', 'US'];
        if (!($code = Locale::code())) return $default;

        $parts = explode('_', str_replace('-', '_', trim($code)));
        if (count($parts) > 2) return $default;

        $lang = strtolower($parts[0]);
        if (!ctype_alpha($lang) || strlen($lang) != 2) return $default;
        if (!isset($parts[1])) return [$lang, null];

        $region = strtoupper($parts[1]);
        if (!ctype_alpha($region) || strlen($region) != 2) return $default;
        return [$lang, $region];
    }
}
