<?php

namespace CocktailRecipes\Core\Helpers;

use InvalidArgumentException;

final class Time
{
    /** Format elapsed time in seconds, milliseconds, or microseconds */
    public static function formatElapsed(float $seconds, bool $unicode = false): string
    {
        if ($seconds >= 1) return round($seconds, 3) . ' sec';
        if ($seconds >= 0.001) return round($seconds * 1000, 2) . ' ms';
        return round($seconds * 1_000_000, 1) . ($unicode ? "\u{00B5}s" : ' us');
    }

    /**
     * Format longer time durations in days, hours, minutes and/or seconds
     *
     * Syntax:
     *   formatDuration(seconds, [options])
     *   formatDuration(seconds, format, [options])
     *
     * Options:
     *   'format'    => string      'full' | 'short' (default) | 'compact'
     *   'limit'     => int         limit number of duration labels shown
     *   'delimiter' => string      default ' ' or ', ' if 'full' format
     *   'zero'      => string      alternate text if zero
     */
    public static function formatDuration(
        int $seconds,
        $format = 'short',
        array $options = []
    ): string {
        // @todo future - option to add 'and' connector with 'full' labels
        if (is_array($format)) {
            $options = $format;
            $format  = $options['format'] ?? 'short';
        } elseif (!is_string($format)) {
            throw new InvalidArgumentException('Argument 2 must be string or array');
        }
        $format = strtolower($format);
        if (!($labels = Locale::{$format . 'Durations'}())) {
            $labels = Locale::shortDurations();
            $format = 'short';
        }
        if ($seconds < 1) {
            if ($zero = $options['zero'] ?? null) return (string) $zero;
            return $format == 'compact'
                ? '0' . $labels[3]
                : Grammar::pluralize($labels[3], 0, true);
        }
        $limit = $options['limit'] ?? 0;
        $delimiter = $options['delimiter'] ?? ($format == 'full' ? ', ' : ' ');
        $pluralize = $format == 'full' || ($format == 'short' && Locale::pluralizeShortDurations());
        $parts = [];
        foreach ([86400, 3600, 60, 1] as $pos => $size) {
            if ($num = intdiv($seconds, $size)) {
                if ($format === 'compact') {
                    $parts[] = $num . $labels[$pos];
                } elseif ($pluralize && mb_strlen($labels[$pos]) > 1) {
                    $parts[] = Grammar::pluralize($labels[$pos], $num, true);
                } else {
                    $parts[] = Grammar::numberWithWord($num, $labels[$pos]);
                }
                $seconds -= $num * $size;
                if ($limit && !--$limit) break;
            }
        }
        return implode($delimiter, $parts);
    }
}
