<?php

namespace CocktailRecipes\Core\Helpers;

use TypeError;

final class Number
{
    /** HTML entities for fractional quantities */
    private const FRACTIONS = [
        "\u{215B}" => 0.125,  // 1/8
        "\u{00BC}" => 0.25,   // 1/4
        "\u{2153}" => 0.333,  // 1/3
        "\u{215C}" => 0.375,  // 3/8
        "\u{00BD}" => 0.5,    // 1/2
        "\u{215D}" => 0.625,  // 5/8
        "\u{2154}" => 0.666,  // 2/3
        "\u{00BE}" => 0.75,   // 3/4
        "\u{215E}" => 0.875,  // 7/8
        #"\u{2155}" => 0.2,   // 1/5
        #"\u{2156}" => 0.4,   // 2/5
        #"\u{2157}" => 0.6,   // 3/5
        #"\u{2158}" => 0.8,   // 4/5
    ];

    /**
     * Render a number using Unicode fraction characters as needed
     *
     * @param   int|float   $num
     */
    public static function withFractions($num): string
    {
        if (!is_numeric($num)) throw new TypeError("Numeric value expected");
        $sign = ($num < 0) ? '-' : '';
        $int = floor($num = abs($num));
        $frac = round($num - $int, 3);
        foreach (self::FRACTIONS as $char => $decimal) {
            if (abs($frac - $decimal) >= 0.015) continue;
            return $sign . ($int ? $int . $char : $char);
        }
        return $sign . self::asDecimal($num, 2);
    }

    /** True if number is one of the predefined fractions */
    public static function isFraction($num): bool
    {
        return isset(self::FRACTIONS[$num]);
    }

    /**
     * Localized number up to fixed number of places
     *
     * @param   int|float   $num
     */
    public static function asDecimal(
        $num,
        int $places = 2,
        bool $grouped = false,
        bool $trimmed = true
    ): string {
        $decimal = Locale::decimalPoint();
        $thousands = $grouped ? Locale::thousandsSeparator() : '';
        $output = number_format($num, $places, $decimal, $thousands);
        if ($trimmed) {
            if ($places > 0) {
                $output = rtrim($output, '0');
            }
            if (substr($output, -strlen($decimal)) == $decimal) {
                $output = substr($output, 0, -strlen($decimal));
            }
        }
        return $output;
    }
}
