<?php

namespace CocktailRecipes\Core\Helpers;

use CocktailRecipes\Plugin;

final class Logger
{
    /** RFC 5424 levels */
    const LEVELS = [
        'emergency' => 8,   // system is unusable
        'alert'     => 7,   // action must be taken immediately
        'critical'  => 6,   // critical conditions
        'error'     => 5,   // error conditions
        'warning'   => 4,   // warning conditions
        'notice'    => 3,   // normal but significant condition
        'info'      => 2,   // informational messages
        'debug'     => 1,   // debug-level messages
    ];

    // Log messages by severity
    public static function emergency(string $message): void { self::write('emergency', $message); }
    public static function alert(string $message): void { self::write('alert', $message); }
    public static function critical(string $message): void { self::write('critical', $message); }
    public static function error(string $message): void { self::write('error', $message); }
    public static function warning(string $message): void { self::write('warning', $message); }
    public static function notice(string $message): void { self::write('notice', $message); }
    public static function info(string $message): void { self::write('info', $message); }
    public static function debug(string $message): void { self::write('debug', $message); }

    private static function write(string $level, string $message): void
    {
        // @todo future - allow capturing to local log when we have admin pages to display it
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // if we need to handle binary or invalid UTF-8 (unlikely)...
            // $message = mb_convert_encoding($message, 'UTF-8', 'UTF-8');
            $message = str_replace(["\r\n", "\r", "\n"], '<LF>', $message);
            $message = str_replace("\t", '<TAB>', $message);
            $message = preg_replace('/[[:cntrl:]]/', '', $message);
            if (($message = trim($message)) != '') {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- only output if WP_DEBUG set
                error_log('[' . Plugin::name() . "][$level] " . $message);
            }
        }
    }
}
