<?php

namespace CocktailRecipes\Core\Helpers;

use CocktailRecipes\Core\i18n\Handler;

/**
 * Locale-specific functions
 *
 * Provides static access to locale-specific behaviors such as
 * number formatting, pluralization and article selection.
 *
 * System properties:
 *   decimalPoint(): string
 *   thousandsSeparator(): string
 *
 * Other locale-dependent functions and values exposed by Locale as
 * delegated functions are provided by the current locale handler.
 *
 * @see CocktailRecipes\Core\i18n\Handler
 */
final class Locale
{
    // properties from WordPress or PHP
    private const SYSTEM_PROPERTIES = [
        'decimalPoint'       => ['decimal_point', '.'],
        'thousandsSeparator' => ['thousands_sep', ','],
    ];

    // cached WordPress/PHP properties
    private static array $properties = [];

    // locale-specific handler
    private static ?Handler $handler = null;

    /** Current locale's code, e.g. en_US */
    public static function code(): string
    {
        static $code = null;
        return $code ??= function_exists('determine_locale')
            ? determine_locale()
            : (function_exists('get_locale') ? get_locale() : '');
    }

    // Delegate other functions to locale-specific handler
    public static function __callStatic(string $name, array $args)
    {
        if ($info = self::SYSTEM_PROPERTIES[$name] ?? null) {
            return self::$properties[$name] ??= self::systemProperty(...$info);
        }
        self::$handler ??= Handler::get();
        return method_exists(self::$handler, $name)
            ? self::$handler->$name(...$args)
            : self::$handler->$name;
    }

    // Get a property defined in self::SYSTEM_PROPERTIES
    private static function systemProperty(string $name, $default)
    {
        return ($GLOBALS['wp_locale']->number_format[$name] ?? null)
            ?? localeconv()[$name]
            ?? $default;
    }
}
