<?php

namespace CocktailRecipes\Core\Helpers;

final class Grammar
{
    /**
     * Adds indefinite article to text based on the current locale
     *
     * For English, prepends "a" or "an" as appropriate.
     * Other locales can override behavior via locale-specific handlers.
     */
    public static function a(string $text): string
    {
        if (($text = trim($text)) == '') return '';
        return Locale::indefinite($text);
    }

    /**
     * Get singular vs plural form based on count
     *
     * @param   string              $text       singular text or text with (singular|plural)
     * @param   int|float|string    $count      strings with numeric values supported
     * @param   bool                $withCount  true = count in output; e.g. "2 dogs"
     */
    public static function pluralize(
        string $text,
        float  $count,
        bool   $withCount = false
    ): string {
        $text = trim($text);
        if (
            strpos($text, '|') !== false
            && ($alt = self::resolveSingularPlural($text, abs($count) == 1)) !== null
        ) {
            $text = $alt;
        } else {
            $text = Locale::pluralize($text, $count);
        }
        return $withCount
            ? self::numberWithWord($count, $text)
            : $text;
    }

    /**
     * Convert all "(singular|plural)" entries in text to <singular> or <plural> text
     *
     * @return  string|null     null if no "(singular|plural)" entries found
     *
     * @internal for use by Grammar::pluralize()
     */
    private static function resolveSingularPlural(string $text, bool $singular): ?string
    {
        $found = false;
        $text  = preg_replace_callback(
            '/\(([^()|]+)\|([^()|]+)\)/iu', // matches '(<singular>|<plural>)' within text
            function (array $matches) use (&$found, $singular) {
                $found = true;
                return trim($matches[$singular ? 1 : 2]);
            },
            $text
        );
        return $found ? $text : null;
    }

    /** Join a number with a word */
    public static function numberWithWord($number, string $word, string $space = ' '): string
    {
        return Locale::numberWithWord($number, $word, $space);
    }

    /** Join a list of items with "and" */
    public static function joinAnd(array $items): string
    {
        return self::joinList($items, 'and');
    }

    /** Join a list of items with "or" */
    public static function joinOr(array $items): string
    {
        return self::joinList($items, 'or');
    }

    /** Join a list of items with "and/or" */
    public static function joinAndOr(array $items): string
    {
        return self::joinList($items, 'and/or');
    }

    /** @internal Join list with "and", "or" or "and/or" */
    private static function joinList(array $items, string $with): string
    {
        $items = array_filter(array_map('trim', $items));
        switch (count($items)) {
            case 0: return '';
            case 1: return $items[0];
            case 2: return Locale::joinTwo($items[0], $items[1], $with);
        }
        return Locale::joinMany(array_slice($items, 0, -1), end($items), $with);
    }
}
