<?php

namespace CocktailRecipes\Core\Helpers;

/**
 * File system helper
 *
 * Uses WP_Filesystem when available. Native PHP filesystem functions are used only
 * as a safe fallback when WP_Filesystem is unavailable or not initialized.
 *
 * phpcs notes:
 * - Alternative filesystem functions are intentionally used as fallbacks
 */
final class File
{
    private static function fs(): ?object
    {
        static $init = false;
        global $wp_filesystem;
        if (!$init) {
            if (empty($wp_filesystem)) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                WP_Filesystem();
            }
            $init = true;
        }
        return $wp_filesystem ?? null;
    }

    /** True if file exists */
    public static function exists(string $file): bool
    {
        return ($fs = self::fs()) ? $fs->exists($file) : file_exists($file);
    }

    /** True if file is a regular file */
    public static function isFile(string $file): bool
    {
        return ($fs = self::fs()) ? $fs->is_file($file) : is_file($file);
    }

    /** True if file is a directory */
    public static function isDir(string $file): bool
    {
        return ($fs = self::fs()) ? $fs->is_dir($file) : is_dir($file);
    }

    /** True if file is readable */
    public static function isReadable(string $file): bool
    {
        return ($fs = self::fs()) ? $fs->is_readable($file) : is_readable($file);
    }

    /** True if file is writable */
    public static function isWritable(string $file): bool
    {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_is_writable
        return ($fs = self::fs()) ? $fs->is_writable($file) : is_writable($file);
    }

    /**
     * Read file
     *
     * @return string|false
     */
    public static function read(string $file)
    {
        return ($fs = self::fs()) ? $fs->get_contents($file) : file_get_contents($file);
    }

    /** Write file */
    public static function write(string $file, $data): bool
    {
        return ($fs = self::fs())
            ? $fs->put_contents($file, $data, FS_CHMOD_FILE)
            : file_put_contents($file, $data, LOCK_EX) !== false;
    }

    /** Deletes file */
    public static function delete(string $file): bool
    {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
        return ($fs = self::fs()) ? $fs->delete($file) : @unlink($file);
    }

    /** Create directory */
    public static function makeDir(string $path, bool $recursive = false): bool
    {
        if ($fs = self::fs()) return $recursive ? self::wpMkdir($fs, $path) : $fs->mkdir($path);
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_mkdir
        return @mkdir($path, FS_CHMOD_DIR, $recursive);
    }
    private static function wpMkdir($fs, $path): bool
    {
        if ($fs->is_dir($path)) return true;
        $parent = dirname($path);
        if ($parent !== $path && !self::wpMkdir($fs, $parent)) return false;
        return $fs->mkdir($path);
    }

    /** Remove directory */
    public static function removeDir(string $path): bool
    {
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir
        return ($fs = self::fs()) ? $fs->rmdir($path) : rmdir($path);
    }
}
