<?php

namespace CocktailRecipes\Core\Helpers;

final class Context
{
    /** True if frontend page, not admin */
    public static function isFrontEnd(): bool
    {
        return !is_admin() || wp_doing_ajax() && !wp_doing_cron();
    }

    /** True if running an Ajax request */
    public static function isAjax(): bool
    {
        return wp_doing_ajax();
    }

    /** True if runing a cron job */
    public static function isCron(): bool
    {
        return wp_doing_cron();
    }

    /** True if on a single post or page (not archives, home, category, etc.) */
    public static function isSingular(string ...$postTypes): bool
    {
        return $postTypes ? is_singular($postTypes) : is_singular();
    }

    /** True if on page with a list of posts (category, tag, date, author, etc.) */
    public static function isPostList(): bool
    {
        return is_archive() || is_home() || is_search();
    }

    /** True if page/post with one or more shortcode(s) is being rendered */
    public static function isPostWithShortcode(string ...$shortcodes): bool
    {
        if (!is_singular()) return false;
        // Classic editor check
        $content = get_post()->post_content;
        foreach ($shortcodes as $shortcode) {
            if (has_shortcode($content, $shortcode)) return true;
        }
        // Done if no Gutenberg blocks
        if (strpos($content, '<!-- wp:') === false) return false;

        return self::blocksContainShortcode(parse_blocks($content), $shortcodes);
    }
    private static function blocksContainShortcode(array $blocks, array $shortcodes): bool
    {
        foreach ($blocks as $block) {
            $html = $block['innerHTML'] ?? '';
            if (!empty($html)) {
                foreach ($shortcodes as $shortcode) {
                    if (
                        ($pos = strpos($html, "[$shortcode")) !== false
                        && ($next = (string) substr($html, $pos + strlen($shortcode) + 1, 1)) !== ''
                        && ($next == ']' || ctype_space($next) || $next == '/')
                    ) {
                        return true;
                    }
                }
            }
            $inner = $block['innerBlocks'] ?? [];
            if ($inner && self::blocksContainShortcode($inner, $shortcodes)) return true;
        }
        return false;
    }

    /** True on an admin page or a specific admin page (e.g. 'dashboard', 'edit-post') */
    public static function isAdminPage(string ...$pages): bool
    {
        return is_admin()
            && (
                !$pages
                || (
                    ($screen = function_exists('get_current_screen') ? get_current_screen() : null)
                    && in_array($screen->id, $pages, true)
                )
            );
    }

    /** True if on an admin screen using block (Gutenberg) editor */
    public static function isBlockEditor(): bool
    {
        return is_admin()
            && function_exists('get_current_screen') && ($screen = get_current_screen())
            && method_exists($screen, 'is_block_editor') && $screen->is_block_editor();
    }
}
