<?php

namespace CocktailRecipes\Core\Helpers;

use CocktailRecipes\Core\Exceptions\InvalidCacheName;
use CocktailRecipes\Plugin;
use CocktailRecipes\Core\Iterators\PhpFileIterator;

class Cache
{
    /** True if caching is enabled */
    final public static function enabled(): bool
    {
        /** @disregard P1011 undefined constant COCKTAIL_RECIPES_NO_CACHE */
        return !defined('COCKTAIL_RECIPES_NO_CACHE') || !COCKTAIL_RECIPES_NO_CACHE;
    }

    /** Path to cache files */
    final protected static function path(): string
    {
        return Plugin::path() . 'cache/';
    }

    /** True if caching is enabled and writable */
    final public static function writable(): bool
    {
        return self::enabled() && File::isWritable(self::path());
    }

    /**
     * @throws InvalidCacheName if invalid
     */
    protected static function checkName(string $name): void
    {
        if ($name == '' || $name == 'index') throw new InvalidCacheName(esc_html($name));
    }

    /** @internal Get full path to cache file */
    protected static function cacheFile(string $name): string
    {
        self::checkName($name);
        return self::path() . $name . '.php';
    }

    /**
     * True if a cache entry exists
     *
     * @throws InvalidCacheName
     */
    final public static function has(string $name): bool
    {
        return self::enabled() && File::isFile(static::cacheFile($name));
    }

    /**
     * Get cache entry contents or null
     *
     * @return  array|string|null   null if entry does not exist
     *
     * @throws InvalidCacheName
     */
    final public static function get(string $name)
    {
        if (!self::enabled()) return null;
        $data = @include static::cacheFile($name);
        return ($data !== false) ? $data : null;
    }

    /** Get list of all cache entries or those matching a prefix */
    final public static function entries(string $prefix = ''): array
    {
        $names = [];
        self::each($prefix, function ($name) use (&$names) {
            $names[] = $name;
        });
        return $names;
    }

    /**
     * Store data to cache
     *
     * @param  string|array $data   data to be cached
     * @return bool         true if saved, false if failed or disabled
     *
     * @throws InvalidCacheName
     */
    final public static function set(string $name, $data): bool
    {
        if (!self::enabled() || !File::isWritable(self::path())) return false;
        $file = static::cacheFile($name);
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_var_export -- intentional cache serialization
        $data = "<?php\nreturn " . var_export($data, true) . ";\n";
        if (!File::write($file, $data)) {
            Logger::error("failed to write cache file: $file");
            return false;
        }
        return true;
    }

    /**
     * Remove a cache entry
     *
     * @return  bool    true if removed, false if missing or failed
     *
     * @throws InvalidCacheName
     */
    final public static function remove(string $name): bool
    {
        if (!self::enabled()) return false;
        $file = static::cacheFile($name);
        // `@` used for edge case if file deleted between file check and deletion
        return File::exists($file) ? File::delete($file) : false;
    }

    /** Clear all cache entries or those matching a prefix */
    final public static function clear(string $prefix = ''): int
    {
        $deleted = 0;
        $path = self::path();
        self::each($prefix, function ($name) use ($path, &$deleted) {
            if (File::delete($path . $name . '.php')) ++$deleted;
        });
        return $deleted;
    }

    /** Clear all cache entries older than a time frame, optionally matching a prefix */
    final public static function clearOld(int $seconds, string $prefix = ''): int
    {
        $deleted = 0;
        $path = self::path();
        $now  = time();
        self::each($prefix, function ($name) use ($path, $now, $seconds, &$deleted) {
            $file = $path . $name . '.php';
            $mtime = @filemtime($file);
            if ($mtime !== false && ($now - $mtime) > $seconds) {
                if (File::delete($file)) ++$deleted;
            }
        });
        return $deleted;
    }

    /**
     * Apply a callback to all cache entries or those matching a prefix
     *
     * @param  callable(string $name): void     $callback
     */
    final public static function each(string $prefix, callable $callback): void
    {
        $prefixLen = strlen($prefix);
        if (self::enabled() && File::isDir($path = self::path())) {
            foreach (new PhpFileIterator($path) as $name) {
                if ($prefixLen && substr($name, 0, $prefixLen) !== $prefix) continue;
                if (static::eachMatch($name)) {
                    $callback($name);
                }
            }
        }
    }

    /** True if file should be included in each() loop */
    protected static function eachMatch(string $name): bool
    {
        return true;
    }

    /** Get diagnostic information */
    final public static function info(): array
    {
        $isDir = File::isDir($path = self::path());
        return [
            'enabled'  => self::enabled(),
            'path'     => $path,
            'exists'   => $isDir,
            'writable' => $isDir && File::isWritable($path),
        ];
    }
}
