<?php

namespace CocktailRecipes\Core\Helpers;

use CocktailRecipes\Plugin;

final class Assets
{
    // asset names and associated handles
    private static array $assets = [];

    /**
     * Used by Plugin to register and/or enqueue js/css assets
     *
     * @param   string  $type       'css' or 'js'
     * @param   string  $name       name for deferred inclusion on page (use '' if none)
     * @param   string  $asset      full url, relative/absolute path to file, or filename w/o ext
     * @param   bool    $enqueue    true to automatically load; else deferred
     */
    public static function add(string $type, string $name, string $asset, bool $enqueue): void
    {
        static $ids = 0;
        $handle = Plugin::slug() . '-' . ++$ids;
        switch ($type) {
            case 'css':
                $func = $enqueue ? 'wp_enqueue_style' : 'wp_register_style';
                $func($handle, self::url('css', $asset), [], Plugin::version());
                break;
            case 'js':
                $func = $enqueue ? 'wp_enqueue_script' : 'wp_register_script';
                $func($handle, self::url('js', $asset), [], Plugin::version(), true);
                break;
            default:
                return;
        }
        if (!$enqueue) {
            // @todo future - if a shared library is used, may want plugin slug as name prefix
            self::$assets[$name][$handle] = $type;
        }
    }

    /**
     * Include js/css assets by name defined in Plugin on the current page
     *
     * IMPORTANT NOTE:
     * This function should be called from the Plugin::addFrontendAssets()
     * and/or Plugin::addAdminAssets() functions. If you defer calls to the
     * Assets::include() function until later, you risk it occurring too late
     * in the load process with some themes, and your assets will not load.
     */
    public static function include(string $name = ''): void
    {
        if ($handles = self::$assets[$name] ?? null) {
            foreach ($handles as $handle => $type) {
                switch ($type) {
                    case 'css':
                        wp_enqueue_style($handle);
                        break;
                    case 'js':
                        wp_enqueue_script($handle);
                        break;
                }
            }
            unset(self::$assets[$name]);
        }
    }

    /**
     * Get full URL to stylesheet or Javascript asset
     *
     * Asset can be passed as:
     * - full URL to a file, if it starts with '//' or protocol
     * - path to file relative to plugin root, if it starts with './'
     * - path to file relative to WordPress root, if it starts with '/'
     * - otherwise, assumed to be name of file without file ext, relative to /assets/{css|js}/
     *
     * @param   string  $type   'css' or 'js'
     */
    private static function url(string $type, string $asset): string
    {
        // Full URL to file
        if (substr($asset, 0, 2) == '//' || strpos($asset, '://') !== false) return $asset;
        // Path relative to the plugin root
        if (substr($asset, 0, 2) == './') return Plugin::url() . ltrim(substr($asset, 1), '/');
        // Path relative to WordPress root
        if (substr($asset, 0, 1) == '/') return site_url($asset);
        // Path to file relative to assets/js/ or assets/css/
        return Plugin::url() . "assets/$type/$asset.$type";
    }
}
